/*
    xchat_notify.c - maemo notifications plugin for xchat

  Copyright (C)
    2010,                       Christian Thaeter <ct@pipapo.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <dbus/dbus.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <mce/dbus-names.h>

#include "xchat-plugin.h"

#define PNAME "MaemoNotify"
#define PDESC "Notify by LED, vibrator and other means";
#define PVERSION "0.1"

struct maemo_notify_plugin_data;

static void
maemo_note (DBusConnection* bus, const char* text);

static void
maemo_dialog (DBusConnection* bus, const char* text);

static void
led_activate (DBusConnection* bus, const char* pattern);

static void
led_deactivate (DBusConnection* bus, const char* pattern);

static void
vibra_activate (DBusConnection* bus, const char* pattern);

static void
vibra_deactivate (DBusConnection* bus, const char* pattern);


static int
maemo_note_cb (char *word[], char *word_eol[], void *userdata);

static int
maemo_dialog_cb (char *word[], char *word_eol[], void *userdata);

static int
led_activate_cb (char *word[], char *word_eol[], void *userdata);

static int
led_deactivate_cb (char *word[], char *word_eol[], void *userdata);

static int
led_pattern_cb (char *word[], char *word_eol[], void *userdata);

static int
vibra_activate_cb (char *word[], char *word_eol[], void *userdata);

static int
vibra_deactivate_cb (char *word[], char *word_eol[], void *userdata);

static int
vibra_pattern_cb (char *word[], char *word_eol[], void *userdata);

static int
notify_mode_cb (char *word[], char *word_eol[], void *userdata);

static int
dialog_throttle_cb (char *word[], char *word_eol[], void *userdata);

static int
notify_hilight_cb (char *word[], void *userdata);

static int
notify_private_cb (char *word[], void *userdata);

static void
notify_cb (char *word[], struct maemo_notify_plugin_data* s);

static int
focus_cb (char *word[], void *userdata);

static int
dialog_throttle_timer_cb (void *userdata);


/* 0 = display off, 1 = display on */
static int
display_state (void);

/* 0 = xchat in background, 1 = xchat in foreground */
static int
xchat_state (void);



static struct maemo_notify_plugin_data
{
  xchat_plugin *ph;
  DBusConnection* dbus;
  DBusError error;

  int led_enable;
  int vibra_enable;
  int note_enable;
  int dialog_enable;

  int channel_notify_enable;
  int private_notify_enable;

  int dialog_throttle_max;
  int dialog_throttle_cur;

  int dialog_throttle_recover_time;
  xchat_hook* dialog_throttle_hook;

  /* which pattern will be used for notifications, NULL turns it off */
  char* led_pattern;
  char* vibra_pattern;

  /* when led or vibra is active these point to the patterns above */
  char* led_active;
  char* vibra_active;
} maemo_notify_private;




static struct commands
{
  const char* command;
  const char* help;
  int (*callback) (char *[], char *[], void *);
} command_table[] =
  {
    /* first the raw commands for testing and scripting */
    {"LED_ACTIVATE", "Usage: LED_ACTIVATE <pattern>; enables a led blinking pattern", led_activate_cb},
    {"LED_CANCEL", "Usage: LED_CANCEL <pattern>; disables a led blinking pattern", led_deactivate_cb},
    {"VIBRA_ACTIVATE", "Usage: VIBRA_ACTIVATE <pattern>; enables a vibration pattern", vibra_activate_cb},
    {"VIBRA_CANCEL", "Usage: VIBRA_CANCEL <pattern>; disables a vibration pattern", vibra_deactivate_cb},
    {"MAEMO_BANNER", "Usage: MAEMO_BANNER <message...>; shows a messages for a short while", maemo_note_cb},
    {"MAEMO_DIALOG", "Usage: MAEMO_DIALOG <message...>; shows a messages which must be acknowledged", maemo_dialog_cb},

    /* configuration primitives */
    {"LED_PATTERN", "Usage: LED_PATTERN [pattern]; set or show the led pattern for notifications", led_pattern_cb},
    {"VIBRA_PATTERN", "Usage: VIBRA_PATTERN [pattern]; set or show the vibrator pattern for notifications", vibra_pattern_cb},

    {"NOTIFY_MODE", "Usage: NOTIFY_MODE [+/-flags]; set notification flags, l=LED, v=Vibrator, b=Banner, d=Dialog", notify_mode_cb},

    {"DIALOG_THROTTLE", "Usage: DIALOG_THROTTLE [max recover]; set or show dialog thottle limits, max=initial limit,"
                                                             " recover=time in seconds for releasing", dialog_throttle_cb},

    {NULL, NULL, NULL}
  };


static struct printhooks
{
  const char* event;
  int (*callback) (char *word[], void *user_data);
} printhook_table[] =
  {
    {"Channel Msg Hilight", notify_hilight_cb},
    {"Private Message", notify_private_cb},
    {"Private Message to Dialog", notify_private_cb},
    {"Focus Tab", focus_cb},
    {"Focus Window", focus_cb},

    {NULL, NULL}
  };




void
xchat_plugin_get_info (char **name,
                       char **desc,
                       char **version,
                       void **reserved)
{
   *name = PNAME;
   *desc = PDESC;
   *version = PVERSION;
}


int
xchat_plugin_init (xchat_plugin *plugin_handle,
                   char **plugin_name,
                   char **plugin_desc,
                   char **plugin_version,
                   char *arg)
{
  /* we need to save this for use with any xchat_* functions */
  maemo_notify_private.ph = plugin_handle;

  /* tell xchat our info */
  *plugin_name = PNAME;
  *plugin_desc = PDESC;
  *plugin_version = PVERSION;

  maemo_notify_private.led_enable = 1;
  maemo_notify_private.vibra_enable = 1;
  maemo_notify_private.note_enable = 1;
  maemo_notify_private.dialog_enable = 1;

  maemo_notify_private.channel_notify_enable = 1;
  maemo_notify_private.private_notify_enable = 1;

  maemo_notify_private.led_pattern = strdup ("PatternCommunicationIM");
  maemo_notify_private.vibra_pattern = strdup ("PatternIncomingMessage");

  maemo_notify_private.led_active = NULL;
  maemo_notify_private.vibra_active = NULL;

  maemo_notify_private.dialog_throttle_max = 5;
  maemo_notify_private.dialog_throttle_cur = 5;

  maemo_notify_private.dialog_throttle_recover_time = 10 * 60 * 1000;   /* 10 minutes */
  maemo_notify_private.dialog_throttle_hook = 0;

  dbus_error_init (&maemo_notify_private.error);
  maemo_notify_private.dbus = dbus_bus_get (DBUS_BUS_SYSTEM, &maemo_notify_private.error);

  if (dbus_error_is_set(&maemo_notify_private.error))
    {
      xchat_printf (maemo_notify_private.ph, "Maemo Notify plugin error!\n");
      xchat_printf (maemo_notify_private.ph, "DBusError.name: %s\n", maemo_notify_private.error.name);
      xchat_printf (maemo_notify_private.ph, "DBusError.message: %s\n", maemo_notify_private.error.message);
      return 0;
    }

  /* load all commands */
  for (struct commands* itr = command_table; itr->command; ++itr)
    xchat_hook_command (maemo_notify_private.ph, itr->command, XCHAT_PRI_NORM, itr->callback,
                        itr->help, &maemo_notify_private);


  /* run the configuration */
  char buffer[4096];
  *buffer = '\0';
  snprintf (buffer, 4096, "%s/maemo_notify.conf", xchat_get_info(maemo_notify_private.ph, "xchatdirfs"));
  FILE* conf = fopen (buffer, "r");
  if (conf)
    {
      while (fgets (buffer, 4096, conf))
        {
          char* nl = strchr (buffer, '\n');
          if (nl)
            *nl = '\0';
          xchat_commandf (maemo_notify_private.ph, "%s", buffer);
        }
      fclose (conf);
    }

  /* establish printhooks */
  for (struct printhooks* itr = printhook_table; itr->event; ++itr)
    xchat_hook_print (maemo_notify_private.ph, itr->event, XCHAT_PRI_NORM, itr->callback, &maemo_notify_private);

  xchat_print (maemo_notify_private.ph, "Maemo Notify plugin loaded successfully!\n");

  return 1;
}

int
xchat_plugin_deinit (void)
{
  if (maemo_notify_private.dbus)
    dbus_connection_unref (maemo_notify_private.dbus);

  maemo_notify_private.dbus = NULL;

  free (maemo_notify_private.led_pattern);
  free (maemo_notify_private.vibra_pattern);

  xchat_printf (maemo_notify_private.ph, "Maemo Notify plugin unloaded\n");

  return 1;
}



static void
maemo_note (DBusConnection* bus, const char* text)
{
  if (bus && text)
    {
      DBusMessage* msg = dbus_message_new_method_call ("org.freedesktop.Notifications",
                                                       "/org/freedesktop/Notifications",
                                                       "org.freedesktop.Notifications",
                                                       "SystemNoteInfoprint");

      if (msg && dbus_message_append_args (msg, DBUS_TYPE_STRING, &text, DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply (msg, TRUE);
          dbus_connection_send (bus, msg, NULL);
          dbus_connection_flush (bus);
          dbus_message_unref (msg);
        }
     }
}

static int
maemo_note_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  maemo_note (s->dbus, word_eol[2]);
  return XCHAT_EAT_ALL;
}



static void
maemo_dialog (DBusConnection* bus, const char* text)
{
  if (bus && text)
    {
      DBusMessage* msg = dbus_message_new_method_call("org.freedesktop.Notifications",
                                                      "/org/freedesktop/Notifications",
                                                      "org.freedesktop.Notifications",
                                                      "SystemNoteDialog");

      int iconType = 1;
      const char* buttonText = "";

      if (msg && dbus_message_append_args(msg,
                                          DBUS_TYPE_STRING, &text,
                                          DBUS_TYPE_UINT32, &iconType,
                                          DBUS_TYPE_STRING, &buttonText,
                                          DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply(msg, TRUE);
          dbus_connection_send(bus, msg, NULL);
          dbus_connection_flush(bus);
          dbus_message_unref(msg);
        }
     }
}

static int
maemo_dialog_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  maemo_dialog (s->dbus, word_eol[2]);
  return XCHAT_EAT_ALL;
}



static void
led_activate (DBusConnection* bus, const char* pattern)
{
  if (bus && pattern)
    {
      DBusMessage* msg = dbus_message_new_method_call("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "req_led_pattern_activate");

      if (msg && dbus_message_append_args(msg,
                                          DBUS_TYPE_STRING, &pattern,
                                          DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply(msg, TRUE);
          dbus_connection_send(bus, msg, NULL);
          dbus_connection_flush(bus);
          dbus_message_unref(msg);
        }
     }
}


static int
led_activate_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  led_activate (s->dbus, word[2]);
  return XCHAT_EAT_ALL;
}


static void
led_deactivate (DBusConnection* bus, const char* pattern)
{
  if (bus && pattern)
    {
      DBusMessage* msg = dbus_message_new_method_call("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "req_led_pattern_deactivate");

      if (msg && dbus_message_append_args(msg,
                                          DBUS_TYPE_STRING, &pattern,
                                          DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply(msg, TRUE);
          dbus_connection_send(bus, msg, NULL);
          dbus_connection_flush(bus);
          dbus_message_unref(msg);
        }
     }
}

static int
led_deactivate_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  led_deactivate (s->dbus, word[2]);
  return XCHAT_EAT_ALL;
}


static int
led_pattern_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;

  if (word[2])
    {
      /* cancel any pattern which is active */
      if (s->led_active)
        {
          led_deactivate (s->dbus, s->led_active);
          s->led_active = NULL;
        }

      /* set new pattern */
      free (s->led_pattern);
      s->led_pattern = strdup (word[2]);
    }

  /* show the configured pattern */
  xchat_printf (s->ph, "LED Pattern: %s\n", s->led_pattern?s->led_pattern:"n/a");

  return XCHAT_EAT_ALL;
}





static void
vibra_activate (DBusConnection* bus, const char* pattern)
{
  if (bus && pattern)
    {
      DBusMessage* msg = dbus_message_new_method_call("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "req_vibrator_pattern_activate");

      if (msg && dbus_message_append_args(msg,
                                          DBUS_TYPE_STRING, &pattern,
                                          DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply(msg, TRUE);
          dbus_connection_send(bus, msg, NULL);
          dbus_connection_flush(bus);
          dbus_message_unref(msg);
        }
     }
}

static int
vibra_activate_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  vibra_activate (s->dbus, word[2]);
  return XCHAT_EAT_ALL;
}



static void
vibra_deactivate (DBusConnection* bus, const char* pattern)
{
  if (bus && pattern)
    {
      DBusMessage* msg = dbus_message_new_method_call("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "req_vibrator_pattern_deactivate");

      if (msg && dbus_message_append_args(msg,
                                          DBUS_TYPE_STRING, &pattern,
                                          DBUS_TYPE_INVALID))
        {
          dbus_message_set_no_reply(msg, TRUE);
          dbus_connection_send(bus, msg, NULL);
          dbus_connection_flush(bus);
          dbus_message_unref(msg);
        }
     }
}

static int
vibra_deactivate_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  vibra_deactivate (s->dbus, word[2]);
  return XCHAT_EAT_ALL;
}


static int
vibra_pattern_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;

  if (word[2])
    {
      /* cancel any pattern which is active */
      if (s->vibra_active)
        {
          vibra_deactivate (s->dbus, s->vibra_active);
          s->vibra_active = NULL;
        }

      /* set new pattern */
      free (s->vibra_pattern);
      s->vibra_pattern = strdup (word[2]);
    }

  /* show the configured pattern */
  xchat_printf (s->ph, "Vibrator Pattern: %s\n", s->vibra_pattern?s->vibra_pattern:"n/a");

  return XCHAT_EAT_ALL;
}


static int
notify_mode_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;

  if (word[2])
    {
      int flag = 1;     /* default is turning on */
      for (const char* itr = word[2]; *itr; ++itr)
        switch (*itr)
          {
          case '+':
            flag = 1;
            break;
          case '-':
            flag = 0;
            break;
          case 'l':
          case 'L':
            s->led_enable = flag;
            break;
          case 'v':
          case 'V':
            s->vibra_enable = flag;
            break;
          case 'b':
          case 'B':
            s->note_enable = flag;
            break;
          case 'd':
          case 'D':
            s->dialog_enable = flag;
            break;
          case 'p':
          case 'P':
            s->private_notify_enable = flag;
            break;
          case 'c':
          case 'C':
            s->channel_notify_enable = flag;
            break;
          default:
            xchat_printf (s->ph, "Unknown notification mode flag: '%c'\n", *itr);
            break;
          }
    }

  xchat_printf (s->ph,
                "Notification Modes: LED:%s Vibrator:%s Banner:%s Dialog:%s Channel:%s Private:%s\n",
                s->led_enable?"ON":"OFF",
                s->vibra_enable?"ON":"OFF",
                s->note_enable?"ON":"OFF",
                s->dialog_enable?"ON":"OFF",
                s->channel_notify_enable?"ON":"OFF",
                s->private_notify_enable?"ON":"OFF"
                );

  return XCHAT_EAT_ALL;
}


static int
dialog_throttle_cb (char *word[], char *word_eol[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;

  if (*word[2])
    {
      int n = atoi (word[2]);
      if (n > 0)
        {
          s->dialog_throttle_max = n;
        }
      else
        xchat_printf (s->ph,"illegal max dialog parameter: %s\n", word[2]);
    }

  if (*word[3])
    {
      int n = atoi (word[3]);
      if (n > 0)
        {
          if (s->dialog_throttle_recover_time != n*1000)
            {
              s->dialog_throttle_recover_time = n*1000;
              if (s->dialog_throttle_hook)
                {
                  xchat_unhook (s->ph, s->dialog_throttle_hook);
                  s->dialog_throttle_hook = xchat_hook_timer (s->ph, s->dialog_throttle_recover_time, dialog_throttle_timer_cb, s);
                }
            }
        }
      else
        xchat_printf (s->ph,"illegal recover time parameter: %s\n", word[3]);
    }

  xchat_printf (s->ph,
                "Dialog Throttle: max dialogs:%d recover time:%d sec\n",
                s->dialog_throttle_max,
                s->dialog_throttle_recover_time/1000
                );

  return XCHAT_EAT_ALL;
}


static int
notify_hilight_cb (char *word[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  if (s->channel_notify_enable)
    notify_cb (word, s);

  return XCHAT_EAT_NONE;
}


static int
notify_private_cb (char *word[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  if (s->private_notify_enable)
    notify_cb (word, s);
  return XCHAT_EAT_NONE;
}


static void
notify_cb (char *word[], struct maemo_notify_plugin_data* s)
{
  int disp_st = display_state ();
  int xchat_st = xchat_state ();

  char buffer[4096];

  if (!s->led_active && s->led_enable && !disp_st)
    led_activate (s->dbus, s->led_active = s->led_pattern);

  if (!s->vibra_active && s->vibra_enable && !disp_st)
    vibra_activate (s->dbus, s->vibra_active = s->vibra_pattern);

  if (s->dialog_enable && s->dialog_throttle_cur && !xchat_st)
    {
      snprintf (buffer, 4096, "%s: %s: %s", xchat_get_info(s->ph, "channel"), word[1], word[2]);
      maemo_dialog (s->dbus, buffer);
      --s->dialog_throttle_cur;
      if (!s->dialog_throttle_hook)
        s->dialog_throttle_hook = xchat_hook_timer (s->ph, s->dialog_throttle_recover_time, dialog_throttle_timer_cb, s);
    }
  else if (s->note_enable && disp_st && !xchat_st)              /* banner only if we don't show a dialog */
    {
      snprintf (buffer, 4096, "%s: %s: %s", xchat_get_info(s->ph, "channel"), word[1], word[2]);
      maemo_note (s->dbus, buffer);
    }
}

static int
focus_cb (char *word[], void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;

  if (s->led_active)
    {
      led_deactivate (s->dbus, s->led_active);
      s->led_active = NULL;
    }

  if (s->vibra_active)
    {
      vibra_deactivate (s->dbus, s->vibra_active);
      s->vibra_active = NULL;
    }

  if (s->dialog_throttle_hook)
    {
      xchat_unhook (s->ph, s->dialog_throttle_hook);
      s->dialog_throttle_hook = NULL;
      s->dialog_throttle_cur = s->dialog_throttle_max;
    }

  return XCHAT_EAT_NONE;
}

static int
dialog_throttle_timer_cb (void *userdata)
{
  struct maemo_notify_plugin_data* s = (struct maemo_notify_plugin_data*) userdata;
  if (s->dialog_throttle_cur < s->dialog_throttle_max)
    ++s->dialog_throttle_cur;

  if (s->dialog_throttle_cur < s->dialog_throttle_max)
    return 1;       /* keep going while max not reached */

  /* otherwise end this timer */
  s->dialog_throttle_hook = NULL;

  return 0;
}


static int
display_state (void)
{
  int ret = 0;

  if (maemo_notify_private.dbus)
    {
      DBusMessage* msg = dbus_message_new_method_call("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "get_display_status");

      if (msg)
        {
          DBusError ignored;
          dbus_error_init (&ignored);
          DBusMessage* reply = dbus_connection_send_with_reply_and_block (maemo_notify_private.dbus,
                                                                          msg,
                                                                          500,  /* 500ms are long, prolly never hit */
                                                                          &ignored);

          const char* display_mode;
          if (reply && dbus_message_get_args (reply, &ignored,
                                              DBUS_TYPE_STRING, &display_mode,
                                              DBUS_TYPE_INVALID))
            {
              ret = !strcmp ("on", display_mode);
              dbus_message_unref (reply);
            }

          dbus_message_unref(msg);
        }
     }
  return ret;
}


static int
xchat_state (void)
{
  return !strcmp ("active", xchat_get_info (maemo_notify_private.ph, "win_status"));
}


/*
// Local Variables:
// mode: C
// c-file-style: "gnu"
// indent-tabs-mode: nil
// End:
*/
