/*****************************************************************************
 * Copyright: 2011 Michael Zanetti <mzanetti@kde.org>                        *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "audioplaylist.h"

#include "xbmcconnection.h"
#include "xdebug.h"

AudioPlaylist::AudioPlaylist(Player *parent):
    Playlist(parent)
{
}

int AudioPlaylist::playlistId() const
{
    return 0;
}

int AudioPlaylist::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return m_itemList.count();
}

void AudioPlaylist::refresh()
{
    xDebug(XDAREA_PLAYLIST) << "Refreshing playlist" << playlistId();
    QVariantMap params;
    QVariantList properties;
    properties.append("duration");
    properties.append("artist");
    properties.append("album");
    params.insert("properties", properties);

    params.insert("playlistid", playlistId());

    int id = XbmcConnection::sendCommand("Playlist.GetItems", params);
    m_requestMap.insert(id, RequestGetItems);
}

void AudioPlaylist::queryItemData(int index)
{
    QVariantMap params;
    QVariantList properties;
    properties.append("title");
    properties.append("artist");
    properties.append("album");
    properties.append("fanart");
    properties.append("thumbnail");
    properties.append("duration");
    params.insert("properties", properties);

    QVariantMap limits;
    limits.insert("start", index);
    limits.insert("end", index + 1);
    params.insert("limits", limits);
    params.insert("playlistid", playlistId());

    xDebug(XDAREA_PLAYLIST) << "Gettin current item data" << params;
    int id = XbmcConnection::sendCommand("Playlist.GetItems", params);
    m_requestMap.insert(id, RequestCurrentData);
}


void AudioPlaylist::responseReveiced(int id, const QVariantMap &response)
{
    if(!m_requestMap.contains(id)) {
        return;
    }

    QVariant rsp = response.value("result");

//    xDebug(XDAREA_PLAYLIST) << "AudioPlaylist response:" << rsp;

    switch(m_requestMap.value(id)) {
    case RequestGetItems: {
        xDebug(XDAREA_PLAYLIST) << "got GetItems response:" << response;
        beginResetModel();
        while(!m_itemList.isEmpty()){
            delete m_itemList.takeFirst();
        }
        QVariantList responseList = rsp.toMap().value("items").toList();
        foreach(const QVariant &itemVariant, responseList) {
            QVariantMap itemMap = itemVariant.toMap();
            AudioPlaylistItem *item = new AudioPlaylistItem();
            item->setLabel(itemMap.value("label").toString());
            item->setDuration(QTime().addSecs(itemMap.value("duration").toInt()));
            item->setArtist(itemMap.value("artist").toString());
            item->setAlbum(itemMap.value("album").toString());
            m_itemList.append(item);
        }
        endResetModel();
        emit countChanged();
        break;
    }
    case RequestCurrentData: {
        xDebug(XDAREA_PLAYLIST) << "Current item data response" << rsp.toMap();
        if(m_itemList.count() > m_currentItem && m_currentItem > -1) {
            AudioPlaylistItem *item = m_itemList.at(m_currentItem);
            QVariantList responseList = rsp.toMap().value("items").toList();
            QVariantMap itemMap = responseList.first().toMap();
            item->setLabel(itemMap.value("label").toString());
            item->setTitle(itemMap.value("title").toString());
            item->setArtist(itemMap.value("artist").toString());
            item->setAlbum(itemMap.value("album").toString());
            item->setFanart(itemMap.value("fanart").toString());
            item->setThumbnail(itemMap.value("thumbnail").toString());
            emit currentChanged();
            break;
        }
        }
    }
}

QVariant AudioPlaylist::data(const QModelIndex &index, int role) const
{
    switch(role) {
    case Qt::DisplayRole:
        return m_itemList.at(index.row())->label();
    case Qt::UserRole+1:
        return "file";
    case Qt::UserRole+2:
        return m_itemList.at(index.row())->artist() + " - " + m_itemList.at(index.row())->album();
    case Qt::UserRole+3:
        return m_itemList.at(index.row())->duration().toString("mm:ss");
    }
    return QVariant();
}

PlaylistItem* AudioPlaylist::at(int index) const
{
    return m_itemList.at(index);
}

QString AudioPlaylist::title() const
{
    return "Now Playing - Music";
}
