/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "widgetinstaller.h"
#include "widgetinstaller_p.h"
#include "w3csettingskeys.h"

#include "storage.h"

// Maemo 5 uses Hildon mime framework for launching Application Manager
#ifdef Q_OS_MAEMO5
#include <dbus/dbus.h>
#include <hildon-mime.h>
#endif

#ifdef Q_OS_MAEMO6
#include "packagemanager.h"
#include "packagemanagerpendingcallwatcher.h"
#endif

#include "unzip.h"
#include "zzip.h"

#include <QDateTime>
#include <QProcess>
#include <QCoreApplication>
#include <sys/stat.h>

const QString TMPDIR("/var/tmp/");

const QString WidgetInstallerPrivate::S_DEFAULT_ICON = QString("Icon-music");
const QString WidgetInstallerPrivate::S_DEFAULT_VERSION = QString("0.0.1");
const QString WidgetInstallerPrivate::S_DEFAULT_VIEW_MODES = QString("floating");
const QString WidgetInstallerPrivate::S_DEFAULT_DESCRIPTION = QString("A web widget");
const QString WidgetInstallerPrivate::S_INSTALLER_VERSION = QString("0.1.1");
const QString WidgetInstallerPrivate::S_COPYRIGHT="Copyright (c) 2009 Nokia Corporation.";

WidgetInstallerPrivate::WidgetInstallerPrivate(SuperWidget * webWidget, WRT::Storage * storage, QObject* parent) :
    QObject(parent), m_webWidget(webWidget), m_storage(storage), m_iconPath(QString())
{
    m_copyRight = WidgetInstallerPrivate::S_COPYRIGHT;
}

WidgetInstallerPrivate::InstallationStatus WidgetInstallerPrivate::install(QString source, QString target, QString appId, QString &desktopfilePath)
{
    m_tempDir = source;
    m_debianRootDir = m_tempDir + "/output/debian";
    m_debianDir = m_debianRootDir + "/DEBIAN";
    m_tmpInstallDir = m_debianRootDir + "/" + target;


#ifdef Q_OS_MAEMO6
    m_desktopDir = m_debianRootDir + "/usr/share/dui/applets";
#else
    m_desktopDir = m_debianRootDir + "/usr/share/applications/hildon-home";
#endif
    m_widgetappdesktopDir = m_debianRootDir + "/usr/share/webwidgets/applications";

#ifdef Q_OS_MAEMO6
    m_appdesktopDir = m_debianRootDir + "/usr/share/applications";
#else
    m_appdesktopDir = m_debianRootDir + "/usr/share/applications/hildon";
#endif

    m_appId = appId;

    // Explicitly null by default.
    m_iconSymLink = QString();

    QString widget = m_webWidget->widgetBundlePath();

    readWidgetAttributes();
    setIcon();

    QString packagename = createPackageName();
    writeControlFile();
    writeChangeLog();
    writeCopyRightFile();
    int error = writePostinstFile();
    error = writePostrmFile();

    bool isDesktopFileNeeded = true;

    // If widget is used as a shared library, but has been defined to not to be launchable widget we omit desktopfile.
    if (m_webWidget->getProperties()->isSharedLibrary() && !m_webWidget->getProperties()->isSharedLibraryWidget()){
        isDesktopFileNeeded = false;
    }

    if (isDesktopFileNeeded){
        desktopfilePath = writeDesktopFile(target);
    }

    if (!createDebianStructure(packagename,widget)) {
        qCritical() << "Couldn't create Debian structure";
        return DebianPackageStructureCreationFailed;
    }
    if (!createDebianPackage(packagename)){
        qCritical() << "Couldn't create Debian package";
        return DebianPackageCreationFailed;
    }
    if (!installDebianPackage(packagename)){
        qCritical() << "Couldn't install Debian package";
        return DebianPackageInstallationFailed;
    }
    if (!secureStoreWidget(packagename, target)){
        qCritical() << "Couldn't securely store widget";
        return WebWidgetSecureStoringFailed;
    }

    return Success;
}

bool WidgetInstallerPrivate::uninstall(QString appId) {
    Q_UNUSED(appId)
    qWarning("WidgetInstallerPrivate::uninstall not removing debian package");
    return true;
    // TODO : Verify uninstallation and take code in use.
    /*
    WebAppRegistry * webAppRegistry = WebAppRegistry::instance();
    WebAppInfo appInfo;
    if (webAppRegistry->isRegistered(appId, appInfo)) {
        QString debFile = appInfo.attributes().value("debFile");

        QStringList params;
        params << "-r" << debFile;
        QProcess buildprocess;
        buildprocess.setStandardOutputFile("/dev/null");
        buildprocess.start("dpkg", params);
        buildprocess.waitForFinished(100000);
        int err = buildprocess.exitCode();
        if (err != 0){
            return false;
        } else {
            return true;
        }
    }
     *
     */
}

void WidgetInstallerPrivate::readWidgetAttributes()
{
    m_widgetName = m_webWidget->value(W3CSettingsKey::WIDGET_NAME);
    if (m_widgetName.isEmpty()) {
        // if the widget name is not provided in config.xml, use the wgt package name instead
        m_widgetName = QFileInfo(m_webWidget->widgetBundlePath()).baseName();
    }

    m_widgetVersion = m_webWidget->value(W3CSettingsKey::WIDGET_VERSION);
    if (m_widgetVersion == 0){
       m_widgetVersion = WidgetInstallerPrivate::S_DEFAULT_VERSION;
   }
    m_widgetDescription = m_webWidget->value(W3CSettingsKey::WIDGET_DESCRIPTION);
    if (m_widgetDescription.isEmpty()) {
        m_widgetDescription = WidgetInstallerPrivate::S_DEFAULT_DESCRIPTION;
    }

    m_authorEmail = m_webWidget->value(W3CSettingsKey::WIDGET_AUTHOR, "email");
    m_authorName = m_webWidget->value(W3CSettingsKey::WIDGET_AUTHOR);
    m_viewmodes = m_webWidget->value(W3CSettingsKey::WIDGET_VIEWMODES);

    //check if viewmode is defined or one of the valid view mode's are defined.If not change it to default view mode.
    if (m_viewmodes.isEmpty() || !(m_viewmodes.contains("floating") || m_viewmodes.contains("mini") ||
       m_viewmodes.contains("application") || m_viewmodes.contains("fullscreen")) ){
       m_viewmodes = WidgetInstallerPrivate::S_DEFAULT_VIEW_MODES;
    }
}

void WidgetInstallerPrivate::writeCommonDuiDesktopEntries(QByteArray & desktopFile, QString widgetRoot, QString viewMode) {

    desktopFile.append("[WRT]\n");
#ifdef Q_OS_MAEMO6
    desktopFile.append("Type=W3CWidget\n");
#endif
    desktopFile.append("StartHtml=file://" + widgetRoot + "\n");
    desktopFile.append("Protected-Store=" + m_packageField + "\n");
    desktopFile.append("WUID=" + m_packageField + "\n");
    desktopFile.append("ViewMode=" + viewMode + "\n");
    desktopFile.append("WebAppId=" + m_appId + "\n");
}


QString WidgetInstallerPrivate::writeDesktopFile(QString widgetRoot)
{
    QString filename = m_packageField + ".desktop";
    QString finalDesktopfilePath;

     QByteArray desktop;
     desktop.append("# This file is generated by Maemo Web Widget Installer "+m_InstallerVersion+"\n");

#ifdef Q_OS_MAEMO6
     desktop.append("[Desktop Entry]\nType=DUIApplet\n");
#else
     desktop.append("[Desktop Entry]\nType=W3CWidget\n");
#endif

     desktop.append("Name=" + m_widgetName + "\n");

     // setIcon called before. There is always icon inplace, no need to double check. If symlink is not set
     // use default icon.
     desktop.append("Icon=" + m_icon + "\n");

#ifdef Q_OS_MAEMO6
     desktop.append("Exec=/usr/bin/webwidgetrunner\n");
     desktop.append("[DUI]\n");
     desktop.append("X-DUIApplet-Applet=libmaemowrt-applet.so\n");
#else
     desktop.append("X-Path="+m_widgetName+"\n");
#endif

     //get the viewmodes from config.xml
     QStringList viewmodes = m_viewmodes.split(" ");

     //if the viewmodes is "application" two .desktop files are written in debian structure
     //first one to /usr/share/webwidgets/applications
     //and the other to /usr/share/applications

     if (viewmodes.contains("application") || viewmodes.contains("fullscreen")) {
         QDir dir;
         dir.mkpath(m_widgetappdesktopDir);
         QString desktopfilePath = m_widgetappdesktopDir + QDir::separator() + filename;
         QFile desktopfile(desktopfilePath);
         finalDesktopfilePath = desktopfilePath.remove(m_debianRootDir);
         desktopfile.open(QIODevice::ReadWrite);

         QByteArray homeScreenDesktopFile_Application = desktop;
         writeCommonDuiDesktopEntries(homeScreenDesktopFile_Application, widgetRoot, "application");
         desktopfile.write(homeScreenDesktopFile_Application);
         desktopfile.close();

         QByteArray appdesktop;
         appdesktop.append("# This file is generated by Maemo Web Widget Installer "+m_InstallerVersion+"\n");
         appdesktop.append("[Desktop Entry]\n");
         appdesktop.append("Type=Application\n");
         appdesktop.append("Name=" + m_widgetName + "\n");

         // setIcon called before. There is always icon inplace, no need to double check.
         appdesktop.append("Icon=" + m_icon + "\n");

         appdesktop.append("Exec=webwidgetrunner /usr/share/webwidgets/applications/"+filename + "\n");

#ifdef Q_OS_MAEMO6
         appdesktop.append("Categories=DUI;\n");
         appdesktop.append("OnlyShown=DUI;\n");
         // Write DUI sequence
         appdesktop.append("[DUI]\n");
#endif
         writeCommonDuiDesktopEntries(appdesktop, widgetRoot, "application");

         dir.mkpath(m_appdesktopDir);
         QFile appdesktopfile(m_appdesktopDir + QDir::separator() + filename);
         appdesktopfile.open(QIODevice::ReadWrite);
         appdesktopfile.write(appdesktop);
         appdesktopfile.close();

     }

     //if the viewmodes is "floating" or "mini", then write home screen desktop file
     // the .desktop file is written at /usr/share/dui/applets in debian structure
     if (viewmodes.contains("floating") || viewmodes.contains("mini")) {
         QDir dir;
         dir.mkpath(m_desktopDir);
         int indexFloating = viewmodes.indexOf("floating");
         int indexMini = viewmodes.indexOf("mini");

         //The priority of the view mode's(floating and mini) depends on the order they appear in config.xml
         //which ever view mode(floating or mini) appear's first that is written in home screen .desktop file and the other is disabled.
         bool floating = false;
         //lets check if the view mode has both mini and floating
         if (viewmodes.contains("floating") && viewmodes.contains("mini")){
             if (indexMini < indexFloating) {
                 //mini appeared first
                 floating = false;
             } else {
                 //floating appeared first
                 floating = true;
             }
         }
         else if (viewmodes.contains("floating")) {
             //view mode is only floating
             floating = true;
         } else {
             //view mode is only mini no more condition checks req here as this is the final possible case
             floating = false;
         }

         if (floating) {
             writeCommonDuiDesktopEntries(desktop, widgetRoot, "floating");
         } else {
             writeCommonDuiDesktopEntries(desktop, widgetRoot, "mini");
         }

         QString desktopfilePath = m_desktopDir + QDir::separator() + filename;
         QFile desktopfile(desktopfilePath);
         finalDesktopfilePath = desktopfilePath.remove(m_debianRootDir);
         desktopfile.open(QIODevice::ReadWrite);
         desktopfile.write(desktop);
         desktopfile.close();
     }
     return finalDesktopfilePath;
 }


 QString WidgetInstallerPrivate::createPackageName()
 {
     QString name = m_appId + "-wrt-widget";
     m_packageField = name;
     name.append(".deb");
     return name;
 }

 int WidgetInstallerPrivate::writeChangeLog()
 {
     QDateTime date = QDateTime::currentDateTime ();
     QByteArray content;
     content.append(m_packageField + " ("+m_widgetVersion+") unstable; urgency=low\n\n");
     content.append("  * Initial Debian release Maemo Web Runtime Widget "+m_widgetName +"\n\n");
     content.append(" -- "+m_authorName +m_authorEmail+" "+date.toString("ddd, d MMM yyyy hh:mm:ss")+" +0200\n\n");
     int error = 0;
     QFile logfile(m_debianDir + QDir::separator() + "changelog");
     if (logfile.open(QIODevice::ReadWrite)) {
         if (logfile.write(content) == -1) error = 1;
     logfile.close();
 } else {
     error = 1;
 }
  return error;
}

 int WidgetInstallerPrivate::writeCopyRightFile()
 {
     //TODO this is just a quick implementation of copyright file
     //need to read debian specs for full implementation!
     QByteArray copy;
     if (m_copyRight.length() > 0)
     { // check that a copyright exists
         QFileInfo tmp(m_copyRight);
         if (tmp.exists() && tmp.isFile())
         {  // if the copyright points to a file, read it
             QFile infile(m_copyRight);
             infile.open(QIODevice::ReadOnly);
             copy = infile.readAll();
             infile.close();
         } else { // copyright is a string
             copy.append(m_copyRight);
         }
     }

     if (copy.length() > 0)
     {
         QFile outfile(m_debianDir + QDir::separator() +"copyright");
         outfile.open(QIODevice::WriteOnly);
         if (outfile.exists())
         {
             outfile.remove();
         }
         outfile.close();

         QFile newoutfile(m_debianDir + QDir::separator() +"copyright");
         newoutfile.open(QIODevice::WriteOnly);
         newoutfile.write(copy);
         newoutfile.close();
     }
     else {
         return 1;
     }

  return 0;
}


 void WidgetInstallerPrivate::writeControlFile()
 {
     QByteArray content;
     QString version = generateValidDebianVersion(m_widgetVersion);

     content.append("Package: " + m_packageField + "\n");
     content.append("Source: " + m_packageField + "\n");
     // TODO: this must be enabled in the product but causes harm during development time
     //content.append("Depends: libmaemo-wrt\n");
     content.append("Priority: optional\n");
     content.append("Section: user/desktop\n");
     content.append("Maintainer: " +m_authorName + "\n");
     content.append("Version: "+ version + "\n");
     content.append("Architecture: all\n");
     // TODO : This is likely not working with more that 70 characters.
     content.append("Description: " + m_widgetName + " (web widget)" + "\n " + m_widgetDescription.simplified() + "\n");
     QDir dir;
     dir.mkpath(m_debianDir);
     QString m_controlFile = m_debianDir + QDir::separator() + "control";
     QFile controlfile(m_controlFile);
     controlfile.open(QIODevice::ReadWrite);
     controlfile.write(content);
     controlfile.close();
 }

void WidgetInstallerPrivate::setIcon() {
    qDebug() << Q_FUNC_INFO << "creating icon";

    WidgetProperties * properties = m_webWidget->getProperties();
    QString icon = properties->iconPath();
    QFileInfo iconFile(icon);

    // Only png can be used. All other icons fallback into default icon.
    if (iconFile.completeSuffix() == "png" && !icon.endsWith(":/resource/default_widget_icon.png")) {
        m_iconSymLink = "/usr/share/duihome/themes/images/" + m_appId + ".png";
        m_iconPath = icon;
        m_icon = m_appId;
    } else {
        m_icon = S_DEFAULT_ICON;
        // TODO : Default icon for Maemo5?
    }
}


 int WidgetInstallerPrivate::writePostinstFile()
 {
     QByteArray content;
     QString postinstFileName = m_debianDir + QDir::separator() + "postinst";

     content.append("#!/bin/sh\n");
     content.append("if [ ! -f /usr/share/webwidgets/registry ]; then\n");
     content.append("    mkdir -p /usr/share/webwidgets/registry\n");
     content.append("fi\n");
     content.append("mkdir /usr/share/webwidgets/registry/" + m_packageField + "\n");
     content.append("chmod 777 /usr/share/webwidgets/registry/" + m_packageField + "\n");

     // Make sure resourcepath is user-writable
     content.append("mkdir -p " + m_webWidget->getProperties()->resourcePath() + "\n");
     content.append("chmod 777 -R " + m_webWidget->getProperties()->resourcePath() + "\n");

#ifdef Q_OS_MAEMO6
     if (!m_iconSymLink.isNull()) {
         content.append("mkdir -p /usr/share/duihome/themes/images/\n");
         content.append(QString("if [ ! -f %1 ]; then\n").arg(m_iconSymLink));
         content.append(QString("ln -s %1 %2\n").arg(m_iconPath).arg(m_iconSymLink));
         content.append("fi\n");
     }
#endif
     //content.append("exec /usr/bin/webappregisterer "+m_webWidget->getProperties()->installPath() +" "+m_webWidget->getProperties()->id() +" "+"register" + "\n");
     content.append("exit 0\n");
     QFile postinstFile(postinstFileName);
     postinstFile.open(QIODevice::ReadWrite);
     postinstFile.write(content);
     postinstFile.close();
     mode_t mode = 0775;
     QByteArray file = postinstFileName.toAscii();
     int error = chmod(file.constData(), mode);
     return error;

}

 int WidgetInstallerPrivate::writePostrmFile()
 {
     QByteArray content;
     QString postrmFileName = m_debianDir + QDir::separator() + "postrm";
     content.append("#!/bin/sh\n");
     content.append("rm -rf /usr/share/webwidgets/registry/" + m_packageField + "\n");
     content.append("rm /etc/secure/s/" + m_packageField + "\n");

     // Remove resourcepath which we created at postinstall
     content.append("rm -rf " + m_webWidget->getProperties()->resourcePath() + "\n");
     //content.append("exec /usr/bin/webappregisterer "+m_webWidget->getProperties()->id() +" "+"unregister" + "\n");

#ifdef Q_OS_MAEMO6
     if (!m_iconSymLink.isNull()) {
         content.append(QString("rm -f %1\n").arg(m_iconSymLink));
     }
#endif
     content.append("exit 0\n");
     QFile postrmFile(postrmFileName);
     postrmFile.open(QIODevice::ReadWrite);
     postrmFile.write(content);
     postrmFile.close();
     mode_t mode = 0775;
     QByteArray file = postrmFileName.toAscii();
     int error = chmod(file.constData(), mode);
     return error;
 }

 bool WidgetInstallerPrivate::createDebianPackage(QString debname)
 {
     qDebug() << "Creating Debian package" << debname;
     QStringList params;
     QString fromdir =  m_tempDir + QDir::separator() + "output/debian/";

     qDebug() << fromdir << debname;

#ifdef Q_OS_MAEMO5
     QProcess buildprocess;
     //buildprocess.setStandardOutputFile("/dev/null");
     params << m_webWidget->widgetBundlePath() << TMPDIR + debname << m_appId; // TODO $$$: m_tempDir + QDir::separator() + debname
     buildprocess.start("wgt2deb", params);
     buildprocess.waitForFinished(100000);
     int err = buildprocess.exitCode();
     qDebug() << Q_FUNC_INFO << __LINE__ << "Error: " << err;
#else
     params << "-b" << fromdir << m_tempDir + QDir::separator() + debname;
     int err = dpkg( params);
#endif
     if (err) {
         qCritical() << "Error happened during Debian package creation:" << err;
         return false;
     }
     return true;
 }

bool WidgetInstallerPrivate::createDebianStructure(QString packagename, QString widget)
{
    Q_UNUSED(widget)

    QDir dir;
    packagename.remove(".deb");
    dir.mkpath(m_tmpInstallDir);

    unsigned long size = 0;
    int err = Zzip::unzip(m_webWidget->widgetBundlePath(), m_tmpInstallDir, size);
    if (err) {
        qCritical() << "Error happened during widget unzipping:" << err;
        return false;
    }
#ifdef Q_OS_MAEMO5
    if (!addIcon("/usr/share/icons/hicolor/scalable/hildon/")) {
        return false;
    }

    if (!addIcon("/usr/share/icons/hicolor/64x64/hildon/")) {
        return false;
    }
#endif
    return true;
}

bool WidgetInstallerPrivate::addIcon(QString iconInstallTarget) {
    if (m_iconPath.isNull()) {
        // m_iconPath is assigned to value when icon is found from widget
        // Let's use default icon from platform. Thus, not adding anything to debian package.
        // TODO This must be verified on maemo6, if taken into use there.
        return true;
    }

    QFile iconFile(m_debianRootDir + m_iconPath);
    if (iconFile.exists()) {
        QDir debianRootDir(m_debianRootDir);
        if (!debianRootDir.exists()) {
            return false;
        }

        if (!debianRootDir.mkpath(m_debianRootDir + iconInstallTarget)) {
            return false;
        }
        if (!iconFile.copy(m_debianRootDir + iconInstallTarget + m_icon + ".png")) {
            return false;
        }
    }
    return true;
}

#if defined(Q_OS_MAEMO5)
static gboolean quit_mainloop(gpointer data)
{
  g_main_loop_quit((GMainLoop*)data);
}
#endif

bool WidgetInstallerPrivate::installDebianPackage(QString debname)
{
    int err = 1;
    QString debfile = m_tempDir + QDir::separator() + debname;
// Maemo6 device
#if defined(Q_OS_MAEMO6) && defined(__ARMEL__)
    qDebug() << "Launching debian package installing with PackageManager.";
    err = pkgmgr(debfile);
    if ( err) {
        qCritical() << "Failed to install debian package with PackageManager.";
        return false;
    }
// Maemo5 device
#elif defined(Q_OS_MAEMO5) && defined(__ARMEL__)
    GMainLoop* mainloop;
    qDebug() << "Using Hildon framework install.";
    // Maemo 5 uses Hildon mime framwork when launching Application Manager
    DBusConnection* dbusConnection;
    DBusError dbusError;

    mainloop = g_main_loop_new(NULL, FALSE);

    if (!mainloop) {
      qCritical() << "Cannot create mainloop";
      return false;
    }

    dbus_error_init(&dbusError);
    dbusConnection = dbus_bus_get(DBUS_BUS_SESSION, &dbusError);

    if (!dbusConnection) {
        qCritical() << "Cannot launch Application Manager. Cannot get DBus connection:" << dbusError.message;
        dbus_error_free(&dbusError);
        return false;
    }

    // Debian packages are created under TMPDIR

    // TODO $$$: This should be enabled for Maemo5 too? And prehaps change variablename debname to debfile
    // debname = "file://" + m_tempDir + QDir::separator() + debname;
    debname = "file://" + TMPDIR + debname;
    err = hildon_mime_open_file(dbusConnection, debname.toAscii().constData());
    if (err != 1) {
        qCritical("Cannot launch Application Manager. hildon_mime_open_file() returned %d", err);
        return false;
    }

    g_timeout_add(1000, quit_mainloop, (void*)mainloop);
    g_main_loop_run(mainloop);
#else // Scratchbox
  qDebug() << "Installing debian package with dpkg -i";
  QStringList params;
#ifdef Q_OS_MAEMO5
  debfile = TMPDIR + debname;
#endif
  params << "-i" << debfile;
  err = dpkg( params);

  if (err) {
      qCritical() << "Error happened during package installation:" << err;
      return false;
    }
#endif

// TODO : Cleanup should be added into post installation
#ifdef Q_OS_MAEMO6
    bool removed = QFile::remove(debfile);
    if (!removed) {
        qCritical() << "Couldn't remove Debian package:" << debfile;
        return false;
    }
#endif

    return true;
}

#if defined(Q_OS_MAEMO6)
int WidgetInstallerPrivate::pkgmgr(QString debname)
{
    qDebug() << "pkgmgr:" << debname;
    PackageManager pkmgr;
    PackageManagerPendingCallWatcher* watcher = NULL;

    connect( &pkmgr, SIGNAL(operationComplete(const QString &, const QString &, const QString &, const bool)),
                     SLOT(pkgmgrOperationComplete(const QString &, const QString &, const QString &)) );
    connect( &pkmgr, SIGNAL(operationProgress(const QString &, const QString &, int)),
                     SLOT(pkgmgrOperationProgress(const QString &, const QString &, int)) );

    m_pkgmgrInstallationMutex.lock();
    m_pkgmgrInstallationSuccess = false;
    watcher = pkmgr.installFile(debname); // installFileUi doesn't work correctly. It just gives error message.
    if ( watcher) {
        connect( watcher, SIGNAL(dbusError( PackageManagerPendingCallWatcher*)),
                         SLOT(pkgmgrDbusError( PackageManagerPendingCallWatcher*)));
    }

    m_pkgmgrInstallationTime = 300;
    bool bRet = false;
    while (m_pkgmgrInstallationTime>0 && !bRet)
    {
        bRet = m_pkgmgrInstallationMutex.tryLock(100); // true if lock is available
        //qDebug() << "  ## waiting" << iWaiting << bRet;
        QCoreApplication::processEvents();
        m_pkgmgrInstallationTime--;
    }
    m_pkgmgrInstallationMutex.unlock();

    //qDebug() << "#### unlocked, exit.";
    //qDebug() << m_pkgmgrInstallationSuccess;
    return m_pkgmgrInstallationSuccess?0:1; // zero return value is success
}

void  WidgetInstallerPrivate::pkgmgrOperationComplete(const QString &operation,
        const QString &packagename, const QString &result)
{
    qDebug() << Q_FUNC_INFO;
    qDebug() << "  Operation:" << operation;
    qDebug() << "  Package:" << packagename;
    qDebug() << "  Result:" << result;

    m_pkgmgrInstallationSuccess = true;
    m_pkgmgrInstallationMutex.unlock();
}

void WidgetInstallerPrivate::pkgmgrDbusError( PackageManagerPendingCallWatcher* watcher)
{
    if ( !watcher) return;

    qDebug() << "####################";
    qDebug() << Q_FUNC_INFO;
    qDebug() << "  Error name:" << watcher->errorName();
    qDebug() << "  Error message:" << watcher->errorMessage();
    qDebug() << "####################";

    m_pkgmgrInstallationSuccess = false;
    m_pkgmgrInstallationMutex.unlock();
}

void WidgetInstallerPrivate::pkgmgrOperationProgress(const QString &operation, const QString &packagename, int percentage)
{
    m_pkgmgrInstallationTime = qMax(m_pkgmgrInstallationTime, 100); // Give more time if needed
    //qDebug() << "  ## Operation progress:" << operation << packagename << percentage ;
}
#endif

bool WidgetInstallerPrivate::secureStoreWidget(QString packagename, QString pathToSecureStore)
{
    packagename.remove(".deb");
    WRT::Storage * signedprivate_storage = WRT::Storage::createInstance(packagename);

    //Preparing the sec storage file
    QDir homedir;
    QString secstorefile = homedir.homePath() + "/" + ".maemosec-secure/s/"+packagename;

    QFile file(secstorefile);
    bool exists = file.exists();
    if (exists) {
        //removing the secure storage file of the widget if already exits
        //TODO check on device if it allows to remove secure stogare file
        file.remove();
    }

    //Adding all files of the widget to sec storage
    QString widgetpath = pathToSecureStore;
    widgetpath.append("/");
    QDir dir(widgetpath);
    QStringList entries = dir.entryList();
    entries.removeAll(".");
    entries.removeAll("..");
    for (int i=0; i<entries.count(); i++) {
        QString name = entries.at(i);
        //get the full path and name of the widget file
        QFile file(widgetpath+name);
        signedprivate_storage->add(file);
    }

    //After adding to sec storage check the signature
    //if it fails for some reason clean up the sec storage and remove the widget package
    //TODO as of now we dont have security checks in place we are not doing any security checks here
    bool retval = false;
    if (retval){
        QFile file(secstorefile);
        file.remove();
        QStringList params;
        params << "-r" << m_packageField;
        dpkg( params);
        //Signature check failed after sec storage
        return retval;
    }
    return true;
}

int WidgetInstallerPrivate::dpkg( const QStringList& params, int maxWaitTime /* =100000 */)
{

    QProcess buildprocess;
    // TMPDIR environment variable must be set, to get enough space
    // for installing big widgets in device
    QStringList env = QProcess::systemEnvironment() << QString("TMPDIR=") + TMPDIR;
#if defined(Q_OS_MAEMO6) && defined(__ARMEL__)
    env.replaceInStrings("PATH=","PATH=/usr/local/bin:"); // To make sure to use own build delivered tar, not busybox tar.
#endif
    buildprocess.setEnvironment( env);

    buildprocess.setStandardOutputFile("/dev/null");

#ifdef __i386__
    QStringList tmpParams = params;
    tmpParams.prepend("dpkg");
    buildprocess.start("fakeroot", tmpParams);
#else
    buildprocess.start("dpkg", params);
#endif

    buildprocess.waitForFinished( maxWaitTime);

    return buildprocess.exitCode();

}

WidgetInstallerPrivate::~WidgetInstallerPrivate()
{
    // TODO: need to implement recursive delete instead of using external program
    QString path = m_tempDir;
    QStringList params;
    params << "-rf" << path;
    QProcess::execute("rm", params);
}

QString WidgetInstallerPrivate::generateValidDebianVersion(const QString& originalVersion)
{
    QString validVersion = originalVersion;
    validVersion.remove(" ");
    if (!isValidDebianVersion(validVersion)) {
        // If supplied originalVersion string would fail in debian packaging,
        // we replace it with default version string + original version where illegal chars have been removed.
        validVersion = WidgetInstallerPrivate::S_DEFAULT_VERSION + "+" + convertToAlphaNumeric(originalVersion);
    }

    return validVersion;
}

QString WidgetInstallerPrivate::convertToAlphaNumeric(const QString& original)
{
    QString converted = original;
    for (int i = 0; i < converted.size(); i++) {
        QChar c = converted.at(i);
        if (!c.isLetterOrNumber()){
            converted.replace(i,1,"_");
        }
    }

    return converted;
}

bool WidgetInstallerPrivate::isValidDebianVersion(const QString& version)
{
    // Version must start with digit
    if (version.size() > 0 && !version.at(0).isDigit()) {
        return false;
    }

    // If there is hyphen in the version string, debian_revision must exist
    if (version.contains("-") && !hasValidDebianRevision(version)){
        return false;
    }

    // If there is colon in the version string, epoch must exist
    if (version.contains(":") && !hasValidDebianEpoch(version)){
        return false;
    }

    // Check upstream_veersion.
    if (!hasValidDebianUpstreamVersion(version)){
        return false;
    }

    return true;
}

bool WidgetInstallerPrivate::hasValidDebianEpoch(const QString& version)
{
    bool valid = false;

    // If there is hyphen in the version string, debian_revision must exist
    int separatorIndex = version.indexOf(":");
    if ((separatorIndex > 0) && (separatorIndex < version.size())){
        QString epoch = version.left(separatorIndex);
        epoch.toInt(&valid);
    }

    return valid;
}

bool WidgetInstallerPrivate::hasValidDebianUpstreamVersion(const QString& version)
{
    for (int i = 0; i < version.size(); i++) {
        QChar c = version.at(i);
        if (!c.isLetterOrNumber() && (c != '.') && (c != '+') && (c != '-') && (c != ':') && (c != '~')){
            return false;
        }
    }
    return true;
}

bool WidgetInstallerPrivate::hasValidDebianRevision(const QString& version)
{
    bool valid = false;

    // If there is hyphen in the version string, debian_revision must exist
    int separatorIndex = version.lastIndexOf("-");
    if ((separatorIndex > 0) && (separatorIndex < version.size())){
        QString revision = version.mid(separatorIndex);
        revision.toInt(&valid);
    }

    return valid;
}


