/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "SuperWidget.h"
#include "WgtWidget.h"

#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
#include "WgzWidget.h"
#endif

#include "signatureparser.h"
#include "w3cxmlplugin.h"
#include "WidgetPListParser.h"
#include "securitymanager.h"
#include "wrtsettings.h"
#include "WidgetUtilsLogs.h"
#include "w3csettingskeys.h"
#include "proprietarysettingskeys.h"
#include "widgetinstaller.h"
#include "featuremapping.h"


#include <QRegExp>
#include <QStringList>
#include <QDir>
#include <QDebug>
#include <QTextDocument>
#include <QFile>
#include <QIODevice>
#include <QHash>
#include <QUuid>
#include <QDateTime>
#include <QDirIterator>


#ifdef Q_OS_SYMBIAN
#include <WidgetRegistryConstants.h>
#include <WidgetRegistryClient.h>
#include <apgcli.h>
#include <apacmdln.h>
#include <S32MEM.H>
#include "f32file.h"
#define PROCESS_UID "processUid"
#endif

const QString TrustedDomain("TrustedWidgets");

WgtWidget::WgtWidget(QString& rootDirectory)
 : m_disableUnsignWidgetSignCheck(false)
{
    initialize(rootDirectory);
    m_widgetType = WidgetTypeW3c;
    m_contentType = CONTENT_TYPE_WGT;
    m_validCertificateAKI = QString();
}


//
// function to find digital signatures
// parameters:
//    digSig  returns list of digital signatures
//    path    path where you have unzipped widget contents
// return:
//    bool    true if successful
//
// Digital signatures are provided in W3C widget as signature.xml files
// There can be multiple digital signatures provided by distributors and
// one signature from author
// Since we are sorting the list in ascending order, author signature will be at index 0
//
bool WgtWidget::findDigitalSignatures(QStringList& digSig,  WidgetInstallError & error, const QString& path)
{
    QDir dir;
    QMap<int,QString> sortMap;

    if (path.isEmpty()) dir.setPath(widgetUnZipPath());
    else dir.setPath(path);

    qDebug() << "WgtWidget:findDigSig path:" << path;
    qDebug() << "findSignature: " << dir.absolutePath();

    QStringList entries = dir.entryList();

    QRegExp distSig("signature([1-9][0-9]?).xml");
    QString dirSlash =dir.path() + dir.separator();
    foreach(QString name, entries) {
        if (!name.compare("author-signature.xml")) {
            QString fullName = dirSlash + name;
            qDebug() << fullName;
            digSig.append(fullName);
        } else if (distSig.exactMatch(name) ) {
            QString fullName = dirSlash + name;
            qDebug() << fullName;
            int sigidx=distSig.cap(1).toInt();
            sortMap.insert(sigidx, fullName);
        }
    }
    for (QMap<int, QString>::const_iterator it=sortMap.constBegin();it !=sortMap.constEnd();++it)
        digSig.append(it.value());
    if (digSig.count() > 0)
        return true;
    qDebug() << "No signature.xml files" ;
    error = WidgetFindSignatureFailed;
    return false;
}

// function to check if the digitial signature is valid
// parameters:
//    sigList       list of paths of signature files to check sorted in numerical order, optionally headed by author-signature.xml
//    widgetPath    path where the widget is unzipped
//    error         returns error code that can be used for prompting
// return:
//    bool          true if digital signature is valid
//                  false if digital signature is invalid
//
bool WgtWidget::validateDigitalSignatures(QStringList& sigList, WidgetInstallError & error, const QString& widgetPath, bool cleanup)
{

    QDir widgetDir;
    if (widgetPath.isEmpty()) widgetDir.setPath(widgetUnZipPath());
    else widgetDir.setPath(widgetPath);
    QString aWidgetResourceDir = widgetDir.absolutePath();

    QByteArray aCaCertFilePath;

    // fetch signatures if method is called without signature list (TODO: remove this and rely on sigList having valid data)
    if (sigList.isEmpty()) {
        qDebug() << "isValidCert: finding signature:" << aWidgetResourceDir;

        if (!findDigitalSignatures(sigList, error, aWidgetResourceDir)) {
            qDebug() << "isValidCert: findDigitalSignature failed";
            if (cleanup)rmDir(aWidgetResourceDir);
            return false;
        }
    }

    // TODO: sort sigList by parsing out issuer and determining - currently it
    // is sorted to look at distributor signatures (in descending order) and
    // look at author-signature last, if necessary
    bool success = false;
    while (!sigList.isEmpty()) {
        QString signaturefile = QString(sigList.takeLast());
        qDebug() << "isValidCert: opening " << signaturefile;

        SignatureParser parser(aWidgetResourceDir, signaturefile);

        qDebug() << "isValidCert: parsing...";
        if (!parser.parse()){
            qDebug() << "isValidCert: Digital signature parsing failed.";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetSignatureParsingFailed;
            continue;
        }

        if (!parser.requiredElementsExist()){
            qDebug() << "isValidCert: error with required elements";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetSignatureOrSignedInfoMissing;
            continue;
        }
        qDebug() << "isValidCert: passed requiredElementsExist";

        QByteArray encyptionCert;
        if (!parser.validateCertificates(encyptionCert)){
            qDebug() << "isValidCert: Certificate validation failed.";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetCertValidationFailed;
            continue;
        }

        qDebug() << "isValidCert: passed validateCertificate";

        QDir dir(aWidgetResourceDir);
        dir.setFilter(QDir::NoDotAndDotDot | QDir::Files);
        QDirIterator dirItr(dir, QDirIterator::Subdirectories);
        QStringList files;
        while (dirItr.hasNext()) {
            dirItr.next();
            int basePathLen = dirItr.path().length();
            QString relativePath = dirItr.filePath();
            relativePath = relativePath.mid(basePathLen + 1);
            files.append(relativePath);
        }

        if (!parser.checkReferencesExist(files)){
            qDebug() << "isValidCert: Reference check failed";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetSignatureRefExistFailed;
            continue;
        }

        qDebug() << "isValidCert: passed checkReferencesExist";

        if (!parser.validateSignature(signaturefile, encyptionCert)){
            qDebug() << "isValidCert: Signature validation failed.";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetSignatureValidationFailed;
            continue;
        }

        qDebug() << "isValidCert: passed validateSignature";

        if (!parser.validateReferences(files)){
            qDebug() << "isValidCert: Reference validation failed.";
            if (cleanup)rmDir(aWidgetResourceDir);
            error = WidgetSignatureRefValidationFailed;
            continue;
        }

        qDebug() << "isValidCert: passed validateReferences";

        success = true;
        m_validCertificateAKI = parser.getAKI();
        break;
    }

    if (!success) return false;

    //All done, Valid widget";
    if (cleanup)rmDir(aWidgetResourceDir);
    qDebug() << "isValidCert: Valid widget.";
    error = WidgetValidSignature;
    return true;
}

//
// function to parse widget manifest (like info.plist or config.xml)
// prereq:    should have set proper WidgetUnZipPath to find the widget contents
// parameters:
//    path    path to bundle
//    force   after first parse m_manifest holds manifest info.
//            if you want to force parsing each time, set this to true.
//            default = false
// return:
//    bool    true if parsing successful
//            false if failure
//
bool WgtWidget::parseManifest(const QString& path,const bool force)
{
    bool status = true;
    if (!force && m_manifest) {
        // Manifest already parsed and not forcing a re-parse
        return status;
    }

    // Create a NEW m_manifest, clean up old one
    if (m_manifest) {
       delete m_manifest;
       m_manifest = 0;
    }

   m_manifest = new W3cXmlPlugin;
   if (path.isEmpty()) {
      qDebug() << "WgtWidget::parseManifest path=widgetUnZipPath=" << widgetUnZipPath();
      m_manifest->setdir(widgetUnZipPath());
   }
   else {
       qDebug() << "WgtWidget::parseManifest path=" << path;
       m_manifest->setdir(path);
   }

   status = m_manifest->process();
   if (status) {
       widgetProperties();
   }

    return status;
}



// function to convert WgtWidet to WgzWidget
// prereq: should parseManifest before calling toWgz()
//
// return:
//    SuperWidget*  a poiner to WgzWidget
//
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
SuperWidget* WgtWidget::toWgz()
{

    WgzWidget *newWidget = new WgzWidget(m_widgetInstallPath);
    WidgetManifestInfo *newManifest = new WidgetInfoPList();

    if (!getWidgetManifestInfo()) return 0;

    QDir installDir (getWidgetManifestInfo()->installedPath());
    newManifest->setdir(installDir);

    AttributeMap* newMap = new AttributeMap();
    AttributeMap oldMap = getWidgetManifestInfo()->getDictionary();

    QList<QString> keyList = oldMap.keys();


    for (int i=0;i<keyList.count();i++)
    {
        if (!getWidgetMap()->key(keyList.at(i)).isEmpty())
        {
            newMap->insert(getWidgetMap()->key(keyList.at(i)),oldMap.value(keyList.at(i)));
            qDebug() << "toWgz: key=" << getWidgetMap()->key(keyList.at(i)) << " value = " <<oldMap.value(keyList.at(i));

        }

    }


    newManifest->setDictionary(newMap);

    // update widget variables
    newWidget->setWidgetManifestInfo(*newManifest);
    newWidget->setWidgetBundlePath(widgetBundlePath());
    newWidget->setWidgetUnZipPath(widgetUnZipPath());
    newWidget->setWidgetInstallPath(widgetInstallPath());

    return newWidget;
}
#endif


//
// function to find widget start file
// parameters:
//    startFile    returns a string that specifies startFile of widget
//    path         path to widget files
// return:
//    bool         true if start file found
//                 false if start file is not found
//
bool WgtWidget::findStartFile(QString& startFile, const QString& path)
{
    // Most widgets (not all) call findStartFile numerous times. Check if we
    // already found the start file, this eliminates searching the
    // file system, which takes TOO MUCH TIME!
    bool savedStartFile = getStartFile(startFile, path);
    if (savedStartFile) {
        return true;
    }

    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo) {
        return false;
    }

    QString contentSrc = manifestinfo->value(W3CSettingsKey::WIDGET_CONTENT, QString("src"));
    if (!contentSrc.isEmpty()) {
        startFile = contentSrc ;
    }

    // If can't locate, try some default filenames.
    QString widgetPath(path);
    if (path.isEmpty()) {
        widgetPath = widgetRootPath();
    }

    QStringList defaultFiles;
    if (!startFile.isEmpty()) {
        defaultFiles.append(startFile);
    }
    defaultFiles.append("index.htm");
    defaultFiles.append("index.html");
    defaultFiles.append("index.svg");
    defaultFiles.append("index.xhtml");
    defaultFiles.append("index.xht");

    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    QString myLang = m_settings->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty() || myLang.compare(" ") == 0)
        myLang = QLocale::system().name().toLower().replace(QString("_"),QString("-"));

    foreach (QString fileName, defaultFiles) {
        QString locFile = getLocalizedFile(fileName, myLang, false, widgetPath);
        if (!locFile.isEmpty() && QFile::exists(widgetPath + QDir::separator() + locFile)) {
            qDebug() << "Localized start file: " << locFile;
            startFile = locFile;
            saveStartFile(startFile, widgetPath);
            return true;
        }
    }

    return false;
}

//
// function to find widget icon files
// parameters:
//    icons    returns a stringlist that specifies icons of widget
//    path     path to widget files
// return:
//    bool     true if icon found
//             false if icon is not found
//
bool WgtWidget::findIcons(QStringList& icons, const QString& path)
{
    QString widgetPath(path);
    qDebug() << "WgtWidget::findIcons ....";

    if (path.isEmpty()) {
        widgetPath=widgetRootPath();
    }

    QString absPath = widgetRootPath();
    if (!absPath.endsWith(QDir::separator())) {
        absPath = absPath+QDir::separator();
    }

    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo) {
        return false ;
    }

    QStringList iconFiles;
    int count = manifestinfo->count(W3CSettingsKey::WIDGET_ICON);

    for (int iter = 1; iter <= count; ++iter) {
        QString icon = manifestinfo->value(W3CSettingsKey::WIDGET_ICON, iter, QString("src"));
        if (!icon.isEmpty()) {
            //icons.append(icon);
            qDebug() <<"Found icon" << icon;

#if defined(Q_OS_SYMBIAN) || defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
            if (icon.endsWith(".png",Qt::CaseInsensitive)) {
                // Symbian and Maemo can only handle png
#endif
                iconFiles.append(icon);

#if defined(Q_OS_SYMBIAN) || defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
            }
#endif

        }
    }

    iconFiles.append("icon.png");
#if !defined(Q_OS_SYMBIAN) && !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6)
    iconFiles.append("icon.gif");
    iconFiles.append("icon.ico");
    iconFiles.append("icon.svg");
    iconFiles.append("icon.jpeg");
#endif

    icons.clear();

    QString myLang = WRT::WrtSettings::createWrtSettings()->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty())
        myLang = QLocale::system().name().toLower().replace(QString("_"), QString("-"));

    foreach (QString fileName, iconFiles) {
        QString locFile = getLocalizedFile(fileName, myLang, false, absPath);
        if (!locFile.isEmpty() && QFile::exists(widgetPath + QDir::separator() + locFile)) {
            qDebug() << "Localized icon: " << locFile;
            icons.append(QDir::toNativeSeparators(locFile));
        }
    }

    return icons.count();
}

//
// function to generate Manifest
// prereq:    should parseManifest before calling writeManifest()
//
// parameters:
//    path    path to write the manifest. Default is /tmp/config.xml
//
void WgtWidget:: writeManifest(const QString& path)
{
    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo() ;
    if (!manifestinfo)
        return ;

    manifestinfo->WriteManifestFile(path) ;
}

//
// function to get values from dictionary
// prereq:      should parseManifest before calling value()
//
// parameters:
//    key       key to query the dictionary
// return:
//    QString   value matching key from dictionary
//
QString WgtWidget::value(const QString& key, const QString & attribute)
{
    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo)   {
        return "";
    }

    // TODO : This is copy-paste code. Should be on constructor and only once.
    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    QString myLang = m_settings->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty() || myLang.compare(" ") == 0) {
        QLocale language = QLocale::system();
        myLang = language.name().toLower();
        myLang.replace(QString("_"),QString("-"));
    }

    QString val = manifestinfo->value(key, attribute, myLang);
    if (val.isEmpty()) {
        val = manifestinfo->value(key, attribute, "");
    }

    return val;
}

bool WgtWidget::contains( const QString & key, const QString & attribute ) {
    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo)   {
        return false;
    }

    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    QString myLang = m_settings->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty() || myLang.compare(" ") == 0) {
        QLocale language = QLocale::system();
        myLang = language.name().toLower();
        myLang.replace(QString("_"),QString("-"));
    }

    if (attribute.isEmpty()) {
        bool hasElement = manifestinfo->getDictionary().contains(myLang + "/" + key);
        if (!hasElement) {
            hasElement = manifestinfo->getDictionary().contains(key);
        }
        return hasElement;
    } else {
        QVariant val = manifestinfo->value(key, attribute, myLang);
        if (!val.isNull()) {
            return true;
        } else {
            val = manifestinfo->value(myLang + "/" + key, attribute, myLang);
            return !val.isNull();
        }

    }
}


//
// function to get widgetProperties
// prereq: should parseManifest before calling ()
//
// return:
//    WidgetProperties    specific widget properties required for WebAppRegistry
//                        if you want more,just use value()
//
WidgetProperties* WgtWidget::widgetProperties()
{
    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo) {
        // No manifestInfo, probably didn't call parseManifest() before this
        return 0;
    }

    if (m_widgetProperties != 0) {
        return m_widgetProperties;
    }

    qDebug() << "Getting Widget Properties";

    WidgetProperties *props = new WidgetProperties;

    //props->setId(value(W3CSettingsKey::WIDGET_ID));
    QString widgetid = manifestinfo->value(W3CSettingsKey::WIDGET_ID );

    /**
     *  1) Try to: set id for widget from config.xml id field
     *  2) Set QUid
     *
     *  This avoid widgets installation failure in following case:
     *  - Install a widget which does not have id in field config.xml (id will be "", this hashed to zero)
     *  - Install a second widget that does not have id field in config.xml (install of second widget failed)
     *
     */
    if (widgetid.isEmpty()) {
        widgetid = QUuid::createUuid().toString();
    }
    widgetid = QString::number(qHash(widgetid));
    props->setId(widgetid);


    // if this is a shared library and not a launchable widget set the type to
    // shared library to prevent this from being confused as a launchable widget
    props->setSharedLibrary(!manifestinfo->value(ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_FOLDER).isEmpty());
    props->setSharedLibraryWidget(manifestinfo->value(ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_WIDGET).startsWith("T",Qt::CaseInsensitive));

    if (props->isSharedLibrary() && !props->isSharedLibraryWidget())
        props->setType(WIDGET_PACKAGE_FORMAT_SHARED_LIBRARY);
    else
        props->setType(WIDGET_PACKAGE_FORMAT_WGT);

    if (props->isSharedLibrary()) {
        QString path(SHARED_LIBRARY_PATH);
        path.append(manifestinfo->value(ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_FOLDER));
        props->setInstallPath(path);
    } else {
        props->setInstallPath(widgetInstallPath()+QDir::separator()+widgetid);
    }

    props->setSource(widgetBundlePath());
    props->setInfoPList(getWidgetManifestInfo()->getDictionary());
    props->setSize(getSize());
    props->setResourcePath(SuperWidget::resourcePath(props->installPath()));

    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    QString myLang = m_settings->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty() || myLang.compare(" ") == 0) {
        QLocale language = QLocale::system();
        myLang = language.name().toLower();
        myLang.replace(QString("_"),QString("-"));
    }

    QString myTitle = manifestinfo->value(W3CSettingsKey::WIDGET_NAME, QString(""), myLang);
    if (!myLang.isEmpty()) {
        if (!myTitle.isEmpty()) {
            props->setTitle(myTitle);
        }
    }

    if (myLang.isEmpty() || myTitle.isEmpty()) {
        QString wName = manifestinfo->value(W3CSettingsKey::WIDGET_NAME);
        if (wName.isEmpty()) {
            wName = "NoName";
        }
        props->setTitle(wName);
    }

    QString myTitleDir = manifestinfo->value(W3CSettingsKey::WIDGET_NAME, QString("its:dir"));
    if (!myTitle.isEmpty() && !myTitleDir.isEmpty() && isDirectionValid(myTitleDir)) {
        props->setTitleDir(myTitleDir);
    }

    QStringList icons;
    if (findIcons(icons)) {
        QString iconPath = props->installPath() + icons.at(0);
        if (!icons.at(0).startsWith(QDir::separator()) && !props->installPath().endsWith(QDir::separator())) {
            iconPath = props->installPath()+QDir::separator()+icons.at(0);
        }
        if (icons.at(0).startsWith(QDir::separator()) && props->installPath().endsWith(QDir::separator())) {
            iconPath = props->installPath();
            iconPath.chop(1);
            iconPath.append(icons.at(0));
        }
        props->setIconPath(iconPath);
    } else {
        qDebug() << "No icons found, using the default one";
        props->setIconPath(":/resource/default_widget_icon.png");
    }
    qDebug() << "Icon path = " << props->iconPath();

    QString myDesc    = manifestinfo->value(W3CSettingsKey::WIDGET_DESCRIPTION, QString(""), myLang);
    QString myDescDir = manifestinfo->value(W3CSettingsKey::WIDGET_DESCRIPTION, QString("its:dir"));
    if (!myDesc.isEmpty() && !myDescDir.isEmpty() && isDirectionValid(myDescDir)) {
        props->setDescriptionDir(myDescDir);
    }

    QString myAuthor    = manifestinfo->value(W3CSettingsKey::WIDGET_AUTHOR, QString(""), myLang);
    QString myAuthorDir = manifestinfo->value(W3CSettingsKey::WIDGET_AUTHOR, QString("its:dir"));
    if (!myAuthor.isEmpty() && !myAuthorDir.isEmpty() && isDirectionValid(myAuthorDir)) {
        props->setAuthorDir(myAuthorDir);
    }

    QString myLic    = manifestinfo->value(W3CSettingsKey::WIDGET_LICENSE, QString(""), myLang);
    QString myLicDir = manifestinfo->value(W3CSettingsKey::WIDGET_LICENSE, QString("its:dir"));
    if (!myLic.isEmpty() && !myLicDir.isEmpty() && isDirectionValid(myLicDir)) {
        props->setLicenseDir(myLicDir);
    }

    m_widgetProperties = props;
    qDebug() << "setting widget properties";

    return props;
}

//
// function to install widget
//
// parameter:
//    silent   true for silent install, false otherwise
//    QWidget* pointer to parent widget
// return:
//    bool     true if successful
//
WidgetInstallError WgtWidget::install(bool update)
{
    m_continueInstallation = false;

    // unzip widget bundle
    QString bundlePath = widgetBundlePath();

#if !defined(Q_OS_MAEMO5) && !defined(Q_OS_MAEMO6)
    setWidgetUnZipPath(widgetInstallPath()+QDir::separator()+"widgetUnZipPath");
#endif

    // Maemo 5 unzips the widget under /home/user/.cache/<timestamp>/
    // most likely similar decision needs to be taken on Maemo 6 too
    // /home has roughly 2 gigabytes of free space by default thus we
    //

#if defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
    // add some randomness in to unzip path so that we won't unzip multiple
    // widgets in to the same directory if multiple installations are happening
    // at the same time
    QString suffix("%1");
    qsrand(QTime::currentTime().msec());
    suffix = suffix.arg(qrand());
    setWidgetUnZipPath(widgetUnZipPath() + suffix);
#endif

    setWidgetRootPath(widgetUnZipPath());

    emit installProgress(10);
    if (QFileInfo(bundlePath).isFile()) {

        // make sure there's enough free disk space
        if (!checkDiskSpaceForInstallation(bundlePath)) {
            emit installationError(WidgetInsufficientDiskSpace);
            cleanup();
            return WidgetInsufficientDiskSpace;
        }

        // unzip widget bundle
        if (!unZipBundle(bundlePath)) {
            emit installationError(WidgetUnZipBundleFailed);
            cleanup();
            return WidgetUnZipBundleFailed;
        }
    }
    else {

#ifdef Q_OS_SYMBIAN
        // we have an unzipped widget folder
        // for security reasons, any process other than backup-restore should not be
        // allowed to install widget from an unzipped folder
        RProcess me;
        TUidType uidType(me.Type());
        TUint32 uid3 = uidType[2].iUid;
        qDebug() << "UID3 of this process : " << uid3;
        if (uid3 != BACKUP_RESTORE_UID) {
            qDebug() << "WgtWidget::install() - Aborting install. This is not Backup-Restore process";
            return WidgetInstallPermissionFailed;
        }
#endif

        setWidgetUnZipPath(bundlePath);
        setWidgetRootPath(widgetUnZipPath());
    }

    emit installProgress(20);
    QStringList digSig;

    WidgetInstallError errorCode;

    // Widget which has digital signatures must have valid dig sigs.
    if (!validateDigitalSignatures(digSig, errorCode))
    {
        m_continueInstallation = false;
        // If m_disableUnsignWidgetSignCheck is true and if widget is an unsigned widget than installation should be continued.
        // Silent mode is signaled back from WidgetManager.cpp.
#ifdef BLOCK_UNSIGNED_WIDGETS
        emit installationError(WidgetSignatureValidationFailed);
#else
        emit aboutToInstallUntrustedWidget(errorCode);
#endif
        if (!m_continueInstallation && !m_disableUnsignWidgetSignCheck) {
            cleanup();
            return WidgetSignatureValidationFailed;
        }
    }
    bool isValidSigned(errorCode == WidgetValidSignature);

    emit installProgress(30);

    // parse manifest
    if (!parseManifest())
    {
        emit installationError(WidgetParseManifestFailed);
        cleanup();
        return WidgetParseManifestFailed;
    }

    emit installProgress(40);

    WidgetProperties* props = widgetProperties();

    // shared libraries must be signed
    if (props->isSharedLibrary() && !isValidSigned) {
        emit installationError(WidgetSharedLibraryNotSigned);
        cleanup();
        return WidgetSharedLibraryNotSigned;
    }

    QString updateWgtId = getWidgetManifestInfo()->value(W3CSettingsKey::WIDGET_ID, QString("id"));
    if (update && WebAppRegistry::instance()->isRegistered(updateWgtId)) {
        qDebug() << "updateWgtId" << updateWgtId;
        QString srcPath = widgetUnZipPath();
        QString dstPath = props->installPath();
        qDebug() << "install path: " << dstPath;
        qDebug() << "unzip path  : " << srcPath;

        // create a backup directory before this widget is updated
        QString updateBackupPath = widgetInstallPath()+QDir::separator()+"widgetUpdateBackupPath"
            +QDir::separator()+QString::number(qHash(props->id()));
        QDir updateBackupDir(updateBackupPath);
        if (!updateBackupDir.exists()) {
            if (!updateBackupDir.mkpath(updateBackupPath)) {
            cleanup();
            return WidgetSystemError;
            }
        }

        // backup the widget to be updated
        SuperWidget::rmDir(updateBackupPath);
        if (!m_widgetInstaller->update(dstPath,updateBackupPath, props->id())) {
            SuperWidget::rmDir(updateBackupPath);
            cleanup();
            return WidgetSystemError;
        }

        // clean dest dir before update files
        SuperWidget::rmDir(dstPath);
        if (!m_widgetInstaller->update(srcPath,dstPath, props->id())) {
            // restore from updateBackup widget in case update failed
            SuperWidget::rmDir(dstPath);
            m_widgetInstaller->update(updateBackupPath,dstPath, props->id());
            SuperWidget::rmDir(updateBackupPath);
            cleanup();
            return WidgetSystemError;
        }

        WebAppRegistry::instance()->setWebAppVersion(props->id(),
                                                     props->plistValue(W3CSettingsKey::WIDGET_VERSION),
                                                     props->iconPath() );

        SuperWidget::rmDir(updateBackupPath);
        cleanup();
        return WidgetInstallSuccess;
    }

    emit installProgress(50);

    if (WebAppRegistry::instance()->isRegistered(props->id())) {
        m_continueInstallation = false;
        emit aboutToReplaceExistingWidget(props->title());
        if (!m_continueInstallation) {
            cleanup();
            return WidgetUserConfirmFailed;
        }

#ifdef Q_OS_SYMBIAN
        emit installProgress(55);
        TRAP_IGNORE( restartNotificationL());
#endif
        emit installProgress(60);

        if (SuperWidget::uninstall(props->id(), false) == WidgetUninstallFailed)
        {
            emit installationError(WidgetReplaceFailed);
            cleanup();
            return WidgetReplaceFailed;
        }
    }

    WidgetFeatures features;
    bool res = findFeatures(features);
    QString trustDn;
    WRT::SecSession* secureSession;
    if (createSecuritySession(&secureSession,trustDn)) {
        if (res || (trustDn.compare(TrustedDomain,Qt::CaseInsensitive) == 0)) {
            qDebug() << "WgtWidget::install(), Found features";
            if (!isFeatureAllowed(features,secureSession)) {
                qDebug() << "WgtWidget::install(), Feature not allowed";
                 emit installationError(WidgetCapabilityNotAllowed);
                 cleanup();
                 delete secureSession;
                 return WidgetCapabilityNotAllowed;
            }
        }
    delete secureSession;
    } else {
        return WidgetSystemError;
    }
    if (m_mapping.getWidgetType() == WidgetTypeJIL)
        props->setType(WIDGET_PACKAGE_FORMAT_JIL);
    emit installProgress(70);

#ifdef Q_OS_SYMBIAN
    QString domainName = getTrustDomain();
    QString procUid = getProcUid(domainName);
    updatePaths(procUid);
    const QString ProcessUid (PROCESS_UID) ;
    AttributeMap map = props->plist();
    map.insert( ProcessUid ,procUid);
    props->setInfoPList(map);
#endif

    // if the start file doesn't exist, treat as invalid widget
    QString startFile;
    if (!findStartFile(startFile, widgetUnZipPath())) {
        qDebug()  << "WgtWidget::install() failed: start file not found";
        emit installationError(WidgetStartFileNotFound);
        cleanup();
        return WidgetStartFileNotFound;
    }

    emit installProgress(80);

    QString srcPath = widgetUnZipPath();
    QString dstPath = props->installPath();
    qDebug() << "install path: " << dstPath;
    qDebug() << "unzip path  : " << srcPath;

    if (!m_widgetInstaller->install(srcPath,dstPath, props->id())) {
        qDebug() << "WgtWidget::install(), Widget install failed";
        emit installationError(WidgetPlatformSpecificInstallFailed);
        cleanup();
        cleanup(dstPath);
        return WidgetPlatformSpecificInstallFailed;
    }

    emit installProgress(90);

#if !defined(POST_REGISTER_WEBWIDGET)
    if (!registerWidget(startFile)) {
        qDebug() << "WgtWidget::install(), Widget register failed";
        emit installationError(WidgetRegistrationFailed);
        cleanup();
        cleanup(dstPath);
        return WidgetRegistrationFailed;
    }
#endif

    QString nativeId = m_widgetInstaller->nativeId();
    if (!nativeId.isEmpty()) {
        WebAppRegistry::instance()->setNativeId(props->id(), nativeId);
    }

    //write widget certificateaki to webappregistry this is needed for runtime capability checks
    if (!m_validCertificateAKI.isEmpty())
       WebAppRegistry::instance()->setCertificateAki(props->id(),m_validCertificateAKI);

    emit installProgress(100);
    emit installationSucceed();
    cleanup();
    return WidgetInstallSuccess;
}

//
// this function provides the widget launcher path
// parameters:
//    pkgPath  path to installed widget
//
// return
//    QString  absolute path to start file
//
QString WgtWidget::launcherPath(const QString &pkgPath)
{
    WidgetProperties *widget = SuperWidget::getProperties(pkgPath);
    if (widget)
    {
        QString startFile;
        if (findStartFile(startFile,widget->installPath())) {
            qDebug() << "WgtWidget::launcherPath start file =" << startFile;
            return (widget->installPath()+QDir::separator()+startFile);
        }
    }

    return "";
}


//
// function to find widget features/capabilities
// prereq:        should parseManifest before calling findFeatures
//
// parameters:
//    features    returns WidgetFeatures
//    path        path to widget files
// return:
//    bool        true if features found
//                false if features not found
//
bool WgtWidget::findFeatures(WidgetFeatures& features, const QString& path)
{
    QString widgetPath(path);
    qDebug() << "WgtWidget::findFeatures ....";
    if (path.isEmpty()) widgetPath=widgetRootPath();

    WidgetManifestInfo *manifestinfo = getWidgetManifestInfo();
    if (!manifestinfo) return false;

    int featureCount = manifestinfo->count(W3CSettingsKey::WIDGET_FEATURE);

    //Change for incorporating required attribute
    for (int iter = 0 ; iter < featureCount ; ++iter)   {
        QString featureName = manifestinfo->value(W3CSettingsKey::WIDGET_FEATURE , iter + 1 , QString("name"));
        QString required_attr = manifestinfo->value(W3CSettingsKey::WIDGET_FEATURE , iter + 1 , QString("required"));
        if (required_attr.isEmpty())
            required_attr = "true";
        if (!featureName.isEmpty()) {
            features.insert(featureName ,required_attr);
        }
    }
    return (features.count() > 0);
}

// returns the AKI of signer of the widget's certificate if the widget was signed and certificate was validated
bool WgtWidget::getCertificateAKI(QString& aki) {
    if (!m_validCertificateAKI.isNull() && !m_validCertificateAKI.isEmpty()) {
        aki = m_validCertificateAKI;
        return true;
    }
    return false;
}

// returns the trust domain associated with this widget
QString WgtWidget::getTrustDomain() {
    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    // get policy files from WrtSettings
    QString trustPath = m_settings->valueAsString("SecurityTrustPolicyFile");

    // create a trust session
    WRT::TrustSession trustSession(trustPath);

    QString aki;
    getCertificateAKI(aki);
    QString domain = trustSession.domainFor("certificate", aki);
    return domain;
}

// function to check if feature is allowed
// parameters:
//    features    list of features from manifest
// return:
//    bool        true if successful
//
bool WgtWidget::isFeatureAllowed(WidgetFeatures& features, WRT::SecSession* secureSession ,bool runtime)
{

    QList<QString> required_capabilities;
    QList<QString> optional_capabilities;
    if (!m_mapping.getCapabilities(features,required_capabilities,optional_capabilities))
        return false;

    // check if capability is allowed
    // According to current implementation of security manager, even if one
    // capability is not allowed isAllowed will return false
    // If trustPath or secPath are empty, isAllowed() will return true

    if (secureSession->isAllowed(required_capabilities)) {
        // done to add these caps to allowed capabilities in secsession
        //secureSession.isAllowed(optional_capabilities);
        QList<QString> caps;
        caps.append(required_capabilities);
        caps.append(optional_capabilities);
        QList<QString> services;
        m_mapping.getServiceNames(features,services);
        if (!services.isEmpty() && !runtime) {
            m_continueInstallation = false;
            emit aboutToInstallWidgetWithFeatures(services);
            if (!m_continueInstallation)
                return false;
        }
        //store the path of widget installed
        WidgetProperties *props = SuperWidget::getProperties();
        if (props) {
            secureSession->setClientInfo(WRT::KWIDGETPATH, props->installPath());
            QByteArray secureSessionByteArr;
            secureSession->persist(secureSessionByteArr);
            QString secureSessionString(secureSessionByteArr);

            m_secSessionStr = secureSessionString;
            WidgetProperties* props = widgetProperties();
            props->setSecureSessionString(getSecureSessionString());
        }
        return true;
    }

    return false;
}

//
// function to install widget through SISX
// This is a temp fix till we get Nokia PKI
//
// parameter:
//    silent   true for silent install, false otherwise
//    QWidget* pointer to parent widget
// return:
//    bool     true if successful
//
WidgetInstallError WgtWidget::install(const QString& sigId, const bool update)
{
    // unzip widget bundle
    QString bundlePath = widgetBundlePath();
    QFileInfo widgetFileInfo(bundlePath);

    setWidgetUnZipPath(widgetInstallPath()+QDir::separator()+"widgetUnZipPath");
    setWidgetRootPath(widgetUnZipPath());

    if (widgetFileInfo.isFile())
    {
        // unzip widget bundle
        if (!unZipBundle(bundlePath))
        {
            emit installationError(WidgetUnZipBundleFailed);
            cleanup();
            return WidgetUnZipBundleFailed;
        }
    }
    else
    {

#ifdef Q_OS_SYMBIAN
        // we have an unzipped widget folder
        // for security reasons, any process other than backup-restore should not be
        // allowed to install widget from an unzipped folder
        RProcess me;
        TUidType uidType(me.Type());
        TUint32 uid3 = uidType[2].iUid;
        qDebug() << "UID3 of this process : " << uid3;
        if (uid3 != BACKUP_RESTORE_UID) {
            qDebug() << "WgtWidget::install() - Aborting install. This is not Backup-Restore process";
            return WidgetInstallPermissionFailed;
        }
#endif

        setWidgetUnZipPath(bundlePath);
        setWidgetRootPath(widgetUnZipPath());
    }

    // parse manifest
    if (!parseManifest())
    {
        emit installationError(WidgetParseManifestFailed);
        cleanup();
        return WidgetParseManifestFailed;
    }

    WidgetProperties* props = widgetProperties();

    QString updateWgtId = getWidgetManifestInfo()->value(W3CSettingsKey::WIDGET_ID, QString("id"));
    if (update && WebAppRegistry::instance()->isRegistered(updateWgtId)) {
        qDebug() << "updateWgtId" << updateWgtId;
        QString srcPath = widgetUnZipPath();
        QString dstPath = props->installPath();
        qDebug() << "install path: " << dstPath;
        qDebug() << "unzip path  : " << srcPath;

        // create a backup directory before this widget is updated
        QString updateBackupPath = widgetInstallPath()+QDir::separator()+"widgetUpdateBackupPath"
            +QDir::separator()+QString::number(qHash(props->id()));
        QDir updateBackupDir(updateBackupPath);
        if (!updateBackupDir.exists()) {
            if (!updateBackupDir.mkpath(updateBackupPath)) {
            cleanup();
            return WidgetSystemError;
            }
        }

        // backup the widget to be updated
        SuperWidget::rmDir(updateBackupPath);
        if (!m_widgetInstaller->update(dstPath,updateBackupPath, props->id())) {
            SuperWidget::rmDir(updateBackupPath);
            cleanup();
            return WidgetSystemError;
        }

        // clean dest dir before update files
        SuperWidget::rmDir(dstPath);
        if (!m_widgetInstaller->update(srcPath,dstPath, props->id())) {
            // restore from updateBackup widget in case update failed
            SuperWidget::rmDir(dstPath);
            m_widgetInstaller->update(updateBackupPath,dstPath, props->id());
            SuperWidget::rmDir(updateBackupPath);
            cleanup();
            return WidgetSystemError;
        }

        WebAppRegistry::instance()->setWebAppVersion(props->id(),
                                                     props->plistValue(W3CSettingsKey::WIDGET_VERSION),
                                                     props->iconPath() );

        SuperWidget::rmDir(updateBackupPath);
        cleanup();
        return WidgetInstallSuccess;
    }

    if (WebAppRegistry::instance()->isRegistered(props->id())) {
        m_continueInstallation = false;
        emit aboutToReplaceExistingWidget(props->title());
        if (!m_continueInstallation) {
            cleanup();
            return WidgetUserConfirmFailed;
        }
        if (SuperWidget::uninstall(props->id(), false) == WidgetUninstallFailed)
        {
            emit installationError(WidgetReplaceFailed);
            cleanup();
            return WidgetReplaceFailed;
        }
    }

    WidgetFeatures features;
    bool res = findFeatures(features);
    QString trustDn;
    WRT::SecSession* secureSession;
    if (createSecuritySession(&secureSession,trustDn,sigId)) {
        if (res || (trustDn.compare(TrustedDomain,Qt::CaseInsensitive) == 0)) {
            qDebug() << "WgtWidget::install(), Found features";
            if (!isFeatureAllowed(features,secureSession)) {
                qDebug() << "WgtWidget::install(), Feature not allowed";
                emit installationError(WidgetCapabilityNotAllowed);
                cleanup();
                delete secureSession;
                return WidgetCapabilityNotAllowed;
            }
        }
        delete secureSession;
    } else {
        return WidgetSystemError;
    }

    if (m_mapping.getWidgetType() == WidgetTypeJIL)
        props->setType(WIDGET_PACKAGE_FORMAT_JIL);

    // if the start file doesn't exist, treat as invalid widget
    QString startFile;
    if (!findStartFile(startFile, widgetUnZipPath())) {
        qDebug()  << "WgtWidget::install() failed: start file not found";
        emit installationError(WidgetStartFileNotFound);
        cleanup();
        return WidgetStartFileNotFound;
    }

    QString srcPath = widgetUnZipPath();
    QString dstPath = props->installPath();
    qDebug() << "install path: " << dstPath;
    qDebug() << "unzip path  : " << srcPath;

    if (!m_widgetInstaller->install(srcPath,dstPath, props->id())) {
        qDebug() << "Widget file install failed";
        emit installationError(WidgetPlatformSpecificInstallFailed);
        cleanup();
        cleanup(dstPath);
        return WidgetPlatformSpecificInstallFailed;
    }

#if !defined(POST_REGISTER_WEBWIDGET)
    if (!registerWidget(startFile)) {
        emit installationError(WidgetRegistrationFailed);
        cleanup();
        cleanup(dstPath);
        return WidgetRegistrationFailed;
    }
#endif

    QString nativeId = m_widgetInstaller->nativeId();
    if (!nativeId.isEmpty()) {
        WebAppRegistry::instance()->setNativeId(props->id(), nativeId);
    }

    emit installationSucceed();
    cleanup();
    return WidgetInstallSuccess;
}

#ifdef Q_OS_SYMBIAN
void WgtWidget::restartNotificationL()
{
    WidgetProperties* props = widgetProperties();
    RWidgetRegistryClientSession clientReg;
    User::LeaveIfError( clientReg.Connect() );
    TBuf<120> bundleId(props->id().utf16());
    TUid aUid;
    aUid.iUid = clientReg.GetWidgetUidL(bundleId);
    if (clientReg.IsWidgetInMiniView(aUid)) {
        RApaLsSession apparcSession;
        User::LeaveIfError( apparcSession.Connect() );
        TApaAppInfo info;
        User::LeaveIfError( apparcSession.GetAppInfo( info, aUid ) );
        HBufC* widgetName(info.iFullName.AllocLC());
        const TInt size( 2* widgetName->Length() + 3*sizeof( TUint32 ) );
        CApaCommandLine* cmd( CApaCommandLine::NewLC() );
        HBufC8* opaque( HBufC8::NewLC( size ) );
        RDesWriteStream stream;
        TPtr8 des( opaque->Des() );
        stream.Open( des );
        CleanupClosePushL( stream );
        // Generate the command.
        stream.WriteUint32L( aUid.iUid );
        stream.WriteUint32L( widgetName->Length() );
        stream.WriteL( reinterpret_cast< const TUint8* >( widgetName->Ptr() ),
                       widgetName->Size() );
        // Number 8 is for EWidgetRestart WidgetOperation.
        stream.WriteInt32L( 8 );
        CleanupStack::PopAndDestroy( &stream );
        // Generate command.
        cmd->SetCommandL( EApaCommandBackgroundAndWithoutViews );
        cmd->SetOpaqueDataL( *opaque );
        CleanupStack::PopAndDestroy( opaque );
        cmd->SetExecutableNameL( KLauncherApp );
        User::LeaveIfError( apparcSession.StartApp( *cmd ) );
        CleanupStack::PopAndDestroy( cmd );
        apparcSession.Close();
        CleanupStack::PopAndDestroy( widgetName );
    }
}
#endif

void WgtWidget::disableBackupRestoreValidation(bool disableUnsignWidgetSignCheck)
{
    m_disableUnsignWidgetSignCheck = disableUnsignWidgetSignCheck;
}

// Method used to get the capability lists based on feature name
//
// parameters:
//      features the feature list for which the device-caps are to be determined
//      required_capabilities the list of required capabilities
//      optional_capabilities the list of optional capabilities
//
// @return:
//      bool false if plugin load fails or feature is not found in the mapping file

bool WgtWidget::createSecuritySession(WRT::SecSession** secureSession,QString& trustDomain,const QString& sigId,bool runtime)
{
    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();

    // get policy files from WrtSettings
    QString trustPath =     m_settings->valueAsString("SecurityTrustPolicyFile");
    QString secPath =     m_settings->valueAsString("SecurityAccessPolicyFile");


    // create a trust session
    WRT::TrustSession trustSession(trustPath);

    //At runtime read the widget certificateAki from webappregistry
    if (runtime) {
       WebAppRegistry * registry = WebAppRegistry::instance();
       WebAppInfo webInfo;
       WidgetProperties* props = widgetProperties();
       if (props) {
       registry->isRegistered(props->id(),webInfo);
           m_validCertificateAKI = webInfo.certificateAki();
       }
    }

    // get trust domain
    if (!sigId.isEmpty()) {
        trustDomain = trustSession.domainFor("certificate", sigId);
    } else {
        trustDomain = trustSession.domainFor("certificate",m_validCertificateAKI);
    }

    // if we end up with the default domain, make sure it's UntrustedWidgets
    // redundant - change default domain in browser_access_policy.xml
    if (trustDomain == trustSession.domainFor("certificate", "")) {
        trustDomain = "UntrustedWidgets";
    }

    // create secure session
    *secureSession = new WRT::SecSession(secPath,trustDomain,"");
    return !(*secureSession == NULL);
}

bool  WgtWidget::isDirectionValid(QString AttributeValue)
{
    if (((AttributeValue.compare("ltr",Qt::CaseInsensitive)) == 0) ||
       ((AttributeValue.compare("rtl",Qt::CaseInsensitive)) == 0) ||
       ((AttributeValue.compare("lro",Qt::CaseInsensitive)) == 0) ||
       ((AttributeValue.compare("rlo",Qt::CaseInsensitive)) == 0))   {
        return true;
    }
    return false;
}

#ifdef Q_OS_SYMBIAN
void WgtWidget::updatePaths(const QString& domainNameUid)
{
    QString currUid = DEFAULT_DIRECTORY;
    m_widgetInstallPath.replace(currUid, domainNameUid, Qt::CaseInsensitive);
    m_widgetRootPath.replace(currUid, domainNameUid, Qt::CaseInsensitive);

    WidgetProperties* props = widgetProperties();
    props->setInstallPath(props->installPath().replace(currUid, domainNameUid, Qt::CaseInsensitive));
    QString iconpath = props->iconPath();
    props->setIconPath(iconpath.replace(currUid, domainNameUid, Qt::CaseInsensitive));
    props->setResourcePath(props->resourcePath().replace(currUid, domainNameUid, Qt::CaseInsensitive));

}

QString WgtWidget::getProcUid(const QString& domain) const
{
    QSettings settings(INI_PATH,QSettings::IniFormat);
    QString uid = settings.value("DomainNames/" + domain).toString();
    if (uid.isEmpty())
        return settings.value("DomainNames/others").toString();
    return uid;
}
#endif
