/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QVariant>
#include <QFile>
#include <QObject>
#include <QSignalMapper>
#include <qwebframe.h>

#include "jsviewmenu.h"
#include "jsmenuitem.h"
#include "jsmenuitemfactory.h"
#include "action.h"
#include "actionadapter.h"

namespace WRT {


/*!
    \class WRT::JSViewMenu
    \brief The WRT::JSViewMenu

    \inmodule WidgetCore

    WRT::JSViewMenu evaluates menu.js JavaScript file into window object.
*/

/*!
    Constructs an instance of JSViewMenu that evaluates menu.js resource into current frame. Native part is exposed to menuCPP.
*/
JSViewMenu::JSViewMenu()
    : JSMenu("menu.js", "menuCPP")
    , m_title("")
{
}

JSViewMenu::~JSViewMenu()
{
}

/*!
    Returns the title of the menu.
*/
QString JSViewMenu::title()
{
    // TODO : Should this reflect html title if not title are being set.
    return m_title;
}

/*!
    Sets \a title to the menu and emits titleChanged with \a title.
*/
void JSViewMenu::setTitle(QString title)
{
    m_title = title;
    emit titleChanged(title);
}

/*!
    Appends given native instance \a id to menu.
*/
void JSViewMenu::append(int id)
{
    if (m_actionAdapter) {
        JSMenuItem* item = JSMenuItemFactory::item(id);
        if (item) {
            Action* menuItem = item->action();
            menuItem->setLocation(Action::ViewMenuLocation);
            m_actionAdapter->addAction(menuItem);
            m_mainLevelMenu.append(menuItem);
        }
    }
}

/*!
    Removes given native instance \a id from menu.
*/
void JSViewMenu::remove(int id)
{
    if (m_actionAdapter) {
        JSMenuItem* item = JSMenuItemFactory::item(id);
        if (item) {
            Action* menuItem = item->action();
            m_actionAdapter->removeAction(menuItem);
            m_mainLevelMenu.removeAll(menuItem);
        }
    }
}

/*!
    Inserts given native instance \a id to menu \a beforeIndex.
*/
void JSViewMenu::insert(int id, int beforeIndex) {
    if (m_actionAdapter) {
        JSMenuItem* item = JSMenuItemFactory::item(id);
        if (item) {
            Action* menuItem = item->action();
            menuItem->setLocation(Action::ViewMenuLocation);

            // Append invalid indexes at the end, then menuItem is appended into menu.
            if (beforeIndex >= 0 && beforeIndex < m_mainLevelMenu.length()) {
                Action* beforeItem = m_mainLevelMenu.at(beforeIndex);
                m_actionAdapter->insertAction(beforeItem, menuItem);
            } else {
                m_actionAdapter->addAction(menuItem);
                m_mainLevelMenu.append(menuItem);
            }
        }
    }
}

/*!
    Clears menu items from menu.
*/
void JSViewMenu::clear() {
    if (m_actionAdapter) {
        QList<Action*> actionsToClear = m_actionAdapter->actions(Action::ViewMenuLocation);
        m_actionAdapter->removeActions(actionsToClear);
        m_mainLevelMenu.clear();
    }
}

}
