/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QFile>
#include <QByteArray>
#include <QTextStream>
#include <qwebpage.h>
#include <qwebframe.h>
#include <QDebug>

#include "widgetmanagerconstants.h"
#include "jsresourcewrapper.h"
#include "jsresourcewrapper_p.h"

namespace WRT {

/*!
    \class WRT::JSResourceWrapper
    \brief WRT::JSResourceWrapper wraps a JavaScript resource files that should be evaluated into QWebPage current frame.

    JavaScript is evaluated once QWebPage is ready to be used, in practice once javaScriptWindowObjectCleared of QWebFrame is emitted.
*/

/*!
    Constructs an instance of WRT::JSResourceWrapper. Resource, \a jsResource, is always loaded from js folder as resource (:js/<jsResource>).
    Thus, only name of JavaScript resource file without path is needed (for instance menu.js). This holds
    \a jsObjectName of the exposed JavaScript object. Give an empty string when object is not supposed to be exported to JavaScript.
    JavaScript evaluation is controlled with \a javaScriptHandling.
*/
JSResourceWrapper::JSResourceWrapper(QString jsResource, QString jsObjectName, JavaScriptHandling javaScriptHandling)
    : QObject() {
    d = new JSResourceWrapperPrivate(jsResource, jsObjectName, javaScriptHandling);

}

JSResourceWrapper::~JSResourceWrapper() {
    delete d;
}

/*!
    Sets \page for JavaScript resource wrapper where JavaScript resource file is evaluated.
*/
void JSResourceWrapper::setWebPage(QWebPage* page) {
    d->setWebPage(page);
    connect(d->m_page, SIGNAL(loadStarted()), this, SLOT(loadStarted()));
    connect(d->m_page, SIGNAL(loadFinished(bool)), this, SLOT(loadFinished(bool)));
}

bool JSResourceWrapper::isLoading() {
    return d->m_loadCount > 0;
}

/*!
    Returns page of JavaScript resource wrapper.
*/
QWebPage * JSResourceWrapper::webPage() {
    return d->m_page;
}

QString JSResourceWrapper::name() const {
    return d->name();
}

QString JSResourceWrapper::toSource() {
    return "[native code]";
}

/*!
    Evaluates the given JavaScript resource file to current frame of the QWebPage.
*/
void JSResourceWrapper::pageOnReload() {
    this->beforeJavaScriptEvaluation();
    d->eval();
}

/*!
    This function is called before JavaScript resource file of is evaluated. You should override this
    function in sub-class if you want to do something before this JavaScript resource is being evaluated to current frame.

    The default implementation does nothing.
*/
void JSResourceWrapper::beforeJavaScriptEvaluation() {
}


void JSResourceWrapper::loadStarted() {
    d->m_loadCount++;
}

void JSResourceWrapper::loadFinished(bool ok) {
    d->m_loadCount--;

    if (d->m_loadCount < 1) {
        d->m_loadCount = 0;
        emit loadFinished();
    }
}

JSResourceWrapperPrivate::JSResourceWrapperPrivate(QString jsResource, QString jsObjectName,
        JSResourceWrapper::JavaScriptHandling javaScriptHandling)
 : m_page(0)
 , m_jsResource(jsResource)
 , m_jsObjectName(jsObjectName)
 , m_loadCount(0)
 , m_jsHandling(javaScriptHandling)
{
}

JSResourceWrapperPrivate::~JSResourceWrapperPrivate() {
}


void JSResourceWrapperPrivate::setWebPage(QWebPage * page) {
    m_page = page;
}

QString JSResourceWrapperPrivate::name() {
    return m_jsObjectName;
}

void JSResourceWrapperPrivate::eval() {
    if (m_jsHandling == JSResourceWrapper::EvalauteJavaScript) {
        QString path(":/js/" + m_jsResource);

        if (!QFile::exists(path)) {
            path = SHARED_LIBRARY_PATH + m_jsResource;
        }

        QString errorMsg;
        if (!QFile::exists(path)) {
            errorMsg = "Could not find JS file " + m_jsResource;
            qWarning(errorMsg.toLatin1().constData());
            return;
        }

        QFile f(path);
        if (!f.open(QIODevice::ReadOnly)) {
            errorMsg = "Could not open JS file " + m_jsResource;
            qWarning(errorMsg.toLatin1().constData());
            return;
        }

        QTextStream in(&f);
        QString jsFile = in.readAll();

        m_page->currentFrame()->evaluateJavaScript(jsFile);
    }

}
}
