/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QVariant>
#include <QFile>
#include <QContextMenuEvent>
#include <QSignalMapper>
#include <qwebframe.h>
#include "jscontextmenu.h"
#include "jsmenuitem.h"
#include "jsmenuitemfactory.h"
#include "action.h"
#include "actionadapter.h"

namespace WRT {

/*!
    \class WRT::JSContextMenu
    \brief The WRT::JSContextMenu

    \inmodule WidgetCore

    WRT::JSContextMenu evaluates contextmenu.js JavaScript file into window object.
*/

/*!
    Constructs an instance of JSContextMenu that evaluates contextmenu.js resource into current frame.
*/
JSContextMenu::JSContextMenu()
    : JSMenu("contextmenu.js", "")
{
}

JSContextMenu::~JSContextMenu()
{
}

/*!
    Shows context menu at give \a point if one context menu instance has been added to have element id under give \a point.

    This function should be called when a context menu invocation
    event occurs in the web page. The coordinates of the event
    (long tap, mouse click, ...) are given in parameter \a point. This
    function sends a contextmenu event to an HTML object defined
    by the coordinates and calls oncontextmenu event handler. If
    oncontextmenu returns true, then checks whether the element
    has a context menu defined for it. If the event handler returns
    true, this function returns container for menu structure.
*/
void JSContextMenu::showContextMenu(const QPoint& point) {
    if (m_actionAdapter) {
        QString htmlContextId;
        QList<Action*>* contextMenu = onContextMenuEvent(point, htmlContextId);
        if (contextMenu) {
            Action* action;
            m_currentHTMLElementId = htmlContextId;
            // This is wrong way to do this. We should or items, if there happens to be already some location
            foreach ( action, *contextMenu ) {
                action->setLocation(Action::ContextMenuLocation);
            }

            QList<Action*> actions = m_actionAdapter->actions(Action::ContextMenuLocation);
            m_actionAdapter->removeActions(actions);
            m_actionAdapter->addActions(*contextMenu);
            delete contextMenu;
        }
    }
}

/*!
    Helper method used from showContextMenu. This method populates context menu to the give view widget using the given
    \a point of the context menu event location. Id of the HTML element that caught the event is returned in \a menuId.

    Returns menu structure as a QList<WRT::Action*>*. Caller becomes owner of the returned action list.
*/
QList<Action*>* JSContextMenu::onContextMenuEvent(const QPoint& point, QString& menuId) {
    QContextMenuEvent *e = new QContextMenuEvent(
        QContextMenuEvent::Mouse, point);
    bool oncontextmenuRetval = !JSResourceWrapper::webPage()->swallowContextMenuEvent(e);
    if (oncontextmenuRetval) {

        QString fncCall = QString("nokia.device.menuManager.getMenu(%1, %2);").arg(point.x()).arg(point.y());
        QVariantList menuItems = JSResourceWrapper::webPage()->currentFrame()->
            evaluateJavaScript(fncCall).toList();

        if (!menuItems.isEmpty()) {
            menuId = menuItems.takeLast().toString();
        }

        if (menuItems.size() > 0) {
            QListIterator<QVariant> iterator(menuItems);
            QList<Action*>* contextMenuAction = new QList<Action*>();
            while (iterator.hasNext()) {
                QVariant nativeId = iterator.next();
                JSMenuItem* item = JSMenuItemFactory::item(nativeId.toInt());
                contextMenuAction->append(item->action());
            }
            return contextMenuAction;
        }

    }
    return 0;
}

}
