/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


window.nokia.device.ContextMenu = function(id) {
    var _elementIds = [];
    var _menuPrivate = new nokia.device.MenuBase();
    
    this.__defineGetter__("size", function() {
        return _menuPrivate.getMenu().length;
    });
    
    this.__defineGetter__("elementIds", function() {
        return _elementIds;
    });
    
    
    /** Adds new nokia.device.MenuItem
    *  @param {nokia.device.MenuItem}
    *             menuItem
    */
    this.addMenuItem = function(menuItem) {
        _menuPrivate.add(menuItem);
    };
    
    /** Removes a nokia.device.MenuItem
     *  @param {nokia.device.MenuItem}
     *             menuItem
     */
    this.removeMenuItem = function(menuItem) {
        _menuPrivate.remove(menuItem);
    };
    
    /**
     * Inserts a nokia.device.MenuItem before given index
     *  @param {nokia.device.MenuItem} menuItem
     *  @param {int} index
     */
    this.insertMenuItemBefore = function(menuItem, index) {
        _menuPrivate.insert(menuItem, index);
    };
    
    /**
     * Adds menu element id to context menu 
     * @param {string} id
     */
    this.addMenuElementId = function(id) {
        if (typeof id === 'string') {
            var _elem = document.getElementById(id);
            if (_elem) {
                _elem.wrtContextMenu = this;
                var index = _elementIds.indexOf(id);
                if (index == -1 ) {
                    _elementIds.push(id);
                }
            }
        }
    };
    
    /**
     * Removes menu element id from context menu 
     * @param {string} id
     */
    this.removeMenuElementId = function(id) {
        if (typeof id === 'string') {
            var len = _elementIds.length;
            var removed = false;
            for (var i = 0; i < len; i++) {
                if (_elementIds[i] === id) {
                    _elementIds.splice(i, 1);
                    removed = true;
                    break;
                }
            }
            
            if (removed) {
                var _elem = document.getElementById(id);
                _elem.wrtContextMenu = null;
            }
            
        }
    };
    
    this.clear = function() {
        _menuPrivate.clear();
    };
    
    this.item = function(index) {
        return _menuPrivate.item(index);
    };
    
    this.addMenuElementId(id);
};


/** nokia.device.menuManager is a singleton object containing utility functions that
 *  are called from C++. In the future menuManager functionality could be
 *  done in C++ side utilizing Qt DOM API.
 */
window.nokia.device.menuManager = function() {

/** Recursive function that searches a HTML element containing attribute
 *  wrtContextMenu. Search starts from htmlElements and continues recursively
 *  through its parentNode.
 *  @param {object}
 *            htmlElem HTML element where the search is started from.
 *  @return {object}
 *            Returns the first HTML element it finds that has wrtContextMenu 
 *            attribute.
 */
    var searchMenu = function(htmlElem) {
        if (!htmlElem) {
            return null;
        } else if (htmlElem.wrtContextMenu === undefined) {
            return searchMenu(htmlElem.parentNode);
        } else {
            return htmlElem;
        }
    };


    return {


/** This function searches the first HTML element under given coordinates
 *  containing wrtContextMenu attribute, calls onShow event handler for
 *  the wrtContextMenu if it exists and returns the menu structure and
 *  HTML element id. The search of the HTML element starts from the
 *  "topmost" HTML element and continues through its parentNodes.
 *  @param {number}
 *            xcoord X coordinate of the context menu event.
 *  @param {number}
 *            ycoord Y coordinate of the context menu event.
 *  @return {Array}
 *            Array containig the nokia.device.MenuItem instances of the ContextMenu. The last
 *            item in the array is not a nokia.device.MenuItem, but a string containig
 *            the id of the menu being returned. If onShow event handler
 *            returned false, the array contains only the menu id, but no
 *            MenuItems.
 */
    getMenu:
        function(xcoord, ycoord) {
            var topElem = document.elementFromPoint(xcoord, ycoord);
            var elem = searchMenu(topElem);
            if (elem === null) {
                return [];
            }
            
            var contextMenuItems = [];
            if (elem.wrtContextMenu !== undefined && elem.wrtContextMenu instanceof nokia.device.ContextMenu) {
                // create new array which is copy of this + elem id
                var index = 0;
                var menuItem = elem.wrtContextMenu.item(index);
                
                var nativeId = 0;
                while (menuItem !== undefined) {
                    nativeId = menuItemFactoryCPP.nativeId(menuItem);
                    contextMenuItems.push(nativeId);
                    ++index;
                    menuItem = elem.wrtContextMenu.item(index);
                }
                
                if (contextMenuItems.length > 0) {
                    contextMenuItems.push(elem.id);
                }
            }
            return contextMenuItems;
        }
    };
}();

