/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <QDebug>
#include <QNetworkRequest>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkProxy>
#include <QProcess>

// D-Bus
// todo: use the Qt API provided by the Browser team once it is ready instead of D-Bus
#include <QDBusConnection>
#include <QDBusInterface>
#include <QDBusError>
#include <QDBusReply>

#include "applicationlauncher.h"
#include "mimehandlerpluginfactory.h"

// Fennec related D-Bus definitions
// todo: use the Qt API provided by the Browser team once it is ready instead of D-Bus
const QString FENNEC_SERVICE("com.nokia.browser");
const QString FENNEC_OBJECT("/");
const QString FENNEC_METHOD("com.nokia.browser.OpenURL");

namespace WRT {

    namespace Maemo {

        ApplicationLauncher::ApplicationLauncher(QObject* parent) : QObject(parent), manager(0) {
            typeFactory = new MimeHandlerPluginFactory(this);
            typeFactory->loadPluginSpecs();

            // creates a persistent connection to D-Bus
            QDBusConnection connection = QDBusConnection::sessionBus();
            if (!connection.isConnected()) {
                qCritical() << Q_FUNC_INFO << "Cannot connect to D-Bus.";
            }
        }

        ApplicationLauncher::~ApplicationLauncher() {
            qDebug() << Q_FUNC_INFO;
            delete typeFactory;
        }

        bool ApplicationLauncher::launchApplication(QString app, QStringList params) {
            qDebug() << Q_FUNC_INFO;
            qDebug() << "Starting application" << app << "with parameters" << params;
            m_proc = new QProcess(0);
            connect(m_proc, SIGNAL(readyReadStandardError()), this, SLOT(readOutput()));
            m_proc->start(app, params);
            qDebug() << "Application started.";
            return true;
        }

        // todo: use the Qt API provided by the Browser team once it is ready instead of D-Bus
        bool ApplicationLauncher::launchApplication(QString mimeType, QUrl url) {
            qDebug() << Q_FUNC_INFO;
            qDebug() << "Mimetype:" << mimeType << "url:" << url.toString();
            // handles type text/html internally
            if (mimeType.startsWith("text/html", Qt::CaseInsensitive)) {
                //todo: D-Bus usage is not the right way nor the usage of QProcess directly
                QStringList params;
                params << url.toString();

                // starts the process and detaches from it
               qDebug() << "Launching fennec:" << params;
               return QProcess::startDetached("/usr/lib/fennec/fennec.sh", params);
            }

            QString app = typeFactory->getAppByMime(mimeType);
            if (!app.isEmpty()) {
                QStringList para;
                para << url.toString();
                return launchApplication(app, para);
            }
            qWarning() << "No suitable application found for" << mimeType;
            //qFatal("CRASHING THIS NOW!!!");
            return false;
        }

        void ApplicationLauncher::replyFinished(QNetworkReply* reply) {
            QVariant redirectReply = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
            if (!redirectReply.toUrl().isEmpty() && startUrl != redirectReply.toUrl()) {
                redirect(redirectReply.toUrl());
                return;
            }
            QString mimeType = reply->rawHeader("content-type");
            if (!mimeType.isEmpty())
                launchApplication(mimeType, startUrl.toString());
        }

        void ApplicationLauncher::redirect(const QUrl& url) {
            manager->get(QNetworkRequest(url));
            startUrl = url;
        }

        bool ApplicationLauncher::launchUrl(const QUrl& url) {
           bool bRet = false;
            if (!url.isValid()) {
                qDebug()<<Q_FUNC_INFO<<" parameter is invalid:"<<url;
            }

            // Launch file with correct application, we are not interested other schemes
            if (url.scheme() == "file") {
                qDebug()<< Q_FUNC_INFO << url;
                QString localFile = url.toLocalFile();
                QStringList parts = localFile.split(".");

                QString extension = parts.at(parts.size()-1);
                QString app = typeFactory->getAppByExtension(extension);
                if (!app.isEmpty()) {
                    QStringList list;
                    list<<localFile;
                    launchApplication(app,list);
                    bRet = true;
                } else {
                    qDebug()<<"No suitable application found for " << app << url.toString();
                }
            }
            else {
                qDebug()<< Q_FUNC_INFO << "is not interested of non-file-url " << url;
            }

            return bRet;
        }

        void ApplicationLauncher::setProxy() {
            QStringList environment = QProcess::systemEnvironment();
            QStringList filtered = environment.filter("http_proxy");

            if (filtered.size() == 0) {
                WrtSettings* storage = WrtSettings::createWrtSettings();
                if (storage->contains("framework/metadata/proxy/http")) {
                    filtered = QStringList(storage->value("framework/metadata/proxy/http").toString());
                }
            }

            if (filtered.size() > 0 ) {
                QString proxysetting = filtered.front();
                QStringList items = proxysetting.split(QRegExp("=|:|/"));
                int i = 0;
                QString host = "";
                QString port = "";

                while (i < items.size() ) {
                    QString item = items.at(i);
                    if ( item != NULL && item != "http" && item!="http_proxy" && item != "" && item.length()>0) {
                        host = item;
                        i++;
                        break;
                    }
                    i++;
                }
                while (i < items.size() ) {
                    QString item = items.at(i);
                    if ( item != "" && item != NULL ) {
                        port = item;
                        i++;
                        break;
                    }
                    i++;
                }

                bool ok;

                unsigned short val;
                val = port.toUShort(&ok );
                if (ok==false ) {
                    val = 8080;
                }
                QNetworkProxy proxy ( QNetworkProxy::HttpProxy,
                                      host, val, NULL, NULL);
                manager->setProxy(proxy);
            }
        }

        void ApplicationLauncher::readOutput() {
            qDebug() << "Application output: " << m_proc->readAllStandardError();
        }
    }
}

