/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "wrtnetworkreply.h"

namespace WRT{

WrtNetworkReply::WrtNetworkReply(QNetworkAccessManager::Operation op, const QNetworkRequest &request)
    :QNetworkReply()
{
    setOperation(op);
    setRequest(request);
    setUrl(request.url());
    m_reply = NULL;
}

WrtNetworkReply::~WrtNetworkReply()
{
}

void WrtNetworkReply::setReply(QNetworkReply* reply)
{
    m_reply=reply;
    reply->setParent(this);
    connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(handleMetaDataChanged()));
    connect(m_reply, SIGNAL(finished()), this, SLOT(handleFinished()));
    connect(m_reply, SIGNAL(readyRead()), this, SLOT(handleReadyRead()));
    connect(m_reply, SIGNAL(sslErrors(QList<QSslError>)), this, SIGNAL(sslErrors(QList<QSslError>)));
    QIODevice::open(QIODevice::ReadOnly);
}

void WrtNetworkReply::abort()
{
    QNetworkReply::close();
    if (m_reply)
        m_reply->abort();
    emit aborted(this);
}

qint64 WrtNetworkReply::readData(char *data, qint64 maxlen)
{
    if (m_readBuffer.isEmpty())
        return  -1;

    if (maxlen == 1) {
        // optimization for getChar()
        *data = m_readBuffer.data()[0];

        return 1;
    }
    maxlen = qMin<qint64>(maxlen, m_readBuffer.size());

    const char* bufdata = m_readBuffer.data();
    memcpy(data,bufdata,maxlen);
    m_readBuffer.remove(0,maxlen);

    return maxlen;
 }

qint64 WrtNetworkReply::bytesAvailable() const
{
    return QNetworkReply::bytesAvailable() + m_readBuffer.size();
}

void WrtNetworkReply::handleReadyRead()
{
    bridgeReplyData();

    QByteArray data = m_reply->read(m_reply->bytesAvailable());
    m_readBuffer.append(data);

    emit readyRead();
}

void WrtNetworkReply::handleMetaDataChanged()
{
    bridgeReplyData();

    emit metaDataChanged();
}

void WrtNetworkReply::handleFinished()
{
    bridgeReplyData();

    emit finished();
}

void WrtNetworkReply::bridgeReplyData()
{
    setError(m_reply->error(), m_reply->errorString());
    setHeader(QNetworkRequest::ContentTypeHeader, m_reply->header(QNetworkRequest::ContentTypeHeader));
    setHeader(QNetworkRequest::ContentLengthHeader, m_reply->header(QNetworkRequest::ContentLengthHeader));
    setAttribute(QNetworkRequest::HttpStatusCodeAttribute, m_reply->attribute(QNetworkRequest::HttpStatusCodeAttribute));
    setAttribute(QNetworkRequest::HttpReasonPhraseAttribute, m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute));
    setAttribute(QNetworkRequest::RedirectionTargetAttribute, m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute));

    foreach (const QByteArray& headerName, m_reply->rawHeaderList()) {
        setRawHeader(headerName, m_reply->rawHeader(headerName));
    }

    setUrl(m_reply->url());
}

void WrtNetworkReply::ignoreSslErrors()
{
    m_reply->ignoreSslErrors();
}

}
