/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "signatureparser.h"

#include <QDomDocument>
#include <QDomNodeList>
#include <QDomNode>
#include <QFile>
#include <QDebug>
#include <QProcess>
#include <QString>
#include "SignatureValidator.h"
#if defined(Q_OS_SYMBIAN)
#include "CertificateManagerS60.h"
#elif defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
#include "CertificateManagerMaemo.h"
#else
#include "CertificateManagerQt.h"
#endif
#include <QCryptographicHash>
#include <QByteArray>
#include <QDir>

#include <openssl/bio.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/ossl_typ.h>
#include <openssl/objects.h>
#include <openssl/sha.h>
#ifndef LIBXML_C14N_ENABLED
#ifndef LIBXML_OUTPUT_ENABLED
#define LIBXML_C14N_ENABLED
#define LIBXML_OUTPUT_ENABLED
#endif /* LIBXML_OUTPUT_ENABLED */
#endif /* LIBXML_C14N_ENABLED */

#ifdef __SYMBIAN32__
#include <libxml2_parser.h>
#include <libxml2_c14n.h>
#include <libxml2_tree.h>
#else
#include <libxml/c14n.h>
#include <libxml/tree.h>
#endif

// libxml 2.6.32 (ubuntu 9.04 stable) does not define xmlC14NMode in c14n.h
namespace WRT {
 typedef enum {
    XML_C14N_1_0            = 0,    /* Origianal C14N 1.0 spec */
    XML_C14N_EXCLUSIVE_1_0  = 1,    /* Exclusive C14N 1.0 spec */
    XML_C14N_1_1            = 2     /* C14N 1.1 spec */
 } xmlC14NMode;
}

namespace W3C {
  const char* dsNamespace = " xmlns=\"http://www.w3.org/2000/09/xmldsig#\"";
}

SignatureParser::SignatureParser(QString workdir, QString widgetpath)
{
    m_workdir = workdir;
    m_widgetpath = widgetpath;
    m_parsed = false;
    m_aki = "";
}



bool SignatureParser::parse(){
    m_parsed = true;


    QFile f( m_widgetpath );
    if ( !f.open(QIODevice::ReadOnly | QIODevice::Text) ) {
        qDebug() << "Failed to open";
        return false;
    }
    QByteArray dat = f.readAll();
    bool parsed = m_doc.setContent( dat, true );
    if (!parsed) {
        qDebug() << "could not parse signature file: " << m_widgetpath;
        return false;
    }

    f.close();
    qDebug() << "Data read";
    return true;
}


bool SignatureParser::requiredElementsExist() {
    if (!m_parsed)
        parse();

    // namespace processing
    QDomNode signature = m_doc.documentElement();
    if (signature.nodeName() != "Signature") return false;
    if (! dsNamespace(signature) ) return false;

    QDomNode signedInfo = m_doc.documentElement().namedItem("SignedInfo");
    if (! dsNamespace(signedInfo) ) return false;

    QDomNode signatureProperties = m_doc.documentElement().elementsByTagName("SignatureProperties").at(0);
    if (! dsNamespace(signatureProperties) ) return false;

    QDomNode profile = m_doc.documentElement().elementsByTagName("Profile").at(0);
    if (! dspNamespace(profile) ) return false;

    QDomNode role = m_doc.documentElement().elementsByTagName("Role").at(0);
    if (! dspNamespace(role) ) return false;

    QDomNode identifier = m_doc.documentElement().elementsByTagName("Identifier").at(0);
    if (! dspNamespace(identifier) ) return false;

    // validate required elements exist
    if ( m_doc.documentElement().nodeName()=="Signature" &&
         !m_doc.documentElement().namedItem("SignedInfo").isNull() &&
         !m_doc.documentElement().namedItem("SignedInfo").namedItem("CanonicalizationMethod").isNull() &&
         !m_doc.documentElement().namedItem("SignedInfo").namedItem("SignatureMethod").isNull() &&
         !m_doc.documentElement().namedItem("SignedInfo").toElement().elementsByTagName("Reference").length() == 0 &&
         !m_doc.documentElement().namedItem("SignatureValue").isNull() &&
         !m_doc.documentElement().namedItem("Object").isNull() ) {

        // validate required elements and their required values
        if (m_doc.documentElement().elementsByTagName("Profile").length() == 1 && m_doc.documentElement().elementsByTagName("Profile").item(0).attributes().namedItem("URI").nodeValue() == "http://www.w3.org/ns/widgets-digsig#profile" &&
            ( m_doc.documentElement().elementsByTagName("Role").length() == 1 && (
                    m_doc.documentElement().elementsByTagName("Role").item(0).attributes().namedItem("URI").nodeValue() == "http://www.w3.org/ns/widgets-digsig#role-author" ||
                    m_doc.documentElement().elementsByTagName("Role").item(0).attributes().namedItem("URI").nodeValue() == "http://www.w3.org/ns/widgets-digsig#role-distributor" )
              ) &&
            m_doc.documentElement().elementsByTagName("Identifier").length() == 1
            )
        {
            // validate that there is a <Reference> that points to the <Object> containing Profile, Role, and Identifier
            QDomNode profile = m_doc.documentElement().elementsByTagName("Profile").item(0);
            QDomNode role = m_doc.documentElement().elementsByTagName("Role").item(0);
            QDomNode identifier = m_doc.documentElement().elementsByTagName("Identifier").item(0);

            QDomNode p_object = profile.parentNode().parentNode().parentNode();
            if (p_object.attributes().namedItem("Id").isNull()) {
                qDebug() << "Object element does not contain an Id";
                return false;
            }
            QString p_id = p_object.attributes().namedItem("Id").nodeValue();
            p_id.prepend("#");
            QDomNodeList refs = m_doc.elementsByTagName("Reference");
            bool foundRef = false;
            // loop to find reference for Object element and to also ensure that no References have Transform elements
            for (unsigned int i=0; i < refs.length(); i++) {
                QDomNode ref = refs.item(i);
                if (!foundRef) {
                    if (ref.attributes().namedItem("URI").nodeValue() == p_id)
                        foundRef = true;
                }
                if (!ref.firstChildElement("Transforms").isNull() || !ref.firstChildElement("Transform").isNull() ) {
                    qDebug() << "ds:Reference elements MUST NOT contain Transform elements";
                    return false;
                }
            }

            if (foundRef &&
                p_object == role.parentNode().parentNode().parentNode() &&
                p_object == identifier.parentNode().parentNode().parentNode()) {

                if (m_widgetpath.indexOf("author-signature") >= 0 && m_doc.documentElement().elementsByTagName("Role").item(0).attributes().namedItem("URI").nodeValue() == "http://www.w3.org/ns/widgets-digsig#role-author" ||
                    m_widgetpath.indexOf("author-signature") < 0 && m_doc.documentElement().elementsByTagName("Role").item(0).attributes().namedItem("URI").nodeValue() == "http://www.w3.org/ns/widgets-digsig#role-distributor")
                {
                    return true;
                }
                qDebug() << "value for <dsp:Role> URI does not match signature filename";
            }
        }
        qDebug() << "signature.xml not valid. Required elements contain illegal values.";
    } else {
        qDebug() << "signature.xml not valid. Some required elements are missing.";
    }
    return false;
}

/**
* This method checks to see if all the files included in the widget package are referenced in the signature files.
* It does NOT check to see if there is a <Reference> element that points to the <Object> element in the signature file.
* Signature files are not required to include a <Reference> of themselves or any other distributor signature.
* Distributor signatures are required to include the author-signature.xml file if it exists in a <Reference> element.
*
* @param files  A list of all in the widget package
*
* @return bool  True if successful, false otherwise
*/
bool SignatureParser::checkReferencesExist(QStringList files){
    if (!m_parsed)
        parse();

    QDomNodeList nodes = m_doc.documentElement().elementsByTagName("Reference");

    // if the signature file is an author signature, it is required to have a Reference element that points to all files except ALL signature files
    // if the signature file is a distributor signature, it is required to have a Reference element that points to all files except distributor signatures
    // there is also a requirement that an Object element exists that is pointed to by a Reference element
    for (int i = 0; i< nodes.count(); i++ ) {
        QDomNode node = nodes.item(i);
        QDomNamedNodeMap refAttributes = node.attributes();
        QString uri = refAttributes.namedItem("URI").nodeValue();

        if ( files.contains(uri, Qt::CaseSensitive ) ) {
            files.removeAt(files.indexOf(uri));
        }
    }

    QFileInfo sigFile(m_widgetpath);
    if (files.count() == 0 ) {
        // simple case: References point to every file
        return true;
    } else {
        QRegExp distSig("signature([1-9][0-9]?).xml");
        foreach(QString filename, files) {
            // a signature will never include itself as a reference (the case where author-signature.xml missing a reference to itself)
            // we can also safely skip all files that match signature[1-9][0-9].xml
            // because these should not be included in either an author or distributor signature
            if (distSig.exactMatch(filename.toLower()) || filename.toLower() == sigFile.fileName() )
            {
                continue;
            } else {
                qDebug() << "Hash for some files not included in the signature.xml (" << filename << ")";
                return false;
            }
        }
        return true;
    }
}


QString SignatureParser::getSubject()
{
    if (!m_parsed) {
        parse();
    }
    QDomNode cert = m_doc.documentElement().namedItem("KeyInfo").namedItem("X509Data").namedItem("X509Certificate");
    if ( cert.isNull() || cert.firstChild().isNull() ) {
        qDebug() << "No KeyInfo, X509Data or X509Certificate found";
    return QString("");
    }
    SignatureValidator * validator = new SignatureValidator();
    QByteArray certificate = cert.firstChild().nodeValue().toUtf8();
    certificate.insert(0,"-----BEGIN CERTIFICATE-----\n");
    certificate.insert(certificate.size(),"\n-----END CERTIFICATE-----");
    QString returnValue(validator->getSubject(certificate));
    delete validator;
    return returnValue;
}

QString SignatureParser::getAKI() {
    return m_aki;
}

/**
 *  Looks at the x509 cert included in the widget's signature.xml file
 *  and compares that certificate to a local version retrieved by matching
 *  the widget's cert's AKI to a local cert's SKI.
 *
 *  TODO: pick a default cert if one is not present in the widget's signature.xml
 *
 * @return bool         true if successful, false otherwise
 */
bool SignatureParser::validateCertificates(QByteArray& endEntityCert) {
    if (!m_parsed)
        parse();

    QList<QByteArray> certificates;
    QDomNodeList certs = m_doc.documentElement().elementsByTagName("X509Certificate");
    for (uint i = 0; i < certs.length(); i++) {
        QDomNode cert = certs.at(i);
        if ( cert.isNull() || cert.firstChild().isNull() ) {
            // TODO: check ALL certs on device if cert not given
            qDebug() << "No KeyInfo, X509Data or X509Certificate found";
            return false;
        }
        QByteArray certificate = cert.firstChild().nodeValue().toUtf8();

        // if certificate is in pseudo-DER format, convert to PEM for OpenSSL methods
        if (certificate.indexOf("\n") == -1) {
            int newlinePos = 0;
            int certSize = certificate.size();
            int newlinesInserted = 0;
            while (newlinePos < certSize) {
                if (newlinePos != 0) {
                    certificate.insert((newlinePos + newlinesInserted), "\n");
                    newlinesInserted++;
                }
                newlinePos += 64;
            }
        }
        certificate.insert(0,"-----BEGIN CERTIFICATE-----\n");
        certificate.insert(certificate.size(),"\n-----END CERTIFICATE-----");
        certificates.append(certificate);
    }

    WRT::CertificateManager* certman;
#if defined(Q_OS_SYMBIAN)
    certman = new WRT::CertificateManagerS60();
#elif defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
    certman = new WRT::CertificateManagerMaemo();
#else
    certman = new WRT::CertificateManagerQt();
#endif

    QByteArray topLevelCert;
    if (! certman->buildCertMapping(certificates, endEntityCert, topLevelCert) ) {
        qDebug() << "Error building certificate mapping on AKI/SKI.";
        return false;
    }
    bool retval = certman->validateCertificates(certificates, endEntityCert, topLevelCert);
    m_aki = certman->getAuthorityKeyIdentifier(topLevelCert);
    delete certman;
    return retval;
}

bool SignatureParser::validateSignature(QString signature_file, QByteArray certificate){
    if (!m_parsed)
        parse();

    QDomNode sig = m_doc.documentElement().namedItem("SignatureValue");
    if ( sig.isNull() || sig.firstChild().isNull() ) {
        qDebug() << "No SignatureValue defined";
        return false;
    }

    QDomNode sigMethod = m_doc.documentElement().namedItem("SignedInfo").namedItem("SignatureMethod");
    if (sigMethod.isNull() || sigMethod.attributes().namedItem("Algorithm").isNull())
    {
        qDebug() << "No SignatureMethod defined";
        return false;
    }

    QFile sigfile(signature_file);
    sigfile.open(QIODevice::ReadOnly);
    QByteArray siginfo = sigfile.readAll();
    if (siginfo.indexOf("<SignedInfo>") < 0 || siginfo.indexOf("</SignedInfo>") < 0) {
        qDebug() << "No SignedInfo found";
        return false;
    }

    siginfo = siginfo.remove(0,siginfo.indexOf("<SignedInfo>"));
    QString tmp("</SignedInfo>");
#if defined(Q_OS_SYMBIAN)
    siginfo = siginfo.remove(siginfo.indexOf(tmp.toAscii(),0)+tmp.length(),siginfo.size());
#else
    siginfo = siginfo.remove(siginfo.indexOf(tmp)+tmp.length(),siginfo.size());
#endif

    // TODO: check to see if SignedInfo already has xmlns
    siginfo.insert((int)QString("<SignedInfo").length(), W3C::dsNamespace);

    SignatureValidator * validator = new SignatureValidator();
    QByteArray signature = sig.firstChild().nodeValue().toUtf8();

    QString algorithm = sigMethod.attributes().namedItem("Algorithm").nodeValue();

    // Canonicalization of signature
    QDomNode c14n = m_doc.documentElement().namedItem("SignedInfo").namedItem("CanonicalizationMethod");
    QString c14nMethod = c14n.attributes().namedItem("Algorithm").nodeValue();
    WRT::xmlC14NMode c14nMode;
    int c14nComments; // 1 for with comments, 0 for omits comments

    if (c14nMethod == "http://www.w3.org/TR/2001/REC-xml-c14n-20010315") {
        c14nMode = WRT::XML_C14N_1_0;
        c14nComments = 0;
    } else if (c14nMethod == "http://www.w3.org/TR/2001/REC-xml-c14n-20010315#WithComments") {
        c14nMode = WRT::XML_C14N_1_0;
        c14nComments = 1;
    } else if (c14nMethod == "http://www.w3.org/2006/12/xml-c14n11") {
        c14nMode = WRT::XML_C14N_1_1;
        c14nComments = 0;
    } else if (c14nMethod == "http://www.w3.org/2006/12/xml-c14n11#WithComments") {
        c14nMode = WRT::XML_C14N_1_1;
        c14nComments = 1;
    } else if (c14nMethod == "http://www.w3.org/2001/10/xml-exc-c14n#") {
        c14nMode = WRT::XML_C14N_EXCLUSIVE_1_0;
        c14nComments = 0;
    } else if (c14nMethod == "http://www.w3.org/2001/10/xml-exc-c14n#WithComments") {
        c14nMode = WRT::XML_C14N_EXCLUSIVE_1_0;
        c14nComments = 1;
    } else {
        qDebug() << "CanonicalizationMethod unknown or not supported: " << c14nMethod;
        return false;
    }

    int i;
    if (algorithm == "http://www.w3.org/2000/09/xmldsig#rsa-sha1")
    {
        i = validator->rsaVerify(siginfo,signature,certificate, validator->sha_1, c14nMode, c14nComments);
    } else if (algorithm == "http://www.w3.org/2001/04/xmldsig-more#rsa-sha256") {
        i = validator->rsaVerify(siginfo, signature, certificate, validator->sha_256, c14nMode, c14nComments);
    } else if (algorithm == "http://www.w3.org/2000/09/xmldsig#dsa-sha1") {
        i = validator->dsaVerify(siginfo, signature, certificate, validator->sha_1, c14nMode, c14nComments);
    } else {
        qDebug() << "Algorithm in SignatureMethod not supported";
        i = SignatureValidator::Verification_Failure;
    }

    bool returnValue(i == SignatureValidator::Verification_Success);
    delete validator;
    return returnValue;
}


bool SignatureParser::validateReferences(QStringList files){
    if (!m_parsed)
        parse();

    QDomNodeList nodes = m_doc.documentElement().elementsByTagName("Reference");

    for (int i = 0; i< nodes.count(); i++ ) {
        QDomNode node = nodes.item(i);
        QDomNamedNodeMap refAttributes = node.attributes();
        QString uri = refAttributes.namedItem("URI").nodeValue();

        QString algorithm = node.namedItem("DigestMethod").attributes().namedItem("Algorithm").nodeValue();
        QByteArray hashValue;
        if ( files.contains(uri, Qt::CaseSensitive ) ) {
            files.removeAt(files.indexOf(uri));
            QString fpath(m_workdir);
            if (fpath.endsWith(QDir::separator())){
                fpath.append(uri);
            } else {
                fpath.append(QDir::separator()+uri);
            }
            QFile f(fpath);

            qDebug() << "Verifying: "+ fpath << " Size= "<< f.size();
            f.open(QIODevice::ReadOnly);
            QByteArray dataArray(f.readAll());

            int rc = this->createDigest(hashValue, algorithm, dataArray);
            if (rc != 1){ return false; }

        } else {
            // This Reference URI is not a file in the widget's directory, but is an IDREF pointing to an XML tag in this signature file
            // we only care to check if the ID matches an <Object> tag for now, to validate that nothing has changed in that element
            QDomNodeList objects = m_doc.documentElement().elementsByTagName("Object");
            bool foundObject = false;
            for (unsigned int i = 0; i < objects.length(); i++)
            {
                QString obj_id = objects.item(i).attributes().namedItem("Id").nodeValue();
                obj_id.prepend("#");
                if ( obj_id == uri)
                {
                    foundObject = true;
                    QFile sigfile(m_widgetpath);
                    sigfile.open(QIODevice::ReadOnly);
                    QByteArray objData = sigfile.readAll();

                    if (objData.indexOf("<Object") < 0 || objData.indexOf("</Object>") < 0 ) {
                        qDebug() << "Error finding object element.";
                        return false;
                    }

                    // this search won't work if <Object> is the root element and first characters of signature.xml
                    // but it should never be anyways, so we don't worry about it.
                    int objectIndex = 0;
                    for (unsigned int objCount = 0; objCount <= i; objCount++) {
                        objectIndex = objData.indexOf("<Object", (objectIndex+1));
                    }

                    objData = objData.remove(0, objectIndex);
                    objectIndex = objData.indexOf("</Object>");
                    objData.remove(objectIndex, objData.size() - objectIndex);
                    objData.append("</Object>");

                    // TODO: check to see if object already has namespace
                    objData.insert((int)QString("<Object").length(), W3C::dsNamespace);

                    // Canonicalization of <Object> element
                    WRT::xmlC14NMode c14nMode = WRT::XML_C14N_1_0;
                    int c14nComments = 0; // 1 for with comments, 0 for omits comments

                    QString canObjData("");
                    SignatureValidator* sigval = new SignatureValidator();
                    if (! sigval->canonicalize(QString(objData),canObjData,c14nMode,c14nComments)) {
                        delete sigval;
                        return false;
                    }
                    delete sigval;

                    int rc = this->createDigest(hashValue, algorithm, canObjData.toUtf8());
                    if (rc != 1) { return false; }
                    break;
                }
            }

            if (!foundObject) {
                // There is a reference that doesn't point to a file or an object
                qDebug() << "Reference has an invalid URI: " << uri;
                return false;
            }
        }

        if ( hashValue.size() > 0 && (node.namedItem("DigestValue").firstChild().nodeValue().toAscii() != hashValue.toBase64()) )
        {
            qDebug() << "Hash check failed for file "+uri;
            qDebug() << "Actual hash: " << hashValue.toBase64();
            qDebug() << "Expected hash: " << node.namedItem("DigestValue").firstChild().nodeValue().toAscii();
            return false;
        }

    }

    return true;
}

int SignatureParser::createDigest(QByteArray& digest, QString uri, QByteArray data) {
    if ( uri == "http://www.w3.org/2000/09/xmldsig#sha1") {
        digest = QCryptographicHash::hash(data, QCryptographicHash::Sha1);
        return 1;
#ifdef SHA256_SUPPORT
    } else if (uri == "http://www.w3.org/2001/04/xmlenc#sha256") {
        unsigned char hashed[SHA256_DIGEST_LENGTH];
        SHA256((const unsigned char*)data.data(), data.size(), hashed);
        QByteArray tmp = QByteArray::fromRawData((const char*)hashed, SHA256_DIGEST_LENGTH);
        for (int i = 0; i < SHA256_DIGEST_LENGTH; i++) {
            digest[i] = tmp[i];
        }
        return 1;
#endif
    } else {
        qDebug() << "Hash algorithm not specified or implemented:\n" << uri;
        return 0;
    }
}

bool SignatureParser::dsNamespace(QDomNode node) {
    // we remove any '#' characters from this namespace to identify "xmldsig" and "xmldsig#" as the same
    QString nsURI = QString(node.namespaceURI()).replace(QRegExp("#"), "");
    if (nsURI == "http://www.w3.org/2000/09/xmldsig")
        return true;
    qDebug() << "Node " << node.localName() << " is not in the 'ds' namespace";
    return false;
}

bool SignatureParser::dspNamespace(QDomNode node) {
    // we remove any '#' characters from this namespace to identify "xmldsig-properties" and "xmldsig-properties#" as the same
    QString nsURI = QString(node.namespaceURI()).replace(QRegExp("#"), "");
    if (nsURI == "http://www.w3.org/2009/xmldsig-properties")
        return true;
    qDebug() << "Node " << node.localName() << " is not in the 'dsp' namespace";
    return false;
}
