/*
* ============================================================================
*  Name        : qsysinfoservice.h
*  Part of     : serviceproviders / WRT
*  Description : Qt class for sysinfo service
*  Version     : %version: 1 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QSYSINFOSERVICE_H_
#define QSYSINFOSERVICE_H_

//INCLUDES

#include <QtDBus/QtDBus>
#include <QObject>
#include <QString>
#include <QStringList>
#include <QVariant>
#include <QMap>
#include <QDesktopWidget>

#include "servicesecurityinterface.h"
#include "qsysinfoserviceinterface.h"
#include "secsession.h"

typedef QMap<QString,qint32> AsyncMap;
typedef QMap<QString,qint32> StartMap;
typedef QMap<QString,qint32> TriggerMap;
typedef QMap<QString,QVariant> ErrorMap;
typedef QMap<QString, QVariant> OutputDataMap;


// CLASS DECLARATION

/**
*  The class implements the interfaces for SysInfo service
*  described in ISysInfo interface class
*  @lib sysinfoserviceplugin.qtplugin
*  @since x.y
*/
class SysInfoService : public QObject,
                       public ISysInfo,
                       public IServiceSecurity
{
    Q_OBJECT
    Q_INTERFACES(ISysInfo)

public:

    /**
     * Default constructor
     * @since x.y
     */
    SysInfoService();

    /**
     * Destructor
     * @since x.y
     */
    virtual ~SysInfoService();

    /**
     * Provides implementation for setSecuritySession derived from
     * IServiceSecurity.
     * Needed to handle capabilities.
     * @since x.y
     * @param aSecSession - Security Manager session
     * @return none
     */
    void setSecuritySession( WRT::SecSession *aSecSession );

signals:
    void asyncCallback( int, int, const QMap<QString,QVariant>& );
    void asyncCallbackStChannel( int, int, const QMap<QString,QVariant>& );


public slots:

    //API
    /**
     * This method is used to get the list of channels supported by the device.
     * This is a Synchronous API
     * @since  x.y
     * @return list of supported channels(QStringList)
     */
    QVariant getChannelList();

    /**
     * Set the properties of the specified channel.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @param aInputData - Data for set to the channel
     * @return Map Containing ErrorCode and ErrorMessage(QMap) or NULL(no errors)
     */
    QVariant setChannelSync( const QString& aChannel, const QMap<QString,QVariant>& aInputData);

    /**
     * Get the properties of the specified channel.
     * This is an synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @param aInputData - specific data channel
     * @return Map with data of channel(QMap)
     */
    QVariant getChannelSync( const QString& aChannel , const QMap<QString,QVariant>& aInputData );

    /**
     * Get the properties of the specified channel.
     * This is an synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map with data channel
     */
    QVariant getChannelSync( const QString& aChannel );

    /**
     * Get the properties of the specified channel.
     * This is an asynchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map with data of channel(QMap)
     */
    QVariant getChannelAsync( const QString& aChannel );

    /**
     * Stops the notifications (started by getChannelAsync) for the channel specified.
     * This is a synchronous method.
     * @since x.y
     * @param aTransId - transaction id.
     * @return Transaction id(qint32) or Map Containing ErrorCode and ErrorMessage(QMap).
     */
    QVariant cancel( const int aTransId );

    /**
     * Registers for notifications from a particular channel.
     * This is an asynchronous method.
     * If the notification for a particular channel has already been registered,
     * and subsequently a call made again, api will return an error.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @param aTrigger - Data of trigger.
     * @return transId - Transaction id(qint32).
     */
    QVariant startChannel( const QString& aChannel, int aTrigger );

    /**
     * Registers for notifications from a particular channel.
     * This is an asynchronous method.
     * If the notification for a particular channel has already been registered,
     * and subsequently a call made again, api will return an error.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @param aInputData - Specific data of channels.
     * @return NULL(NOT IMPLEMENTED)
     */
    QVariant startChannel(const QString&, const QMap<QString, QVariant>&);

    /**
     * Stops the notifications (started by startChannel) for the channel specified.
     * This is a synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map Containing ErrorCode and ErrorMessage(QMap) or NULL(no errors).
     */
    QVariant stopChannel( const QString& aChannel );

private slots:

    //CB functions
    //---------------------------------------
    // Callback function for resizing of display size.
    //---------------------------------------
    void displayResizedCb();

    //---------------------------------------
    // Callback function for BatteryLevel channel.
    //---------------------------------------
    void asyncBatteryLevelChannelCb(const QDBusMessage& message);

    //---------------------------------------
    // Callback function for Network channel.
    //---------------------------------------
    void asyncNetworkChannelCb(const QDBusMessage& message);

    //---------------------------------------
    // Callback function for SignalStrength channel.
    //---------------------------------------
    void asyncSignalStrengthChannelCb(const QDBusMessage& message);

    //---------------------------------------
    // Callback function for HomeNetworkChannel channel.
    //---------------------------------------
    void asyncHomeNetworkChannelCb(const QDBusMessage& message);

    //---------------------------------------
    // Callback function for ChargingChannel channel.
    //---------------------------------------
    void startChargingChannelCb(const QDBusMessage & message);

    //---------------------------------------
    // Callback function for BatteryLevel channel.
    //---------------------------------------
    void startBatteryLevelChannelCb(const QDBusMessage & message);

    //---------------------------------------
    // Callback function for BluetoothOn channel.
    //---------------------------------------
    void startBluetoothOnChannelCb( const QDBusMessage & message );

    //---------------------------------------
    // Callback function for Network channel.
    //---------------------------------------
    void startNetworkChannelCb(const QDBusMessage& message);

    //---------------------------------------
    // Callback function for SignalStrength channel.
    //---------------------------------------
    void startSignalStrengthChannelCb( const QDBusMessage & message );

    //---------------------------------------
    // Callback function for FlipOpen channel.
    //---------------------------------------
    void startFlipOpenChannelCb( const QDBusMessage & message );

private:
    // Security session
    WRT::SecSession *_sec_session;

    //---------------------------------------
    // Function contains all possible error checking (called at the beginning of each method)
    // param aChannel - Name of the channel.
    // param Data - contains a date-specific for each mode.
    // param Mode - mode (which method calls this function).
    // return Map Containing ErrorCode and ErrorMessage.
    //---------------------------------------
    ErrorMap CheckingOfErrors( const QString& aChannel , const QVariant& Data, const QString& Mode );

    // Counter for generating transaction id for async requested
    qint32 m_transIdCount;

    // Trigger data
    qint32 m_Trigger;

    // Maps
    AsyncMap m_asyncMap;
    StartMap m_startMap;
    TriggerMap m_triggerMap;

    // List
    QStringList m_ChannelList;

    // Size of screen
    QDesktopWidget *desktop;
    qint32 height;
    qint32 width;
};
#endif //QSYSINFOSERVICE_H_
