/*
* ============================================================================
*  Name        : qsensorservice.h
*  Part of     : serviceproviders / WRT
*  Description : Qt class for sensor service
*  Version     : %version: 6 % << Don't touch! Updated by Synergy at check-out.
*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef QSENSORSERVICE_H_
#define QSENSORSERVICE_H_

//Project includes
#include "servicesecurityinterface.h"
#include "qsensorserviceinterface.h"
#include "secsession.h"

#include <QObject>
#include <QString>
#include <QStringList>
#include <QMap>
#include <QVariant>
#include <QTimer>
#include <QVector3D>


typedef QMap<QString,qint32>    StartMap;
typedef QMap<QString, qint32>   Frequencies;
typedef QMap<QString,QVariant>  ErrorMap;
typedef QMap<QString,QVariant>  OutputDataMap;
typedef QMap<QString,QTimer*>   TimerMap;
typedef QMap<QString,QVariant>  StateChannelsMap;
typedef QList<QVector3D>        HistoryDtList;
typedef QMap<qint32,QVariant>   SingleStartMap;

/**
 * Sensor provider class. Provides methods to get the sensor channel available
 * as well as subscribe to get continuous notification of sensor data.
 * @since x.y
 */
class QSensorService : public QObject,          //Need for QT
                       public ISensor,          //The Sensor Interface Definition
                       public IServiceSecurity  //Security stuff
{
    Q_OBJECT
    Q_INTERFACES(ISensor)

public:

    /**
     * Default constructor
     * @since x.y
     */
    QSensorService();

    /**
     * Destructor
     * @since x.y
     */
    virtual ~QSensorService();

    /**
     * Provides implementation for setSecuritySession derived from
     * IServiceSecurity.
     * Needed to handle capabilities.
     * @since x.y
     * @param aSecSession - Security Manager session
     * @return none
     */
    void setSecuritySession(WRT::SecSession *aSecSession);

public slots:

    //API
    /**
     * This method is used to get the list of channels supported by the device.
     * This is a Synchronous API
     * @since  x.y
     * @return list of supported channels(QStringList)
     */
    QVariant getChannels ();

    /**
     * Get the properties of the specified channel.
     * This is an synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map with data of channel(QMap)
     */
    QVariant getChannelEvent(const QString& aChannel);

    /**
     * Get the properties of the specified channel.
     * This is an synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map with ErrorCode and data of channel(QMap)
     */
    QVariant getChannelData( const QString& aChannel );

    /**
     * Registers for notifications from a particular channel.
     * This is an asynchronous method.
     * If the notification for a particular channel has already been registered
     * api will return an error.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @param transId -  Transaction id.
     * @return Map Containing ErrorCode and ErrorMessage(QMap)
     */
    QVariant startChannel( const QString& aChannel,const int transId );

    /**
     * Stops the notifications (started by startChannel) for the channel specified.
     * This is a synchronous method.
     * @since x.y
     * @param aChannel - Name of the channel as specified in the channel descriptor.
     * @return Map Containing ErrorCode and ErrorMessage(QMap).
     */
    QVariant stopChannel( const QString& aChannel );

    /**
     * Set desired frequency (only for AccelerometerAxis and Rotation channels).
     * This is a synchronous method.
     * @since x.y
     *
     * @param aChannel - Name of the channel.
     * @param msec - the count of milliseconds for frequency.
     * @return the count of milliseconds set(qint32).
     */
    QVariant setDesiredUpdateFrequency( const QString& aChannel, const int msec );

    /**
     * Get scale factor (only for AccelerometerAxis channel).
     * This is a synchronous method.
     * @since x.y
     *
     * @param aChannel - Name of the channel.
     * @return scale factor(qreal)
     */
    QVariant getScaleFactor( const QString& aChannel );

    //---------------------------------------
    // Callback function for AccelerometerAxis channel.
    //---------------------------------------
    void startAccelerometerChannelCb();

    //---------------------------------------
    // Callback function for Orientation channel.
    //---------------------------------------
    void startOrientationChannelCb();

    //---------------------------------------
    // Callback function for Rotation channel.
    //---------------------------------------
    void startRotationChannelCb();

    //---------------------------------------
    // Callback function for AccelerometerDoubleTapping channel.
    //---------------------------------------
    void startAccelerometerDoubleTappingChannelCb();

    //---------------------------------------
    // Callback function for Proximity channel.
    //---------------------------------------
    void startProximityChannelCb();

    //---------------------------------------
    // Callback function for Illumination channel.
    //---------------------------------------
    void startIlluminationChannelCb();

    //---------------------------------------
    // Callback function for Single Start.
    //---------------------------------------
    void singleStartCb();

signals:
    void asyncCallback(int transactionId, int status, QVariant sensorData);

protected:
    // Security session
    WRT::SecSession *m_secSession;

private:

    //---------------------------------------
    // Function contains all possible error checking (called at the beginning of each method)
    // param aChannel - Name of the channel.
    // param Data - contains a date-specific for each mode.
    // param Mode - mode (which method calls this function).
    // return Map Containing ErrorCode and ErrorMessage.
    //---------------------------------------
    ErrorMap CheckingOfErrors( const QString& aChannel , const QVariant& Data, const QString& Mode );

    //---------------------------------------
    // returns the correct orientation of analyzing the data from the mce service
    // param orientation - portrait or landscape (including inverted)
    // param stand - on_stand or off_stand.
    // param facing - face_up or face_down.
    // return String correct orientation
    //---------------------------------------
    QString  getOrientation( const QString& orientation, const QString& stand, const QString& facing );

    //---------------------------------------
    // returns the correct rotation of analyzing accelerometer from the mce service
    // param axis - accelerometer data
    // return vector with rotation coord (in degrees)
    //---------------------------------------
    QVector3D getRotationCoord( const QVector3D& axis );

    //---------------------------------------
    // returns the correct illumination of analyzing light sensor from hal service
    // brightness - data from hal
    // return qint32 correct illumination (in percentage)
    //---------------------------------------
    qint32 getIllumination( qint32 brightness);

    //---------------------------------------
    // returns the db tapping state of analyzing accelerometer using timer
    // axis - accelerometer data
    // qint32 - state of db tapping
    //---------------------------------------
    qint32 getDbTapping( const QVector3D& axis );

    //List
    QStringList m_channels;

    //Maps
    StartMap    m_startMap;
    Frequencies m_Frequencies;
    TimerMap    m_timerMap;
    StateChannelsMap m_stateChannelsMap;
    HistoryDtList history_list;
    SingleStartMap m_SingleStart;

    QTimer* interval_tapping;
    qreal scale_factor;
};

#endif // QSENSORSERVICE_H_

