/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "getmsglisttask.h"
#include <QStringList>
#include <QDateTime>
#include <QDebug>

#include "qmessagingservice.h"
#include "qmessagingiterator.h"
#include "qmessagingconstants.h"
#include "messagedata.h"

//QtMobility includes
#include <QMessage>
#include <QMessageFilter>
#include <QMessageId>

class qMsgSorter
{
public:
    qMsgSorter(int aSortKey, bool aAscending): iSortKey(aSortKey), iAscending(aAscending) {}
    bool operator()(const QVariant &t1, const QVariant &t2) const
    {
        if (iSortKey == SORT_BY_DATE)
            return (t1.toMap().value(KCTIME).toDateTime() < t2.toMap().value(KCTIME).toDateTime()) == iAscending;
        else if (iSortKey == SORT_BY_SENDER)
            return (t1.toMap().value(KCSender).toString() < t2.toMap().value(KCSender).toString()) == iAscending;
        else
            return false; //default
    }
    int iSortKey;
    bool iAscending;
};


GetMsgListTask::GetMsgListTask(qint32 transaction_id, const MessageFilter& matchPattern, QMessageManager& manager)
: iTransactionId(transaction_id),iMatchPattern(matchPattern), iManager(manager)
{
    iSynchronous = false;
}

void GetMsgListTask::finishTask()
{
    qDebug() << "GetMsgListTask::finishTask. Id=" << iTransactionId;
    iMutex2.lock();
    iMutex2.unlock();
    iWait.wakeAll();
    qDebug() << "GetMsgListTask::finishTask end";
}

int GetMsgListTask::getMessagesNow(QVariantList& aResults)
{
    iSynchronous = true;
    return getMessages(aResults);
}

int GetMsgListTask::getMessages(QVariantList& aResults)
{
    int res = SUCCESS;
    QString errMsg = "";
    qDebug() << __FUNCTION__;

    if (iMatchPattern.allowSMS() || iMatchPattern.allowMMS())
    {
        qDebug() << "Get SMS events";
        QMessageFilter msgFilter(QMessageFilter::byType(QMessage::Sms));
        QMessageIdList msgIdsList;
        msgIdsList = iManager.queryMessages(msgFilter);
        foreach (const QMessageId &id, msgIdsList)
        {
            QMessage message(iManager.message(id));
            QVariantMap msg = MessageData::fillMessageMap(message);
            if (iMatchPattern.checkFilter(msg))
            {
                aResults.append(static_cast<QVariant>(msg));
            }
        }
    }

    //Sort here
    if (res == SUCCESS)
    {
        qSort(aResults.begin(), aResults.end(), qMsgSorter(iMatchPattern.sortKey(), iMatchPattern.sortMethod() == SORT_ASCENDING));
    }
//    qDebug() << "GET MESSAGES RESULTS:\n" << aResults;
    return res;
}

void GetMsgListTask::run()
{
    qDebug() << "GetMsgListTask::run(): transaction id=" << iTransactionId;

    int err = getMessages(iMessages);

    iMutex2.lock();
    emit signalGetMsgListCb(this, iTransactionId, err, iMessages);
    iMutex.lock();
    iMutex2.unlock();
    iWait.wait(&iMutex);
    iMutex.unlock();
    qDebug() << "GetMsgListTask::run() - end";
}
