/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef RESIZEIMAGETASK_H
#define RESIZEIMAGETASK_H

#include <QObject>
#include <QUrl>

class QPixmap;

/**
 * Class representing resize image task. Provides methods to run task asynchronously by invoking appropriate steps through even loop iteration
 * This task cannot be executed in separate thread since QPixmap operations must be executed in the gui thread (main thread)
 * @lib libmediaserviceplugin.so
 * @since x.y
 */
class ResizeImageTask : public QObject
{
    Q_OBJECT

public:
    /**
     * Constructor
     * @param aTransactionId transaction identifier representing asynchronous request
     * @param aUrl URI to original message
     * @param aWidth Required width of resized image
     * @param aWidth Required height of resized image
     * @param aAspectRatioMode Indicates whether resized image should keep aspect ratio (see Qt::AspectRatioMode)
     * @since x.y
     */
    ResizeImageTask(int aTransactionId, QUrl aUrl, int aWidth, int aHeight, int aAspectRatioMode, const QString& aDestination);

    /** Default destructor */
    ~ResizeImageTask();

    /** Starts the task. Calls doLoad, then doScale, then doSave */
    void run(bool synchronous);

    /** Starts the scaling task. Calls asynchronously doScale, then doSave */
    void runFromScaling(const QPixmap& aOriginalPixmap);

    /**
     * Finishes the task. Task becomes deleted after invoking this method. This MUST be called by client during signalResizeImageCb signal processing
     * @since x.y
     */
    void finishTask();

    QUrl getResultUri() const {return iResultUri;}
    int getResultError() const {return iResultError;}

Q_SIGNALS:
    /**
     * Signal which is emitted when resizing of the image is completed. Reciever client MUST call aTask->finishTask() to free task resources
     * @param aTask Task which emitted this signal. Client MUST call aTask->finishTask to free task resources
     * @param aUri URI of resized image
     * @param aErrorCode Error code. See @deviceError enumeration
     * @param aTransactionId Transaction identifier of asynchronous request
     * @since x.y
     */
    void signalResizeImageCb(ResizeImageTask* aTask, QString aUri, int aErrorCode, int aTransactionId);

private slots:
    /** Performs loading of original image data. Synchronously/Asynchronously runs doScale */
    void doLoad(bool synchronous);

    /** Performs scaling of original image data. Synchronously/Asynchronously runs doSave */
    void doScale(bool synchronous);

    /** Save the scaled image*/
    void doSave(bool synchronous);

signals:
    void asyncLoad(bool synchronous);
    void asyncScale(bool synchronous);
    void asyncSave(bool synchronous);
private:
    int iTransactionId;
    QUrl iSourceUri;
    int iWidth;
    int iHeight;
    int iAspectRatioMode;
    QUrl iResultUri;
    int iResultError;
    QString iDestination;

    QPixmap* iOriginal;
    QPixmap* iResult;
};
 
#endif // RESIZEIMAGETASK_H
