/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QUrl>

#include "addmusichandler.h"
#include "qmediaerrorcodes.h"
#include "qmediakeyconstants.h"
#include "mediacontenthelper.h"
#include "rdfselect.h"



AddMusicHandler::AddMusicHandler(qint32 aTransactionId, const QString& aUri):
iTransactionId(aTransactionId),
iUri(aUri)
{

}

AddMusicHandler::~AddMusicHandler()
{

}

void AddMusicHandler::start(int& aErrorCode, QString& aErrorMessage)
{
    qDebug() << Q_FUNC_INFO;
    aErrorCode = NO_ERROR;
    aErrorMessage = "";
    //firstly, check the uri is correct
    qDebug() << "check the uri is correct" << checkCorrectUri(aErrorCode, aErrorMessage) << "/n" << "   aErrorCode = " << aErrorCode << "   aErrorMessage = " << aErrorMessage;
    if (checkCorrectUri(aErrorCode, aErrorMessage))
    {
        qDebug() << "aErrorCode = " << aErrorCode;
        if (aErrorCode != NO_ERROR)
        {
            return;
        }
    }
    else
    {
        emit musicAdded(this, iTransactionId, aErrorCode);
        aErrorCode = NO_ERROR;
        aErrorMessage = "";
        return;
    }
    qDebug() << "checkFileIndexed() = " << checkFileIndexed();
    if (checkFileIndexed())
    {
        emit musicAdded(this, iTransactionId, NO_ERROR); //do not add any code below this signal
        return;
    }
    else
    {
         emit musicAdded(this, iTransactionId, NOT_SUPPORTED_ERR); //do not add any code below this signal
        return;
    }

    aErrorCode = GENERAL_ERROR;
    aErrorMessage = "General error - unable to subscribe for notifications";

}


void AddMusicHandler::finish()
{
//    qDebug() << Q_FUNC_INFO;
    delete this;
//    qDebug() << "End " << Q_FUNC_INFO;
}


bool AddMusicHandler::checkFileIndexed() const
{
    RDFselect select(CONTENT_MUSIC);
    QUrl iUrl=QUrl(iUri);
    return select.checkFileIndexed(iUrl.path());
}


bool AddMusicHandler::checkCorrectUri(int& aErrorCode, QString& aErrorMessage) const
{
    bool synchronousError = true;
    QUrl mediaUri(iUri);
    if (mediaUri.isValid() && iUri.startsWith("file://") && iUri.length() <= KMaxFileUriLength)
    {

        aErrorCode = NO_ERROR;
        aErrorMessage = "Success";
        QString mimeTypeByContent = "";
        QString mimeTypeByExt = "";
        aErrorCode = MediaContentHelper::getMimeType(iUri, mimeTypeByContent, mimeTypeByExt);
        if (aErrorCode == NO_ERROR)
        {

            if ( !mimeTypeByExt.startsWith("audio"))
            {
                aErrorCode = NOT_SUPPORTED_ERR;
                aErrorMessage = "File type is not supported";
                synchronousError = false;
            }
        }
        else
        {

            if (aErrorCode == URI_NOT_FOUND_ERR)
            {

                //if the file doesn't exist - interpret this as INVALID_URI_ERR as per spec
                aErrorCode = INVALID_URI_ERR;
            }
            synchronousError = false;
            aErrorMessage = "Operation failed";
        }
    }
    else
    {

        aErrorCode = INVALID_URI_ERR;
        aErrorMessage = "Operation failed";
        synchronousError = false;
    }
    qDebug()<<"AddMusicHandler::checkCorrectUri ret="<<synchronousError;
    return synchronousError;
}
