/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef FileSystemService_H_
#define FileSystemService_H_

//INCLUDES
#include <QMap>
#include <QVariant>
#include <secsession.h>
#include "servicesecurityinterface.h"
#include "filesysteminterface.h"
#include "fslog.h"
#include "fsconstants.h"

class FileSystemWorker;
class FileIOWorker;

#define SYNC_CALL -1

// CLASS DECLARATION

/**
 *  The class implements the interface for filesystem service
 *  described in IFileSystem interface class
 *  @lib filesystemserviceplugin.qtplugin
 *  @since x.y
 */
class FileSystemService : public QObject,
public IFileSystem,
public IServiceSecurity
    {
    Q_OBJECT
    Q_INTERFACES(IFileSystem)

public:
    //public interfaces:

    /**
     * Constructor
     */
    FileSystemService();

    /**
     * Destructor
     */
    virtual ~FileSystemService();

    /**
     * Provides implementation for setSecuritySession derived from
     * IServiceSecurity.
     * Needed to handle capabilities.
     * @since x.y
     * @param aSecSession - Security Manager session
     * @return none
     */
    void setSecuritySession(WRT::SecSession *secSession);

public slots:
//Public Slots

/**
 * Provides implementation for createDir() function defined in IFileSystem.
 * creates the directory with the user specified path
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param srcUri - path to be created.
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant createDir(const QString& srcUri, int callBackId = SYNC_CALL);

/**
 * Provides implementation for remove() function defined in IFileSystem.
 * removes the user specified path
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param path - path to be removed.
 * @param recursive - remove recursively?.
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant remove(const QString& path, bool recursive = false, int callBackId = SYNC_CALL);

/**
 * Provides implementation for rename() function defined in IFileSystem.
 * renames the user specified path
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param path - path to be renamed.
 * @param newName - new name proposed.
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant rename(const QString& path, const QString& newName, int callBackId = SYNC_CALL);

/**
 * Provides implementation for getDirContents() function defined in IFileSystem.
 * Returns the contents of a directory based on match pattern
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param srcUri - source directory path
 * @param matchPattern - pattern to be matched
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant getDirContents(const QString& srcUri, const QString& matchPattern = "*",
        int callBackId = SYNC_CALL);

/**
 * Provides implementation for copy() function defined in IFileSystem.
 * copies a file or directory from source to destination
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param srcUri - source path
 * @param destUri - desitnation path
 * @param newName - new Name provided
 * @param overWrite - overwrite if destination exists
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant copy(const QString& srcUri, const QString& destUri, QString newName = NULL,
        bool overWrite = false, int callBackId = SYNC_CALL);

/**
 * Provides implementation for move() function defined in IFileSystem.
 * moves a file or directory from source to destination
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param srcUri - source path
 * @param destUri - desitnation path
 * @param newName - new Name provided
 * @param overWrite - overwrite if destination exists
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant move(const QString& srcUri, const QString& destUri, QString newName = NULL,
        bool overWrite = false, int callBackId = SYNC_CALL);

/**
 * Provides implementation for search function defined in IFileSystem.
 * Returns the searched items based on match pattern and input path.
 * This is an asycnhronous method.
 * @since x.y
 * @param callBackId - call back id.
 * @param matchPattern - pattern to be matched
 * @param srcUri - source directory path
 * @return Returns trans id.
 */
QVariant search(int callBackId, const QString& matchPattern, const QString& srcUri);

/**
 * Provides implementation for getMountPoints function defined in IFileSystem.
 * Returns the available Mount points on the device
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant getMountPoints(int callBackId = SYNC_CALL);

/**
 * Provides implementation for getDefaultPath function defined in IFileSystem.
 * Returns the available default paths on the device
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param contentType - content type
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant getDefaultPath(const QString& contentType, int callBackId = SYNC_CALL);

/**
 * Provides implementation for notifyMountEvents  function defined in IFileSystem.
 * Notifies about the newly available mount point.
 * This acts as asynchronous notification method.
 * @since x.y
 * @param callBackId - call back id.
 * @return Returns dummy transaction id.
 */
QVariant notifyMountEvents(int callBackId);

/**
 * Provides implementation for cancelNotify function defined in IFileSystem.
 * cancels the notification of mount events.
 * This acts as both synchronous method.
 * @since x.y
 * @return Returns map with error code & message
 */
QVariant cancelNotify();


/**
 * Provides implementation for openFile() function defined in IFileSystem.
 * Returns the file handle
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param uri - source file path
 * @param mode - mode in which the file is to be opened
 * @param encoding - encoding format
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant openFile(const QString& uri, const QString& mode, const QString& encoding,
        int callBackId = SYNC_CALL);

/**
 * Provides implementation for close() function defined in IFileSystem.
 * closes the file handle
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant close(int handleId, int callBackId = SYNC_CALL);

/**
 * Provides implementation for readLine() function defined in IFilesystem
 * Reads data till a new line is encountered
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param maxlength - maximum length of data to be read from file
 * @pos - position in the file from where to read
 * @param callBackId - call back id.
 * @return Returns map with erro code & data in sync mode
 * or trans id in async mode.
 */
QVariant read(int handleId, int maxLength, int pos = -1, int callBackId = SYNC_CALL);

/**
 * Provides implementation for readLine() function defined in IFilesystem
 * Reads data till a new line is encountered
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param maxlength - maximum length of data to be read from file
 * @param callBackId - call back id.
 * @return Returns map with erro code & data in sync mode
 * or trans id in async mode.
 */
QVariant readLine(int handleId, int maxLength = 0, int callBackId = SYNC_CALL);
/**
 * Provides implementation for readBase64() function defined in IFilesystem
 * Reads data till maxLength from the file and returns it in Base64 encoded format
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param maxlength - maximum length of data to be read from file
 * @param callBackId - call back id.
 * @return Returns map with error code & data in sync mode
 * or trans id in async mode.
 */
QVariant readBase64(int handleId, int maxLength, int callBackId = SYNC_CALL);


/**
 * Provides implementation for write() function defined in IFileSystem
 * writes the given data in the file
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param data - data to be written in the file
 * @pos - position in the file to write
 * @param callBackId - call back id.
 * @return Returns map with erro code & data in sync mode
 * or trans id in async mode.
 */
QVariant write(int handleId, const QString& data, int pos = -1, int callBackId = SYNC_CALL);

/**
 * Provides implementation for writeLine() function defined in IFileSystem
 * writes the given data along with a newline character at the end
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param data - data to be written in the file
 * @param callBackId - call back id.
 * @return Returns map with erro code & data in sync mode
 * or trans id in async mode.
 */
QVariant writeLine(int handleId, const QString& data, int callBackId = SYNC_CALL);
/**
 * Provides implementation for writeBase64() function defined in IFileSystem
 * writes the Base64 encoded data in the file
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param data - data to be written in the file
 * @param callBackId - call back id.
 * @return Returns void in sync mode
 * or trans id in async mode.
 */
QVariant writeBase64(int handleId, const QString& data, int callBackId = SYNC_CALL);


/**
 * function to cancel async operation.
 * cancels any ongoing async operation based on transaction id
 * This is a synchronous method.
 * @since x.y
 * @param transId - Transaction Id.
 * @return Returns map with error code & message.
 */
QVariant cancel(int cbId);

/**
 * Provides implementation for flush() function defined in IFileSystem.
 * Performs a flush on a file handle
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param handleId - handle id of the file
 * @param callBackId - call back id.
 * @return Returns map with error code & message in sync mode
 * or trans id in async mode.
 */
QVariant flush(int handleId, int callBackId = SYNC_CALL);

/**
 * Provides implementation for seek() function defined in IFileSystem.
 * Moves the postion of file handle
 * This is a synchronous method.
 * @since x.y
 * @param handleId - handle id of the file
 * @param seekOption - seekoption shoule be one of SEEKSTART, SEEKCURRENT, SEEKEND
 * @param position - position to be moved.
 * @return Returns map with error code & message along with position
 */
QVariant seek(int handleId, int seekOption, int position = 0);

/**
 * Provides implementation for getElementInfo() function defined in IFileSystem.
 * Returns the properties of an element
 * This acts as both synchronous & asycnhronous method based on callback id.
 * @since x.y
 * @param srcUri - source element path
 * @param callBackId - call back id.
 * @return Returns map with error code & message, data in sync mode
 * or trans id in async mode.
 */
QVariant getElementInfo(const QString& srcUri, int callBackId = SYNC_CALL);


/**
 * Provides implementation for search function defined in IFileSystem.
 * Returns the searched items based on match pattern.
 * This is an asycnhronous method.
 * @since x.y
 * @param callBackId - call back id.
 * @param matchObject - may contain <Date> createDate, <String> fileName,
 *                      <String> fileName, <Number> fileSize, <boolean> isDirectory,
 *                      <Date> lastModifyDate
 * @return Returns trans id.
 */
QVariant searchMatchObject(int callBackId, const QMap<QString,QVariant>& matchObject);

private:
    //private member functions
    /**
     * This function will extract the file path from the uri provided by
     * the user.
     * @since x.y
     * @param Uri - Uri provided by the user.
     * @return Returns true/false based on Uri validity
     */
    bool ExtractPathFromUri(QString &Uri);

    signals:
    //file operation signals
    void createDirSignal(int tid, QVariant returnData);
    void removeSignal(int tid, QVariant returnData);
    void renameSignal(int tid, QVariant returnData);
    void getDirContentsSignal(int tid, QVariant returnData);
    void copySignal(int tid, QVariant returnData);
    void moveSignal(int tid, QVariant returnData);
    void searchSignal(int tid, QVariant returnData);
    void openFileSignal(int tid, QVariant returnData);
    void closeSignal(int tid, QVariant returnData);
    void getElementInfoSignal(int tid, QVariant returnData);

    //file IO operation signals
    void readSignal(int tid, QVariant returnData);
    void readLineSignal(int tid, QVariant returnData);
    void readBase64Signal(int tid, QVariant returnData);
    void writeSignal(int tid, QVariant returnData);
    void writeLineSignal(int tid, QVariant returnData);
    void writeBase64Signal(int tid, QVariant returnData);
    void flushSignal(int tid, QVariant returnData);

    //mount point signals
    void getMountPointsSignal(int tid, QVariant returnData);
    void getDefaultPathSignal(int tid, QVariant returnData);
    void notifyMountEventsSignal(int tid, QVariant returnData);

    //search with object
    void searchMatchObjectSignal(int tid, QVariant returnData);

public slots:
//slot called by worker thread on task completion
void ServiceSlot(int task, int cbId, QMap<QString,QVariant> data);

private:
    //Private members

    //executes File worker threads
    void ExecFileSystemCall(FileSystemWorker* thread, int callBackId);

    //executes File IO worker threads
    void ExecFileIOCall(FileIOWorker* thread, int callBackId);

    //deletes the thread pointer
    void deleteThreadPointer(int cbId);
    
    //for maintaining allowed paths
    QVariant mount(QString cjseKey);

    /**
     *  Map to maintain open file handles
     */
    QList<FileInfo> m_openFileList;

    /**
     *  Mount Point Notification handler
     */
    FileSystemWorker* m_NotifyingWorker;

    /**
     *  Mount Point Notification id
     */
    int m_NotificationId;

    /**
     *  Handle to security session
     */
    WRT::SecSession *_sec_session;

    /**
     *  map based on tid and list of live threads
     */
    QMap<int,QThread*> m_threadList;

    /**
     *  data to be sent back to JS
     */
    QVariant m_data;

    /*
     * Hash table of mounted Virtual paths vs Actual paths
     */

    QHash<QString,QString> mountTable;


#ifdef DEBUG_MODE
    MyLog log;
#endif

    };

#endif //FileSystemService_H_
