/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 * ============================================================================
 *  Name        : qcommlogservice.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Qt class for CommLog Service Provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 * ============================================================================
 */

#include <QDebug>
#include <QtDBus/QtDBus>

#include "qcommlogservice.h"
#include "qcommlogerrorcodes.h"
#include "qcommlogkeyconstants.h"
#include "qcommlogiterator.h"

#define DBUS_PATH              "/rtcomeventlogger/signal"
#define DBUS_INTERFACE         "rtcomeventlogger.signal"
#define DBUS_SIGNAL            "NewEvent"


QCommLogService::QCommLogService()
{
    qDebug() << __PRETTY_FUNCTION__;
    iNotifierSet = false;
    connect(this, SIGNAL(asyncGetListComplete(GetCommLogListTask*, qint32, int, QVariantList)),
            this, SLOT (getListCallback(GetCommLogListTask*, qint32, int, QVariantList)), Qt::QueuedConnection);
}

QCommLogService::~QCommLogService()
{
}

void QCommLogService::setSecuritySession(WRT::SecSession *aSecSession)
{
    iSecSession = aSecSession;
}

void QCommLogService::getListCallback(GetCommLogListTask *aTask, qint32 aTransactionId, int aErrorCode, QVariantList aMessages)
{
    qDebug() << __PRETTY_FUNCTION__ << "Id=" << aTransactionId << ", Error=" << aErrorCode << "MsgCount=" << aMessages.count();

    if (iGoingAsyncTransactions.contains(aTransactionId))
    {
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(aTransactionId));
        QCommLogIterator* commLogIter = new QCommLogIterator(aMessages, aTransactionId);
        commLogIter->setParent(this);
        qDebug() << "Notifying wrt framework";
        emit asyncCallback(aErrorCode, aTransactionId, commLogIter);

    }
    if (aTask)
    {
        aTask->finishTask();
    }
}

void QCommLogService::newEventCallback(int aNewEventId)
{
    qDebug() << __PRETTY_FUNCTION__ << "Notifier set" << iNotifierSet << ", Id=" << aNewEventId << ", TransactionId=" << iNotifierTransactionId;
    if (iNotifierSet)
    {
        QVariantMap jsRes = iEventModel.getEvent(aNewEventId);
        if (!jsRes.isEmpty())
        {
            qDebug() << "New event - Notifying wrt framework";
            emit notificationCallback(NO_ERROR, iNotifierTransactionId, jsRes);
        }
    }
}

QVariant QCommLogService::getList (  const QMap<QString, QVariant>& match,   const int& transId )
{
    qDebug() << __PRETTY_FUNCTION__;
    int retErr = NO_ERROR;
    QString retMsg;
    QMap<QString,QVariant> retMap;

    CommLogFilter filter;
    bool parseResult = filter.parseFromMap(match, retErr, retMsg);
    if (parseResult)
    {
        qDebug() << "Parsing results: Flags: " << filter.FilterFlags();
        qDebug() << "Event type [" << filter.EventType() << "], senders [" << filter.PhoneNumber() << "] number of logs [" << filter.NumberOfLogs() << "]";
        qDebug() << "Filter Start date [" << filter.StartTime() << "]";
        qDebug() << "Filter End date [" << filter.EndTime() << "]";
        qDebug() << "Running task";

        GetCommLogListTask * task = new GetCommLogListTask(transId, filter, iEventModel);
        connect(task, SIGNAL(signalGetCommLogListCb(GetCommLogListTask*, qint32, int, QVariantList)), this,
                SLOT(getListCallback(GetCommLogListTask*, qint32, int, QVariantList)), Qt::QueuedConnection);

        qDebug() << "add to transaction list";
        qint32 trId = transId;
        iGoingAsyncTransactions.append(trId);
        qDebug() << "Event Service::getList -> real starting";
        if (QThreadPool::globalInstance()->activeThreadCount() >= QThreadPool::globalInstance()->maxThreadCount() )
        {
            qDebug()<<"All treads are busy. Increase quantity of threads";
            QThreadPool::globalInstance()->setMaxThreadCount(QThreadPool::globalInstance()->maxThreadCount() + 1);
        }
        QThreadPool::globalInstance()->start(task);

    }
    else
    {
        qDebug() << "Filter parsing error" << retErr << retMsg;
    }
    //if
    retMap.insert(KErrCode, NO_ERROR);
    retMap.insert(KErrMsg, "");
    retMap.insert(KTransId, transId);
    if (retErr == MISSING_ARG_ERR || retErr == INVALID_ARG_ERR)
    {
        //Device Exception
        qDebug() << "Returning from getList" << retErr << retMsg;
        retMap.insert(KErrCode, retErr);
        retMap.insert(KErrMsg, retMsg);
    }
    else if (retErr != NO_ERROR)
    {
        //in these cases we should throw error callback asynchronously
        QVariantList emptyList;
        qint32 trId = transId;
        iGoingAsyncTransactions.append(trId);
        emit asyncGetListComplete(NULL, transId, retErr, emptyList);
    }

    return retMap;
}

QObject* QCommLogService::getList (const QMap<QString, QVariant>& match)
{
    qDebug() << __FUNCTION__ << " Synchronous";
    int retErr = NO_ERROR;
    QString retMsg;
    QObject* commLogIterObj = NULL;
    QVariantMap retMap;
    CommLogFilter filter;
    bool parseResult = filter.parseFromMap(match, retErr, retMsg);
    if (parseResult)
    {
        qDebug() << "Parsing results: Flags: " << filter.FilterFlags();
        qDebug() << "Event type [" << filter.EventType() << "], senders [" << filter.PhoneNumber() << "] number of logs [" << filter.NumberOfLogs() << "]";
        qDebug() << "Filter Start date [" << filter.StartTime() << "]";
        qDebug() << "Filter End date [" << filter.EndTime() << "]";
        qDebug() << "Running task";

        GetCommLogListTask * task = new GetCommLogListTask(0, filter, iEventModel);

        qDebug() << "add to transaction list";

        QVariantList messages;
        retErr = task->getMessagesNow(messages);
        if (retErr == NO_ERROR)
        {
            QCommLogIterator* commLogIter = new QCommLogIterator(messages, 0);
            commLogIter->setParent(this);
            commLogIterObj = static_cast<QObject*>(commLogIter);
            retMap.insert(KErrCode,retErr);
            retMap.insert(KErrMsg, retMsg);
            QList<QString> keyList = retMap.keys();
            foreach(const QString& keyString, keyList)
            {
                commLogIterObj->setProperty(keyString.toUtf8().constData(),retMap[keyString]);
            }
        }
        delete task;
    }
    else
    {
        qDebug() << "Filter parsing error" << retErr << retMsg;
    }

    qDebug() << "Returning from synchronous getList" << retErr << retMsg;
    if(retErr == INVALID_ARG_ERR || retErr == DATA_OUT_OF_RANGE_ERR)
    {
        QVariantList messages;
        QCommLogIterator* commLogIter = new QCommLogIterator(messages, 0);
        commLogIter->setParent(this);
        commLogIterObj = static_cast<QObject*>(commLogIter);
        retMap.insert(KErrCode, retErr);
        retMap.insert(KErrMsg, retMsg);
        QList<QString> keyList = retMap.keys();
        foreach(const QString& keyString, keyList)
        {
            commLogIterObj->setProperty(keyString.toUtf8().constData(),retMap[keyString]);
        }

        return commLogIterObj;

    }

    return commLogIterObj;
}

QVariant QCommLogService::cancel( const int& transactionID )
{
    qDebug() << __PRETTY_FUNCTION__<<"Id = "<<transactionID;
    int retErr = NO_ERROR;
    QString retMsg = "";
    QMap<QString,QVariant> retMap;
    QVariantMap retVal;
    retMap.insert(KTransId, transactionID);

    if (iGoingAsyncTransactions.contains(transactionID))
    {
        qDebug() << "Event Provider: removing transaction success";
        // Remove request from the array
        iGoingAsyncTransactions.removeAt(iGoingAsyncTransactions.indexOf(transactionID));
    }
    else
    {
        qDebug() << "Event Provider: not found: transaction doesn't exist";
        retErr = DATA_NOT_FOUND_ERR;
        retMsg = KDataNotFound;
    }

    retMap.insert(KErrCode, retErr);
    retMap.insert(KErrMsg, retMsg);
    qDebug() << __FUNCTION__ << " End";
    return retMap;
}

QVariant QCommLogService::setNotification ( const int& transId)
{
    qDebug() << __PRETTY_FUNCTION__ << transId;
    QMap<QString,QVariant> retMap;
    retMap.insert(KErrCode,NO_ERROR);
    retMap.insert(KErrMsg,"");
    //Registering callback for listening to SMS and CALL
    if (!iNotifierSet)
    {
        if (!QDBusConnection::sessionBus().connect(
                 QString(), DBUS_PATH, DBUS_INTERFACE,DBUS_SIGNAL,
                 this, SLOT(newEventCallback(int))) )
        {
            qDebug()<<"QCommLogService::connect to interface "<<DBUS_INTERFACE<<" failed";
            iNotifierSet = false;
            retMap.insert(KErrCode,INVALID_ARG_ERR);
            retMap.insert(KErrMsg,"setNotification failed");
            return retMap;
        }
    }
    iNotifierSet = true;
    iNotifierTransactionId = transId;
    retMap.insert(KTransId,transId);
    return retMap;
}

QVariant QCommLogService::cancelNotification(const int& /*transId*/)
{
    qDebug() << __PRETTY_FUNCTION__;
    QMap<QString,QVariant> retMap;
    iNotifierSet = false;
    iNotifierTransactionId = 0;
    //Disconnect
    QDBusConnection::sessionBus().disconnect(
                     QString(), DBUS_PATH, DBUS_INTERFACE,DBUS_SIGNAL,
                     this, SLOT(newEventCallback(int)));
    retMap.insert(KErrCode, NO_ERROR);
    retMap.insert(KErrMsg,"");
    return retMap;
}

QVariant QCommLogService::deleteLogEntry(const int& logId)
{
    qDebug() << __PRETTY_FUNCTION__ << logId;
    int retErr = NO_ERROR;
    QString retMsg;
    QMap<QString,QVariant> retMap;

    bool deleteSuccess = iEventModel.deleteEvent(logId);
    if (!deleteSuccess)
    {
        retErr = DATA_NOT_FOUND_ERR;
        retMsg = KDataNotFound;
    }
    retMap.insert(KErrCode, retErr);
    retMap.insert(KErrMsg, retMsg);
    return retMap;
}
