/*
 * ============================================================================
 *  Name        : qcommlogfilter.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Filter class for CommLog service provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include <QDebug>

#include "qcommlogfilter.h"
#include "qcommlogerrorcodes.h"
#include "qcommlogkeyconstants.h"

CommLogFilter::CommLogFilter():
        iEventType (0),
        iRecent (false),
        iDirectionFlag (0x0),
        iNumberOfLogs (0),
        iFilterFlags(0x0)
{
    for (int i=0;i<eventFieldList.length();i++)
        iHashTable.insert(eventFieldList[i],KBASE<<i);
}

CommLogFilter::~CommLogFilter()
{
}

bool CommLogFilter::parseFromMap(const QVariantMap& aMap, int &aErrorCode, QString& aErrorMessage)
{
    qDebug() << __FUNCTION__<<": "<< aMap;
    aErrorCode = NO_ERROR;
    aErrorMessage = "";

    iFilterFlags = 0;

    QVariant copyValue = aMap.value(KLType);

    //type
    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::String)
        {
           qulonglong eventType = iHashTable.value(copyValue.toString().toLower());
           switch (eventType)
           {
               case KEFCALL:
                  iEventType = KCallEventType;
                  iFilterFlags |= EType;
                  break;

               case KEFSMS:
                  iEventType = KSmsEventType;
                  iFilterFlags |= EType;
                  break;

               default:
                  aErrorCode = DATA_OUT_OF_RANGE_ERR;
                  aErrorMessage = "Event:getList:Type is out of range";
                  return false;
           }
       }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:Type is invalid";
            return false;
        }
    }

    //recent indication
    copyValue = aMap.value(KLRecent);

    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::Bool)
        {
            iRecent = copyValue.toBool();
            iFilterFlags |= ERecent;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:Recent field is invalid";
            return false;
        }
    }

    //Direction flag
    copyValue = aMap.value(KLFlag);
    iDirectionFlag = 0;
    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::String)
        {
            qulonglong eventFlag = iHashTable.value(copyValue.toString().toLower());
            switch (eventFlag)
            {
            case KEFINCOMING:
                iDirectionFlag = KEFINCOMING;
                break;
            case KEFRECEIVED:
                iDirectionFlag = KEFRECEIVED;
                break;
            case KEFOUTGOING:
                iDirectionFlag = KEFOUTGOING;
                break;
            case KEFMISSED:
                if (iFilterFlags & EType)
                {
                    if (iEventType == KCallEventType)
                    {
                        iDirectionFlag = KEFMISSED;
                    }
                    else
                    {
                        aErrorCode = DATA_OUT_OF_RANGE_ERR;
                        aErrorMessage = "Event:getList: Flag is MISSED for SMS - so out of range";;
                        return false;
                    }
                }
                break;
            default:
                aErrorCode = DATA_OUT_OF_RANGE_ERR;
                aErrorMessage = "Event:getList:Flag is out of range";
                return false;
            }
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:Flag is invalid";
            return false;
        }
    }
    if (iDirectionFlag)
    {
        iFilterFlags |= EDirectionFlag;
    }

    //phoneNumber
    copyValue = aMap.value(KLPhone);
    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::String)
        {
            QString phonenumber = aMap.value(KLPhone).toString();
            if (phonenumber.size()>100)
                phonenumber.truncate(100);
            iPhoneNumber = phonenumber;
            iFilterFlags |= EPhoneNumber;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:PhoneNumber is invalid";
            return false;
        }
    }

    //Number of logs
    copyValue = aMap.value(KLNoOfLogs);
    qDebug() << __FUNCTION__<<":iFilterFlags: "<<iFilterFlags;
    qDebug()<<__FUNCTION__<<":copyValue: "<<copyValue;
    if (!copyValue.isNull())
    {
         qDebug()<<__FUNCTION__<<":copyValue: "<<copyValue;
        bool ok = true;
        if ((copyValue.type())== QVariant::Int)
        {
            iNumberOfLogs = copyValue.toInt(&ok);
            iFilterFlags |= ENumberOfLogs;
        }
        else if ((copyValue.type())== QVariant::UInt)
        {
            iNumberOfLogs = copyValue.toUInt(&ok);
            iFilterFlags |= ENumberOfLogs;
        }
        else if ((copyValue.type())== QVariant::LongLong)
        {
            iNumberOfLogs = copyValue.toLongLong(&ok);
            iFilterFlags |= ENumberOfLogs;
        }
        else if ((copyValue.type())== QVariant::ULongLong)
        {
            iNumberOfLogs = copyValue.toULongLong(&ok);
            iFilterFlags |= ENumberOfLogs;
        }
        else if ((copyValue.type())== QVariant::Double)
        {
            double db = copyValue.toDouble(&ok);
            iNumberOfLogs = db;
            if (db != iNumberOfLogs)
            {
                qDebug() << "Double value is not an integrer - out of range";
                aErrorCode = DATA_OUT_OF_RANGE_ERR;
                aErrorMessage = "Event:getList:nOfLogs field is out of range";
                return false;
            }
            iFilterFlags |= ENumberOfLogs;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:nOfLogs field is invalid";
            return false;
        }

        if (!ok)
        {
            qDebug("Unable to convert number");
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:nOfLogs field is invalid";
            return false;
        }

        if (iNumberOfLogs < 0)
        {
            qDebug() << "Double value is less than 0 - out of range";
            aErrorCode = DATA_OUT_OF_RANGE_ERR;
            aErrorMessage = "Event:getList:nOfLogs field is out of range";
            return false;
        }
    }

    //start Date
    copyValue = aMap.value(KLStartTime);
    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::DateTime)
        {
            qDebug()<<"startTime "<<copyValue.type();
            iStartTime = copyValue.toDateTime();
            iFilterFlags |= EStartTime;
            qDebug()<<"startTime "<<iStartTime;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:Start date field is invalid";
            return false;
        }
    }
    //end Date
    copyValue = aMap.value(KLEndTime);

    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::DateTime)
        {
            iEndTime = copyValue.toDateTime();
            iFilterFlags |= EEndTime;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:End date field is invalid";
            return false;
        }
    }

    //contact name
    copyValue = aMap.value(KLContactName);
    if (!copyValue.isNull())
    {
        if ((copyValue.type())== QVariant::String)
        {
            QString cntName = copyValue.toString();
            iContactName = cntName;
            iFilterFlags |= EContactName;
        }
        else
        {
            aErrorCode = INVALID_ARG_ERR;
            aErrorMessage = "Event:getList:Contact name is invalid";
            return false;
        }
    }

    qDebug() << "Parse OK.Flags "<< iFilterFlags <<".Type[" << iEventType << "].Recent[" << iRecent << "].Phone[" << iPhoneNumber <<"].Direction[" << iDirectionFlag << "]";
    return true;
}

bool CommLogFilter::allowSms() const
{
    bool res = ((FilterFlags() & EType) && (EventType() == KSmsEventType)) ||
        !(FilterFlags() & EType);
    return res;
}

bool CommLogFilter::allowIncoming() const
{
    bool res = ((FilterFlags() & EDirectionFlag) && (DirectionFlag() == KEFINCOMING)) ||
                (FilterFlags() & EDirectionFlag) && (DirectionFlag() == KEFRECEIVED) ||
                !(FilterFlags() & EDirectionFlag);
    return res;
}

bool CommLogFilter::allowOutgoing() const
{
    bool res = ((FilterFlags() & EDirectionFlag) && (DirectionFlag() == KEFOUTGOING)) ||
        !(FilterFlags() & EDirectionFlag);
    return res;
}

bool CommLogFilter::allowMissed() const
{
    bool res = ((FilterFlags() & EDirectionFlag) && (DirectionFlag() == KEFMISSED)) ||
        !(FilterFlags() & EDirectionFlag);
    return res;
}

bool CommLogFilter::allowCall() const
{
    bool res = ((FilterFlags() & EType) && (EventType() == KCallEventType)) ||
        !(FilterFlags() & EType);
    return res;
}

bool CommLogFilter::allowStartTime(const QDateTime& aEventTime) const
{
    bool res = (FilterFlags() & EStartTime);
    if ((res && StartTime() <= aEventTime) || !res)
    {
        return true;
    }
    return false;
}

bool CommLogFilter::allowEndTime(const QDateTime& aEventTime) const
{
    bool res = (FilterFlags() & EEndTime);
    if ((res && EndTime() >= aEventTime) || !res)
    {
        return true;
    }
    return false;
}

bool CommLogFilter::allowPhoneNumber(const QString& aPhoneNumber) const
{
    bool res = (FilterFlags() & EPhoneNumber);
    if (res)
    {
        QRegExp regExp (iPhoneNumber, Qt::CaseSensitive, QRegExp::Wildcard);
        if (!aPhoneNumber.contains(regExp))
        {
            return false;
        }
    }
    return true;
}

bool CommLogFilter::allowContactName(const QString& aContactName) const
{
    bool res = (FilterFlags() & EContactName);
    if (res)
    {
        QRegExp regExp (iContactName, Qt::CaseSensitive, QRegExp::Wildcard);
        if (!aContactName.contains(regExp))
        {
            return false;
        }
    }
    return true;
}

