/*
 * ============================================================================
 *  Name        : qcommlogcallandsmsmodel.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Call and SMS Model for CommLog service provider
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QThread>

#include "qcommlogcallandsmsmodel.h"
#include "qcommlogerrorcodes.h"
#include "qcommlogkeyconstants.h"

CommCallAndSMSModel::CommCallAndSMSModel():
        iSmsModelInitialized(false),
        iCallModelInitialized(false)
{
    qDebug() << "CommCallAndSMSModel::CommCallAndSMSModel " << QThread::currentThreadId();
    /*New RTCOM object*/
    dbElement = rtcom_el_new();
    if (!RTCOM_IS_EL(dbElement))
    {
        qDebug()<< "[commlog]: Couldn't create RTComEl. Aborting.";
    }
}

CommCallAndSMSModel::~CommCallAndSMSModel()
{
    g_object_unref(dbElement);
}

bool CommCallAndSMSModel::getEvents(const CommLogFilter& aFilter, QVariantList& aEvents, int &aErrorCode, QString& aErrorMessage)
{
    qDebug() <<__FUNCTION__ << aFilter.EventType() << aFilter.DirectionFlag();
    bool res = true;
    aErrorCode = NO_ERROR;
    Q_UNUSED(aErrorMessage);
    /**Get sms events**/
    if (aFilter.allowSms())
    {
        if (!iSmsModelInitialized)
        {
            res = smsInit();
        }
    }

    /**Get call events**/
    if (res && aFilter.allowCall())
    {
        if (!iCallModelInitialized)
        {
            res = callInit();
        }
    }

    if (res)
    {
        if (aFilter.allowSms())
        {
            qDebug()<<"Allow sms";
            fillDataFromModel(aFilter, iSmsModel, aEvents);
        }

        if (aFilter.allowCall())
        {
            qDebug()<<"Allow calls";
            fillDataFromModel(aFilter, iCallModel, aEvents);
        }
    }
    else
    {
        aErrorCode = GENERAL_ERROR;
    }
    qDebug() << "CommCallAndSMSModel::getEvents end. " << res << aErrorCode;
    return res;
}

QVariantMap CommCallAndSMSModel::getEvent(int aEventId)
{

    qDebug()<<__PRETTY_FUNCTION__<<"new event was happened. aEventId = "<<aEventId;
    bool res = true;
    QVariantMap rMap;
    Event aEvent;
    res = CommHistoryModel::getEvent(dbElement,aEventId,aEvent);
    if (res)
    {
        qDebug()<<"Event found. Mapping it";
        rMap = CommHistoryModel::fillMessageData(aEvent);
        res = (aEvent.getServiceId() == Event::SMSService || aEvent.getServiceId() == Event::CallService);
    }
    qDebug() << "CommCallAndSMSModel::getEvent " << aEventId <<". res = " << res ;
    return rMap;
}

bool CommCallAndSMSModel::deleteEvent(int aEventId)
{
    bool res = true;
    GError *err = 0;
    /*Delete event from DB*/
    rtcom_el_delete_event(dbElement,aEventId,&err);
    if (err)
    {
        res = false;
        qDebug() << "[commlog]: CommCallAndSMSModel::deleteEvent " << aEventId <<". res = " << res << " err=" << QString(err->message);
        g_object_unref(err);
        return res;
    }
    qDebug() << "[commlog]: CommCallAndSMSModel::deleteEvent " << aEventId <<" was successful.";
    return res;
}

bool CommCallAndSMSModel::smsInit()
{
    qDebug() << "CommCallAndSMSModel::smsInit";
    iSMSMutex.lock();
    bool res = iSmsModelInitialized;
    if (!iSmsModelInitialized)
    {
        res = iSmsModel.getEvents(dbElement);
        iSmsModelInitialized = true;
    }
    iSMSMutex.unlock();
    qDebug() << "CommCallAndSMSModel::smsInit end" << res;
    return res;
}

bool CommCallAndSMSModel::callInit()
{
    qDebug() << "CommCallAndSMSModel::callInit";
    iCallMutex.lock();
    bool res = iCallModelInitialized;
    if (!iCallModelInitialized)
    {
        res = iCallModel.getEvents(dbElement);
        iCallModelInitialized = true;
    }
    iCallMutex.unlock();
    qDebug() << "CommCallAndSMSModel::callInit end" << res;
    return res;
}

void CommCallAndSMSModel::fillDataFromModel(const CommLogFilter& aFilter, CommHistoryModel& aModel, QVariantList& aEvents)
{
    const int count = aModel.getEventTree().count();
    qDebug() << "CommCallAndSMSModel::fillDataFromModel" << count;
    int added = 0;
    for (int i = 0; i < count; i++)
    {
        if ((aFilter.FilterFlags() & CommLogFilter::ENumberOfLogs) && added >= aFilter.NumberOfLogs())
        {
            break;
        }
        if (checkFilter(aFilter, aModel.getEventTree().value(i)))
        {
            aEvents.append(static_cast<QVariant>(aModel.getEventTree().value(i)));
            added++;
        }


    }
}


bool CommCallAndSMSModel::checkFilter(const CommLogFilter& aFilter, const QVariantMap& aEvent) const
{
    bool res = true;
    if (aEvent.value(KLType).toString() != KLSms &&
        aEvent.value(KLType).toString() != KLCall)
    {
        res = false;
    }
   if (aEvent.value(KLType).toString() == KLSms && !aFilter.allowSms())
    {
        res = false;
    }
    else if (aEvent.value(KLType).toString() == KLCall && !aFilter.allowCall())
    {
        res = false;
    }

    //filter direction flag
    if (aEvent.value(KLFlag).toString() == KLOutgoing && !aFilter.allowOutgoing())
    {
        res = false;
    }
    else if ( ( aEvent.value(KLFlag).toString() == KLIncoming || aEvent.value(KLFlag).toString() == KLReceived) && !aFilter.allowIncoming())
    {
        res = false;
    }
    else if (aEvent.value(KLFlag).toString() == KLMissed && !aFilter.allowMissed())
    {
        res = false;
    }

    //filter phone number
    if (!aFilter.allowPhoneNumber(aEvent.value(KLPhone).toString()))
    {
        res = false;
    }

    //filter contact name
    if (!aFilter.allowContactName(aEvent.value(KLContactName).toString()))
    {
        res = false;
    }

    //filter start time
    if (!aFilter.allowStartTime(aEvent.value(KLTime).toDateTime()))
    {
        res = false;
    }

    //filter end time
    if (!aFilter.allowEndTime(aEvent.value(KLTime).toDateTime()))
    {
        res = false;
    }
    return res;
}

