/*
 * ============================================================================
 *  Name        : qcameraservice.cpp
 *  Part of     : serviceframework / WRT
 *  Description : Implementation of Camera Service
 *  Version     : %version: 14 % << Don't touch! Updated by Synergy at check-out.
 *
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QtCore>
#include <QDebug>
#include <QPixmap>
#include <QDir>
#include <QFileInfo>

#include "qcameraservice.h"

#define HILDON_APPMGR_PATH "/com/nokia/HildonDesktop/AppMgr"
#define HILDON_APPMGR_IF "com.nokia.HildonDesktop.AppMgr"
#define HILDON_SERVICE "com.nokia.HildonDesktop.AppMgr"

#define CAM_UI_IF "com.nokia.cameraui"
#define CAM_UI_SERVICE "com.nokia.cameraui"

#define CAM_PLAY_SERVICE "org.maemo.Playback"
#define CAM_PLAY_IF "org.freedesktop.DBus.Properties"

#define CAM_SIGNAL_NE  //camera-ui doesn't emit any signals about pictures saving

#define DEFAULT_PHONE_PATH "/home/user/MyDocs/DCIM"
#define DEFAULT_MMC_PATH   "/media/mmc1/DCIM"

#define NOTIFY_SIGNAL  "Notify"
#define PICTUREDONE_SIGNAL "PictureDone""







QCameraService::QCameraService()
{
    qDebug()<<__FUNCTION__;
    iNotifySet = false;
    //Connecting to Camera-UI interface
    if( !QDBusConnection::sessionBus().connect(
                                              CAM_UI_SERVICE,
                                              QString(),
                                              CAM_PLAY_IF,
                                              NOTIFY_SIGNAL,
                                              this,
                                              SLOT(getNotify(const QDBusMessage &)) ) )
    {
        qDebug()<<"Connect to camera-ui failed.Signal Notify";
    }
#ifndef CAM_SIGNAL_NE
    if( !QDBusConnection::systemBus().connect(
                                              QString(),
                                              QString(),
                                              CAM_UI_IF,
                                              PICTUREDONE_SIGNAL,
                                              this,
                                              SLOT(pictureDone(const QString &) ) ) )
    {
        qDebug()<<"Connect to camera-ui failed. Signal PictureDone";
    }
#endif
    //Initialize supported size
    QMap<QString, QVariant> supportedFormat;
    supportedFormat.insert(KCMimeType,KCSupportedMimeImageType);

    supportedFormat.insert(KCSupportedHeight,CAMERA_4X3_HIGH_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_4X3_HIGH_WIDTH);
    supportedSizesImage.append(supportedFormat);

    supportedFormat.insert(KCSupportedHeight,CAMERA_4X3_MEDIUM_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_4X3_MEDIUM_WIDTH);
    supportedSizesImage.append(supportedFormat);

    supportedFormat.insert(KCSupportedHeight,CAMERA_4X3_LOW_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_4X3_LOW_WIDTH);
    supportedSizesImage.append(supportedFormat);

    supportedFormat.insert(KCSupportedHeight,CAMERA_16X9_HIGH_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_16X9_HIGH_WIDTH);
    supportedSizesImage.append(supportedFormat);

    supportedFormat.insert(KCSupportedHeight,CAMERA_16X9_MEDIUM_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_16X9_MEDIUM_WIDTH);
    supportedSizesImage.append(supportedFormat);

    supportedFormat.insert(KCSupportedHeight,CAMERA_16X9_LOW_HEIGHT);
    supportedFormat.insert(KCSupportedWidth,CAMERA_16X9_LOW_WIDTH);
    supportedSizesImage.append(supportedFormat);

}


QCameraService::~QCameraService()
{
}

void QCameraService::setSecuritySession(WRT::SecSession *aSecSession)
{
    iSecSession = aSecSession;
}


QVariant QCameraService :: getList (QDateTime aStartTime, QDateTime aEndTime)
{
    Q_UNUSED(aStartTime);
    Q_UNUSED(aEndTime);
    return KCNotSupported;
}


const QVariant QCameraService::supportedSizes()
{
    return supportedSizesImage;
}


QVariant QCameraService::startCamera()
{
    qDebug()<<__FUNCTION__;
    setErrorMap(SUCCESS,KCSuccess);
    //TODO check interface errors
    QDBusInterface interface(HILDON_SERVICE,
                           HILDON_APPMGR_PATH,
                           HILDON_APPMGR_IF,
                           QDBusConnection::sessionBus());
    if( !interface.isValid() )
    {
        setErrorMap(INVALID_ARG_ERR,KCBadArgument);
        qDebug()<<"Interface isn't valid";
        return iErrorMap;
    }
    if ( !iNotifySet)
    {
        qDebug()<<"Calling camera-ui application";
        iNotifySet = true;
        QDBusMessage msg = interface.call("LaunchApplication","camera-ui");
        if( msg.type() == QDBusMessage::ErrorMessage )
        {
            setErrorMap(INVALID_ARG_ERR,KCBadArgument);
            qDebug()<<"Cant launch camera application";
            return iErrorMap;
        }
    }
    else
    {
        qDebug()<<"Currently only one instantiation of camera is supported";
    }
    return iErrorMap;
}


void QCameraService :: setErrorMap(const int aErrCode, const QString& aErrMsg)
{
    iErrorMap.insert(KCErrorCode, aErrCode);
    iErrorMap.insert(KCErrorMessage, aErrMsg);
}

#ifndef CAM_SIGNAL_NE
void QCameraService::pictureDone(const QString &uri)
{
    //Filling pictureData list
    /*
    PictureData
    {
       "uri"
       "Format Data"
    }
    FormatData
    {
       "height"
       "width"
       "type""
    }*/
    qDebug()<<__FUNCTION__;

    QMap<QString, QVariant> format;//format
    QMap<QString, QVariant> onePictureData;
    QString muri;

    //TODO implement FormatData
    if(uri.startsWith(KCSupportedFileProtocol))
    {
        muri = uri;
    }
    else
    {
        muri = KCSupportedFileProtocol + uri;
    }
    onePictureData.insert(KCUri,muri); //insert uri. Getting uri from PictureDone signal

    format.insert(KCSupportedHeight,QPixmap(uri).height());//insert height
    format.insert(KCSupportedWidth,QPixmap(uri).width());//insert width
    format.insert(KCMimeType,KCSupportedMimeImageType);//insert type. Type doesn't change.
    onePictureData.insert(KCFormatData,format);

    pictureData.append(onePictureData);
}
#endif

void QCameraService::sendURIs()
{
    if(iNotifySet && startTime.isValid())
    {
        //Trying to search new images in default paths: phone and mmc
        QDir dirPhone(DEFAULT_PHONE_PATH);
        QDir dirMmc(DEFAULT_MMC_PATH);
        QVariantMap format;//format
        QMap<QString, QVariant> onePictureData;
        QStringList nameFilters;
        nameFilters.append("*.jpg");
        QDateTime finTime = QDateTime::currentDateTime();
        if( !dirPhone.exists() )
        {
            qDebug()<<"Default phone dir not found";

        }
        else
        {
            //Get files
            QList<QFileInfo> picFiles = dirPhone.entryInfoList(nameFilters,QDir::Files,QDir::Time);//Sorting jpeg files by time
            for (int i = 0;i < picFiles.count();i++)
            {
                if( picFiles[i].created() >= startTime && picFiles[i].created()<= finTime)
                {
                    onePictureData.insert(KCUri,KCSupportedFileProtocol + picFiles[i].filePath());
                    format.insert(KCSupportedHeight, QPixmap(picFiles[i].filePath()).height());
                    format.insert(KCSupportedWidth, QPixmap(picFiles[i].filePath()).width());
                    format.insert(KCMimeType,KCSupportedMimeImageType);//insert type. Type doesn't change.
                    onePictureData.insert(KCFormatData,format);

                    pictureData.append(onePictureData);

                }
            }
        }

        if( !dirMmc.exists() )
        {
            qDebug()<<"Default mmc dir not found";
        }
        else
        {
            //Get files
            QList<QFileInfo> picFiles = dirMmc.entryInfoList(nameFilters,QDir::Files,QDir::Time);//Sorting jpeg files by time
            for (int i = 0;i < picFiles.count();i++)
            {
                if( picFiles[i].created() >= startTime && picFiles[i].created()<= finTime)
                {
                    onePictureData.insert(KCUri,KCSupportedFileProtocol + picFiles[i].filePath());
                    format.insert(KCSupportedHeight, QPixmap(picFiles[i].filePath()).height());
                    format.insert(KCSupportedWidth, QPixmap(picFiles[i].filePath()).width());
                    format.insert(KCMimeType,KCSupportedMimeImageType);//insert type. Type doesn't change.
                    onePictureData.insert(KCFormatData,format);

                    pictureData.append(onePictureData);

                }
            }
        }
        //CameraUI closed. Notifying WRT
        //void asyncCallback(int aStatus, int aTransactionId, QVariant aPictureData);
        qDebug()<<"CameraUI closed. Notifying WRT. Count of pictures = "<<pictureData.count();

        iNotifySet = false;
        emit asyncCallback(SUCCESS,1,pictureData);

    }
    else
    {
        qDebug()<<"starCamera() wasn't launched";
    }
}
void QCameraService::getNotify(const QDBusMessage &msg)
{
    if(msg.arguments().count() != 3)
    {
         qDebug()<<"Not supported notify";
         return;
    }
    if(msg.arguments().at(0).toString() == CAM_PLAY_SERVICE &&
       msg.arguments().at(1).toString() == "State")
    {
        if (msg.arguments().at(2).toString() == "Play")
        {
            qDebug()<<"State = Play. CameraUI on top";
            startTime = QDateTime::currentDateTime();
            //Clear list PictureData
            pictureData.clear();
        }
        else if (msg.arguments().at(2).toString() == "Stop")
        {
            qDebug()<<"State = Stop. CameraUI in background";
            sendURIs();
        }
        else
        {
            qDebug()<<"State not recognized";
        }
    }
}
