/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#ifndef SFWMETHOD_H
#define SFWMETHOD_H

#include <QString>
#include <QList>
#include "servicemetadata_global.h"

namespace WRT {

    // FORWARD DECLARATIONS
    class SFWParameter;

    class METADATA_EXPORT SFWMethod
    {

    public:

        /*
         *  Class constructor
         *
         * @param aMethodName method name
         */
        SFWMethod(const QString& aName, const QString& aDescription);

        /**
         *  Class destructor
         */
        ~SFWMethod(void);

        /**
         *  Gets the method name
         *
         * @return method name or default value (empty string) if it is not available
         */
        QString name();

        /**
        *  Gets the method description
        *
        * @return return method description or default value (empty string) if it is not available
        */
        QString description();


        /**
         *  Gets the method detailed description
         *
         * @return return method detailed description or default value (empty string) if it is not available
         */
        QString detailDescription();


        /**
         *  Gets the method response type(synchronous or asynchronous)
         *
         * @return return method response type or default value (empty string) if it is not available
         */
        QString response();

        /**
         *  Number of input parameters for the method
         *
         * @return Number of input parameters
         */
        int inputParametersCount();

        /**
         *  Gets the input parameter metadata
         *
         * @param anIndex [in] input parameter index
         * @return input parameter metadata if available, null otherwise
         */
        SFWParameter* getInputParameter(int anIndex);

        /**
         *  Number of output parameters for the method
         *
         * @return Number of output parameters
         */
        int outputParametersCount();

        /**
         *  Gets the output parameter metadata
         *
         * @param anIndex [in] output parameter index
         * @return output parameter metadata if available, null otherwise
         */
        SFWParameter* getOutputParameter(int anIndex);

        /**
         * Sets the method response type
         *
         * @param aType response type
         */
        void setResponse(const QString& aResponse);

        /**
         * Sets the method detailed description
         *
         * @param aDescription method detailed description
         */
        void setDetailDescription(const QString& aDescription);

        /**
         * Adds a method parameter to the method metadata
         *
         * @param aName method parameter name
         * @param aType method parameter type
         * @param aDescription parameter description
         * @param aMandatory parameter status (true if mandatory)
         * @param aDirection parameter direction (input or output)
         */
        void addParameter(const QString& aName, const QString& aType,
                const QString& aDescription, bool aMandatory, bool aDirection);

    private:

        QString methodResponse;
        QString methodName;
        QString methodDetailDescription;
        QString methodDescription;
        bool mandatory;
        bool direction;
        QList<SFWParameter*> inputParameters;
        QList<SFWParameter*> outputParameters;

    };

}

#endif // SFWMETHOD_H
