/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef servicedatabase_h
#define servicedatabase_h

#include <QStringList>
#include <QSqlQuery>
#include <QtSql>

#include <errdefs.h>
#include <servicedatabasedefines.h>
#include <serviceinfolist.h>
#include <serviceresolutioncriteria.h>
#include <servicemetadata.h>


#ifdef SERVICEDATABASE_LIB
# define SERVICE_DATABASE_EXPORT Q_DECL_EXPORT
#else
# define SERVICE_DATABASE_EXPORT Q_DECL_IMPORT
#endif

namespace WRT {

    class SERVICE_DATABASE_EXPORT ServiceDatabase
    {
    public:

        //! ServiceDatabase::ServiceDatabaseErr
        /*!
         This enum describes the errors that may be returned by the Service Database.
         */
        enum ServiceDatabaseErr {
            SFW_ERROR_INVALID_SEARCH_CRITERIA = DATABASE_ERR_START_VALUE,   /*!< Invalid Search Criteria */
            SFW_ERROR_CANNOT_OPEN_DATABASE,                                 /*!< Can not open database */
            SFW_ERROR_CANNOT_CREATE_TABLES,                                 /*!< Can not create database tables */
            SFW_ERROR_CANNOT_DROP_TABLES,                                   /*!< Error deleting tables */
            SFW_ERROR_DATABASE_NOT_OPEN,                                    /*!< Database is not open */
            SFW_ERROR_INVALID_DATABASE_CONNECTION,                          /*!< Invalid database connection */
            SFW_ERROR_SERVICE_NOT_FOUND,                                    /*!< Can not find the requested service */
            SFW_ERROR_CANNONT_CLOSE_DATABASE,                               /*!< Can not close database */
            SFW_ERROR_BINDING_PARAM_VALUE,                                  /*!< Error binding value to query parameter */
            SFW_ERROR_SERVICE_ALREADY_REGISTERED,                           /*!< A service with the same name and version
                                                                                already registered */
            SFW_ERROR_FILE_PATH_UPDATED,                                    /*!< A service with the same name and version
                                                                                but different file path already registered.
                                                                                Updated the file path in DB.*/
            SFW_ERROR_DB_RECREATED                                         /*!< Service DB is recreated */
        };

    public:

        ServiceDatabase(void);

        virtual ~ServiceDatabase();

        int open();

        int registerService(SFWService &aService);

        int getService(ServiceResolutionCriteria &aCriteria, ServiceInfo &aServiceInfo);

        int listServices(ServiceInfoList &aServiceInfoList);

        int deleteService(const QString &serviceName);

        int close();

        void setDatabasePath(const QString &aDatabasePath);

        bool isOpen();

        int getServiceVersion(const QString &aServiceName, QString &aServiceVersion, QString &aFilePath);

#ifdef __SYMBIAN32__
        int getRomImageVersion(QString &aRomImageVersion);

        int updateRomImageVersion(const QString &aRomImageVersion);

        int insertRomImageVersion(const QString &aRomImageVersion);
#endif

    private:

        int createTables();

        bool checkTables();

        int dropTables();

        int checkExistingService(SFWService &aService);

        bool executeQuery(QSqlQuery &aQuery, const QString &aStatement);

        int insertInterfaceData(QSqlQuery &aQuery, SFWInterface &anInterface, int aModuleID);

        int getLatestVersion(QSqlQuery &aQuery, ServiceResolutionCriteria &aCriteria);

    private:
        QString m_DatabasePath;
        bool m_IsDatabaseOpen;
        QStringList m_ServiceFields;
        QStringList m_InterfaceFields;

#ifdef __SYMBIAN32__
        QStringList m_RomInfoFields;
#endif
    };

}

#endif //servicedatabase_h
