/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "servicedatabase.h"
#include "serviceframeworkdefs.h"

#include <QSqlDatabase>
#include <QSqlRecord>
#include <QSqlError>
#include <QDir>
#include <QVariant>

namespace WRT {

static const char  QT_PATH_SEPARATOR = '/';
#ifdef __linux__
static const char  PATH_SEPARATOR = '/';
#else
static const char  PATH_SEPARATOR[] = "\\";
#endif

enum TBindIndexes
    {
        EBindIndex=0,
        EBindIndex1,
        EBindIndex2,
        EBindIndex3,
        EBindIndex4,
        EBindIndex5
    };

/*!
    \class ServiceDatabase
    Utility class (used by service resolver) that offers support for resolver database management. \n
    A Sqlite database is used to store metadata for each registered service. \n
    The supported operations are:
     - open/close database(create database tables)
     - register service(populate database tables)
     - query service metadata(retrieve data from database tables)
*/

/*!
    Constructor
*/
ServiceDatabase::ServiceDatabase(void)
    : m_IsDatabaseOpen(false)
{
    m_ServiceFields << "ID" << "Name" << "Version" << "FilePath" << "Description" << "XMLFileName";
    m_InterfaceFields << "ID" << "ServiceID" << "Name" << "Version" << "Capabilities" << "Description";
#ifdef __SYMBIAN32__
    m_RomInfoFields << "RomImageVersion";
#endif
}

/*!
    Destructor
*/
ServiceDatabase::~ServiceDatabase()
{
    close();
    QSqlDatabase::removeDatabase(RESOLVERDATABASE_DEFAULT_CONNECTION);
}

/*!
    Opens the service resolver database \n
    The method creates or opens the resolver database and creates tables if they are not present
    @return 0 if operation was successful(database and tables are available), error code otherwise
*/
int ServiceDatabase::open()
{
    int result(0);
    if (!m_IsDatabaseOpen){
        QString path;

        //Create full path to database
#if defined Q_WS_X11
        // In Linux/Maemo use an absolute path to the database file.
        path = QString("/home/user/.local/wrt/api/services.db");
#else
        if (m_DatabasePath.isEmpty ()) {
            // On windows and only in debug mode QDir::currentPath() returns
            // the path of the current cpp file (instead of the path of the
            // folder that contains dll file)
            path = QDir::currentPath();
        } else {
            path = m_DatabasePath;
        }

        if (path.lastIndexOf(QT_PATH_SEPARATOR) != path.length() -1 &&
            path.lastIndexOf(PATH_SEPARATOR) != path.length() -1 ) {
            path.append(PATH_SEPARATOR);
        }
        path.append(RESOLVERDATABASE);
#endif

        //Create and/or open database and create tables if necessary
        QSqlDatabase  database;
        if (QSqlDatabase::contains(RESOLVERDATABASE_DEFAULT_CONNECTION)) {
            database = QSqlDatabase::database(RESOLVERDATABASE_DEFAULT_CONNECTION);
        } else {
            database = QSqlDatabase::addDatabase("QSQLITE");
            database.setDatabaseName(path);
        }

        if (database.isValid()){
            //Create or open resolver database
            if (!database.isOpen()) {
                m_IsDatabaseOpen = database.open();
            } else {
                m_IsDatabaseOpen = true;
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }

        if (m_IsDatabaseOpen) {
            //Check database structure (tables) and recreate tables if necessary
            //If one of the tables is missing remove all tables and recreate them
            //This operation is required in order to avoid data coruption
            if (!checkTables()) {
                if (dropTables() == 0) {
                    result = createTables();
                }
                else {
                    result = ServiceDatabase::SFW_ERROR_CANNOT_DROP_TABLES;
                }
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_CANNOT_OPEN_DATABASE;
        }

        if (result != ServiceDatabase::SFW_ERROR_DB_RECREATED && result != 0) {
            close();
        }
        m_IsDatabaseOpen = (result == 0 || result == ServiceDatabase::SFW_ERROR_DB_RECREATED)? true: false;
    }
    return result;
}

/*!
    Deletes DB entries for a service provider\n
    Custom error codes: \n
    SFW_ERROR_DATABASE_NOT_OPEN in case database is not open \n
    SFW_ERROR_SERVICE_RECORD_ALREADY_REMOVED in case service record already removed from DB
    @param serviceName service provider name
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::deleteService(const QString& serviceName)
{
    int result(0);

    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        int serviceId = -1;
        //Prepare delete query, bind criteria values and execute search
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            bool isActive(false);

            if (!query.prepare(RESOLVERDATABASE_GET_SERVICEID_BY_SERVICENAME)) {
                result = query.lastError().number();
            } else {
                query.bindValue(EBindIndex, QVariant::fromValue(serviceName));
                if (!query.exec()) {
                    result = query.lastError().number();
                } else if (query.next()) {
                    serviceId = query.value(EBindIndex).toInt();
                    query.clear();
                    query.finish();
                    // Unregisteration involves deleting from Service and Interface tables,
                    // which should be performed as an atomic operation(transaction).
                    //Start Transaction
                    isActive = database.transaction();
                    if (!query.prepare(RESOLVERDATABASE_DELETE_SERVICE_BY_ID)) {
                        result = query.lastError().number();
                    } else {
                        query.bindValue(EBindIndex, serviceId);
                        if (!query.exec()) {
                            result = query.lastError().number();
                            if (isActive) {
                                database.rollback();
                            }
                        } else {
                            query.finish();
                            query.clear();
                            if (!query.prepare(RESOLVERDATABASE_DELETE_INTERFACE_BY_SERVICEID)) {
                                result = query.lastError().number();
                                if (isActive) {
                                    database.rollback();
                                }
                            } else {
                                query.bindValue(EBindIndex, serviceId);
                                if (!query.exec()) {
                                    result = query.lastError().number();
                                    if (isActive)
                                        database.rollback();
                                }
                            }
                        }
                    }
                }
            }
            if ( result == 0) {
                //Commit transaction
                if (isActive) {
                    database.commit();
                }
            }
            query.finish();
            query.clear();
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}


/*!
    Registers a new service to resolver database \n
    Registering the same version of the service will remove the previous records for that service and insert new records\n
    Custom error codes: \n
    SFW_ERROR_DATABASE_NOT_OPEN in case database is not open \n
    SFW_ERROR_SERVICE_ALREADY_REGISTERED in case service is already registered \n
    SFW_ERROR_INVALID_DATABASE_CONNECTION in case database connection is not valid \n
    @param aService service metadata
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::registerService(SFWService& aService)
{
    int result(0);
    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        int serviceID;
        bool isActive(false);
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            //Check if the service is already in the database
            if ((result = checkExistingService(aService)) == 0) {
                // Register requires inserts in 2 different tables:
                // Service,Interface
                // Registration is performed as an atomic operation(transaction)
                // in order to make sure all operations succeed
                //Start Transaction
                isActive = database.transaction();
                //Insert information into Service table
                if (query.prepare(RESOLVERDATABASE_INSERT_SERVICE_RECORD)) {
                    query.bindValue(EBindIndex,aService.name());
                    query.bindValue(EBindIndex1,aService.version());
                    query.bindValue(EBindIndex2,aService.filePath());
                    query.bindValue(EBindIndex3,aService.description());
                    query.bindValue(EBindIndex4,aService.xmlFileName());
                    if (!query.exec()) {
                        result = query.lastError().number();
                        query.finish();
                        query.clear();
                        if (isActive) {
                            database.rollback();
                        }
                    } else {
                        serviceID = query.lastInsertId().toInt();
                        //Handle interface related data
                        int interfaceCount = aService.interfaceCount();
                        for (int index=0; index < interfaceCount;index++) {
                            if ((result = insertInterfaceData(query,*(aService.getInterface(index)),serviceID)) != 0) {
                                query.finish();
                                query.clear();
                                if (isActive) {
                                    database.rollback();
                                }
                                break;
                            }
                        }
                        if ( result == 0) {
                            //Commit transaction
                            if (isActive) {
                                database.commit();
                            }
                            query.finish();
                            query.clear();
                        }
                    }
                } else {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                    if (isActive) {
                        database.rollback();
                    }
                }
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}

/*!
    Helper method that checks if the the provided service is already registered to the database\n
    @param aService service metadata
    @param aQuery Sql query
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::checkExistingService(SFWService &aService)
{
    int result(0);
    QSqlDatabase database = QSqlDatabase::database();
    if (database.isValid()) {
        QSqlQuery aQuery(database);
        if (aQuery.prepare(RESOLVERDATABASE_GET_SERVICE)) {
            aQuery.bindValue(EBindIndex,aService.name());
            if (!aQuery.exec()) {
                result = aQuery.lastError().number();
                aQuery.finish();
                aQuery.clear();
            } else {
                if (aQuery.next()) {
                    if (result == 0) {
                        //compare new service version with registered version
                        if (aService.version().compare(aQuery.value(EBindIndex2).toString()) >= 0) {
                            //close current query
                            aQuery.finish();
                            //delete old service registration records
                            result = deleteService(aService.name());
                        } else {
                            result = SFW_ERROR_SERVICE_ALREADY_REGISTERED;
                            aQuery.finish();
                        }
                    }
                } else
                    aQuery.finish();
            }
        } else {
            result = aQuery.lastError().number();
            aQuery.finish();
        }
    } else {
        result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
    }
    return result;
}


/*!
    Helper method that saves interface related data in the Interface table
    @param aQuery database query
    @param anInterface interface metadata
    @param aServiceID service identifier
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::insertInterfaceData(QSqlQuery& aQuery, SFWInterface& anInterface, int aServiceID)
{
    int result(0);
    //Insert interface information into Interface table
    if (aQuery.prepare(RESOLVERDATABASE_INSERT_INTERFACE_RECORD)) {
        aQuery.bindValue(EBindIndex,aServiceID);
        aQuery.bindValue(EBindIndex1,anInterface.name());
        aQuery.bindValue(EBindIndex2,anInterface.version());
        aQuery.bindValue(EBindIndex3,anInterface.capabalities());
        aQuery.bindValue(EBindIndex4,anInterface.description());
        if (!aQuery.exec()) {
            result = aQuery.lastError().number();
        }
    } else {
        result = aQuery.lastError().number();
    }
    return result;
}

/*!
    Helper method that executes a sql statement
    @param aQuery database query
    @param aStatement sql statement
    @return true if operation was successful, false otherwise
*/
bool ServiceDatabase::executeQuery(QSqlQuery &aQuery, const QString &aStatement)
{
    bool bSuccess(false);
    if (aQuery.prepare(aStatement)) {
        if (aQuery.exec()) {
            bSuccess = true;
        }
    }
    return bSuccess;
}

/*!
    Gets the latest version
    @param aQuery identifier query
    @param aVersion version string
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::getLatestVersion(QSqlQuery &aQuery, ServiceResolutionCriteria &aCriteria)
{
    int result(0);
    //Get latest version from DB
    if (aQuery.prepare(RESOLVERDATABASE_GET_LATEST_INTERFACE_VERSION_BY_NAME)) {
        aQuery.bindValue(EBindIndex, aCriteria.interfaceName());
        aQuery.bindValue(EBindIndex1, aCriteria.serviceName());
        if (!aQuery.exec()) {
            result = aQuery.lastError().number();
        }
        else if (aQuery.next()) {
            aCriteria.setInterfaceVersion(aQuery.value(EBindIndex).toString());
        }
    }
    else {
        result = aQuery.lastError().number();
    }
    return result;
}

/*!
    Retrieves service metadata based on search criteria \n
    Custom error codes: \n
    SFW_ERROR_INVALID_SEARCH_CRITERIA in case the search criteria is invalid \n
    SFW_ERROR_DATABASE_NOT_OPEN in case database is not open \n
    SFW_ERROR_SERVICE_NOT_FOUND in case service was not found in the database \n
    SFW_ERROR_INVALID_DATABASE_CONNECTION in case the database connection is not valid \n
    @param aCriteria search criteria
    @param aServiceInfo result of the search, service information
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::getService(ServiceResolutionCriteria &aCriteria,ServiceInfo &aServiceInfo)
{
    int result(0);
    //Retrieve the service file path
    ServiceInfo serviceInfo;

    if (!aCriteria.isValid()) {
        result = ServiceDatabase::SFW_ERROR_INVALID_SEARCH_CRITERIA;
    } else {
        if (!m_IsDatabaseOpen) {
            result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
        } else {
            //Prepare search query, bind criteria values and execute search
            QSqlDatabase database = QSqlDatabase::database();
            if (database.isValid()){
                QSqlQuery query(database);
                if (((aCriteria.interfaceVersion()).compare(QString(KDefaultVersion))) == 0){
                    //get latest version
                    getLatestVersion(query, aCriteria);
                }
                if (!query.prepare(RESOLVERDATABASE_GET_SERVICE_INTERFACE_INFO)) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                } else {
                    query.bindValue(EBindIndex,aCriteria.serviceName());
                    int majorInterfaceVersion = aCriteria.interfaceVersion().split('.').at(0).toInt();
                    query.bindValue(EBindIndex1,aCriteria.interfaceVersion());
                    query.bindValue(EBindIndex2,majorInterfaceVersion + 1);
                    query.bindValue(EBindIndex3,aCriteria.interfaceName());
                    if (!query.exec()) {
                        result = query.lastError().number();
                        query.finish();
                        query.clear();
                    } else if (query.next()) {
                        serviceInfo.setServiceName(aCriteria.serviceName());
                        serviceInfo.setServiceVersion(query.value(EBindIndex).toString());
                        serviceInfo.setServiceDllFileName(query.value(EBindIndex1).toString());
                        serviceInfo.setXmlFileName(query.value(EBindIndex2).toString());
                        serviceInfo.setInterfaceVersion(query.value(EBindIndex3).toString());
                        serviceInfo.setInterfaceCapabilities(query.value(EBindIndex4).toString());
                    } else {
                        result = ServiceDatabase::SFW_ERROR_SERVICE_NOT_FOUND;
                    }
                }
                query.finish();
                query.clear();
            } else {
                result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
            }
        }
    }
    if (result == 0) {
        aServiceInfo = serviceInfo;
    }
    return result;
}

/*!
    Retrieves metadata of the registered services \n
    Custom error codes: \n
    SFW_ERROR_DATABASE_NOT_OPEN in case database is not open \n
    SFW_ERROR_INVALID_DATABASE_CONNECTION in case database connection is invalid \n
    @param aServiceInfoList result of the search, list of services
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::listServices(ServiceInfoList &aServiceInfoList)
{
    int result(0);
    ServiceInfoList serviceInfoList;

    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        //Prepare and execute list query
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            if (!query.prepare(RESOLVERDATABASE_GET_SERVICES)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
            } else {
                if (!query.exec()) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                } else {
                    //Get result values
                    while (query.next()) {
                        ServiceInfo *aService = new ServiceInfo(query.value(EBindIndex).toString(), query.value(EBindIndex1).toString(),
                            query.value(EBindIndex2).toString(),"",query.value(EBindIndex3).toString());
                        serviceInfoList.addServiceInfo(*aService);
                        delete(aService);
                    }
                }
                query.finish();
                query.clear();
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    if (result == 0) {
        aServiceInfoList = serviceInfoList;
    }
    return result;
}

/*!
    Closes the service resolver database
    Custom error codes: \n
    SFW_ERROR_CANNONT_CLOSE_DATABASE in case database can't be closed \n
    SFW_ERROR_INVALID_DATABASE_CONNECTION in case database connection is invalid \n
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::close()
{
    int result(0);
    if (m_IsDatabaseOpen) {
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            if (database.isOpen()) {
                database.close();
                m_IsDatabaseOpen = database.isOpen();
                result = !m_IsDatabaseOpen? 0: ServiceDatabase::SFW_ERROR_CANNONT_CLOSE_DATABASE;

            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}

/*!
    Sets the path of the service resolver database
    @return void
*/
void ServiceDatabase::setDatabasePath(const QString &aDatabasePath)
{
    m_DatabasePath = aDatabasePath;
}

/*!
    Creates the database tables: Service, Interface
    @return 0 if operation is successful, error code otherwise
*/
int ServiceDatabase::createTables()
{
    int result(0);
    //Execute transaction for creating database tables
    bool isActive(false);
    QSqlDatabase database = QSqlDatabase::database();
    isActive = database.transaction();

    QSqlQuery query(database);
    if (!executeQuery(query,RESOLVERDATABASE_CREATE_SERVICE_TABLE)) {
        result = query.lastError().number();
        query.finish();
        query.clear();
        if (isActive) {
            database.rollback();
        }
    } else {
        if (!executeQuery(query,RESOLVERDATABASE_CREATE_INTERFACE_TABLE)) {
            result = query.lastError().number();
            query.finish();
            query.clear();
            if (isActive) {
                database.rollback();
            }
        } else {
#ifdef __SYMBIAN32__
            if (!executeQuery(query,RESOLVERDATABASE_CREATE_ROMINFO_TABLE)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
                if (isActive) {
                    database.rollback();
                }

            } else {
#endif
                query.finish();
                query.clear();
                if (isActive) {
                    database.commit();
                }
#ifdef __SYMBIAN32__
            }
#endif
        }
    }
    if (result == 0) {
        result = ServiceDatabase::SFW_ERROR_DB_RECREATED;
    }
    return result;
}

/*!
    Checks if the service related tables exist in the database
    @return true if tables are present, false otherwise
*/
bool ServiceDatabase::checkTables()
{
    bool bTables(false);
    QStringList tables = QSqlDatabase::database().tables();
    if (tables.contains(RESOLVERDATABASE_SERVICE)
        && tables.contains(RESOLVERDATABASE_INTERFACE)){
#ifdef __SYMBIAN32__
        if (tables.contains(RESOLVERDATABASE_ROMINFO)) {
#endif
            //check service table's fields
            bTables = true;
            QSqlRecord currentFields = QSqlDatabase::database().record(RESOLVERDATABASE_SERVICE);
            int i = 0;
            while (i < m_ServiceFields.count()) {
                if (!currentFields.contains(m_ServiceFields[i])) {
                    bTables = false;
                    break;
                }
                i++;
            }
            //check number of fields in existing service table
            if (bTables && i != currentFields.count()) {
                bTables = false;
            }

            //check interface table's fields
            if (bTables) {
                currentFields.clear();
                currentFields = QSqlDatabase::database().record(RESOLVERDATABASE_INTERFACE);
                i = 0;
                while (i < m_InterfaceFields.count()) {
                    if (!currentFields.contains(m_InterfaceFields[i])) {
                        bTables = false;
                        break;
                    }
                    i++;
                }
                //check number of fields in existing interface table
                if (bTables && i  != currentFields.count()) {
                    bTables = false;
                }
            }

#ifdef __SYMBIAN32__
            //check rominfo table fields
            if (bTables) {
                currentFields.clear();
                currentFields = QSqlDatabase::database().record(RESOLVERDATABASE_ROMINFO);
                i = 0;
                while (i < m_RomInfoFields.count()) {
                    if (!currentFields.contains(m_RomInfoFields[i])) {
                        bTables = false;
                        break;
                    }
                    i++;
                }
                //check number of fields in existing rominfo table
                if (bTables && i  != currentFields.count()) {
                    bTables = false;
                }
            }
        }
#endif
    }
    return bTables;
}

/*!
    Removes service related tables from database
    @return 0 if was sucessful, error code otherwise
*/
int ServiceDatabase::dropTables()
{
    int result(0);
    //Execute transaction for deleting the database tables
    bool isActive(false);
    QSqlDatabase database = QSqlDatabase::database();
    QStringList tables = database.tables();
    if (tables.size() > 0) {
        isActive = database.transaction();
        QSqlQuery query(database);

        if ((tables.contains(RESOLVERDATABASE_SERVICE))
            && (!executeQuery(query,RESOLVERDATABASE_DROP_SERVICE_TABLE))) {
                result = query.lastError().number();
                query.finish();
                query.clear();
                if (isActive) {
                    database.rollback();
                }
        }
        if ((result == 0) && (tables.contains(RESOLVERDATABASE_INTERFACE))
            && (!executeQuery(query,RESOLVERDATABASE_DROP_INTERFACE_TABLE))) {
                result = query.lastError().number();
                query.finish();
                query.clear();
                if (isActive) {
                    database.rollback();
                }
        }
#ifdef __SYMBIAN32__
        if ((result == 0) && (tables.contains(RESOLVERDATABASE_ROMINFO))
            && (!executeQuery(query,RESOLVERDATABASE_DROP_ROMINFO_TABLE))) {
                result = query.lastError().number();
                query.finish();
                query.clear();
                if (isActive) {
                    database.rollback();
                }
        }
#endif
        if (result == 0) {
            query.finish();
            query.clear();
            if (isActive) {
                database.commit();
            }
        }
    }
    return result;
}

/*!
    Checks if the resolver database is open
    @return true if database is open, false otherwise
*/
bool ServiceDatabase::isOpen()
{
  return m_IsDatabaseOpen;
}


/*!
    Gets the service version
    @param aServiceName service names
    @param aServiceVersion servcie version string
    @param aFilePath file path of the service
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::getServiceVersion(const QString &aServiceName,
                                       QString &aServiceVersion,
                                       QString &aFilePath)
{
    int result(0);
    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        //Prepare and execute list query
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            if (!query.prepare(RESOLVERDATABASE_GET_SERVICE)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
            } else {
                query.bindValue(EBindIndex, aServiceName);
                if (!query.exec()) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                } else if (query.next()) {
                    aServiceVersion = query.value(EBindIndex2).toString();
                    aFilePath = query.value(EBindIndex1).toString();
                }
                query.finish();
                query.clear();
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;

}

#ifdef __SYMBIAN32__
/*!
    Gets the ROM image version
    @param aRomImageVersion ROM image version string
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::getRomImageVersion(QString &aRomImageVersion)
{
    int result(0);
    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        //Prepare and execute query
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            if (!query.prepare(RESOLVERDATABASE_GET_ROMIMAGE_VERSION)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
            } else {
                if (!query.exec()) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                } else if (query.next()) {
                    aRomImageVersion = query.value(EBindIndex).toString();
                }
                query.finish();
                query.clear();
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}

/*!
    Update the ROM image version in DB
    @param aRomImageVersion ROM image version string
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::updateRomImageVersion(const QString &aRomImageVersion)
{
    int result(0);
    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        //Prepare and execute list query
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            if (!query.prepare(RESOLVERDATABASE_UPDATE_ROMINFO_RECORD)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
            } else {
                query.bindValue(EBindIndex, aRomImageVersion);
                if (!query.exec()) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                }
                query.finish();
                query.clear();
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}

/*!
    Insert ROM image version in DB
    @param aRomImageVersion ROM image version string
    @return 0 if operation was successful, error code otherwise
*/
int ServiceDatabase::insertRomImageVersion(const QString &aRomImageVersion)
{
    int result(0);
    if (!m_IsDatabaseOpen) {
        result = ServiceDatabase::SFW_ERROR_DATABASE_NOT_OPEN;
    } else {
        //Prepare and execute list query
        QSqlDatabase database = QSqlDatabase::database();
        if (database.isValid()){
            QSqlQuery query(database);
            if (!query.prepare(RESOLVERDATABASE_INSERT_ROMINFO_RECORD)) {
                result = query.lastError().number();
                query.finish();
                query.clear();
            } else {
                query.bindValue(EBindIndex, aRomImageVersion);
                if (!query.exec()) {
                    result = query.lastError().number();
                    query.finish();
                    query.clear();
                }
                query.finish();
                query.clear();
            }
        } else {
            result = ServiceDatabase::SFW_ERROR_INVALID_DATABASE_CONNECTION;
        }
    }
    return result;
}
#endif

}
