/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "npdef.h"

#ifndef HIBYTE
#define HIBYTE(x) ((((uint32)(x)) & 0xff00) >> 8)
#endif

#ifndef LOBYTE
#define LOBYTE(W) ((W) & 0xFF)
#endif

// The global object which maintains the pointers to the browser's functions
//
NPNetscapeFuncs* NPNFuncs = NULL;

/*!
    Returns version information for the Plug-in API.

    @param plugin_major Pointer to a plug-in's major version number; changes with
                        major code release number.
    @param plugin_minor Pointer to a plug-in's minor version number; changes with
                        point release number.
    @param netscape_major Pointer to the browser's major version; changes with
                          major code release number.
    @param netscape_minor Pointer to the browser's version; changes with point
                          release number.
*/
void NPN_Version(int* plugin_major, int* plugin_minor,
                 int* netscape_major, int* netscape_minor)
{
    *plugin_major   = NP_VERSION_MAJOR;
    *plugin_minor   = NP_VERSION_MINOR;
    *netscape_major = HIBYTE(NPNFuncs->version);
    *netscape_minor = LOBYTE(NPNFuncs->version);
}

/*!
    Asks the browser to create a stream for the specified URL

    @param instance Pointer to the plug-in instance to delete
    @param url Pointer to the URL of the request. Can be of any type,
               such as HTTP, FTP, news, mailto, gopher
    @param target Name of the target window or frame, or one of the following
                  special target names. Values:
                  _blank or _new: Load the link in a new blank unnamed window.
                                  Safest target, even though, when used with a mailto
                                  or news URL, this creates an extra blank the
                                  browser instance.
                 _self or _current: Load the link into the same window the plug-in
                                    instance occupies. Not recommended; see Warning.
                                    If target refers to the window or frame containing
                                    the instance, the instance is destroyed and
                                    the plug-in may be unloaded. Use with NPN_GetURL only
                                    if you want to terminate the plug-in.
                 _parent: Load the link into the immediate FRAMESET parent of the
                         plug-in instance's document. If the plug-in instance's
                         document has no parent, the default is _self.
                 _top: Load the link into the plug-in instance window. The default is _self,
                      if the plug-in instance's document is already at the top.
                      Use for breaking out of a deep frame nesting.
                 If null, the browser creates a new stream and delivers the data
                          to the current instance regardless of the MIME type of the URL.
                          In general, if a URL works in the location box of the Navigator,
                          it works here, except for the _self target
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NP_LOADDS NPN_GetURL(NPP instance, const TDesC& url, const TDesC* target)
#else
NPError NPN_GetURL(NPP instance, const char *url, const char *target)
#endif
{
    return NPNFuncs->geturl(instance, url, target);
}

/*!
    Posts data to a URL

    @param instance Pointer to the plug-in instance to delete
    @param url URL of the request, specified by the plug-in
    @param target Display target, specified by the plug-in. If null, pass the new
                  stream back to the current plug-in instance regardless of MIME type.
                  For values, @see NPN_GetURL.
    @param len Length of the buffer buf.
    @param buf Path to local temporary file or data buffer that contains the data to post.
               Temporary file is deleted after use. Data in buffer cannot be posted
               for a protocol that requires a header
    @param file A boolean value that specifies whether to post a file. Values:
                true: Post the file whose the path is specified in buf, then delete the file.
                false: Post the raw data in buf
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NP_LOADDS NPN_PostURL(NPP instance, const TDesC& url,
                              const TDesC* target,
                              const TDesC& buf, NPBool file)
#else
NPError NPN_PostURL(NPP instance, const char* url, const char* window, uint32 len,
                    const char* buf, NPBool file)
#endif // __SYMBIAN32__
{
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
    NPError rv = NPNFuncs->posturl(instance, url, target, buf, file );
#else
    NPError rv = NPNFuncs->posturl(instance, url, window, len, buf, file);
#endif
    return rv;
}

/*!
    Requests a range of bytes for a seekable stream

    @param instance Pointer to the plug-in instance to delete
    @param ranglist Range of bytes in the form of a linked list of NPByteRange objects,
                    each of which specifies a request for a range of bytes
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
NPError NPN_RequestRead(NPStream* stream, NPByteRange* rangeList)
{
    return NPNFuncs->requestread(stream, rangeList);
}

/*!
    Requests the creation of a new data stream produced by the plug-in and consumed
    by the browser

    @param instance Pointer to the current plug-in instance
    @param type Pointer to MIME type of the stream.
    @param target Name of the target window or frame, or one of several special
                  target names. For values, @see NPN_GetURL
    @param stream Stream to be created by the browser.
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NP_LOADDS NPN_NewStream (NPP instance, NPMIMEType type,
                                 const TDesC* target, NPStream** stream)
#else
NPError NPN_NewStream(NPP instance, NPMIMEType type,
                      const char* target, NPStream** stream)
#endif
{
    int navMinorVersion = NPNFuncs->version & 0xFF;

    NPError rv = NPERR_NO_ERROR;

    if ( navMinorVersion >= NPVERS_HAS_STREAMOUTPUT ) {
        rv = NPNFuncs->newstream(instance, type, target, stream);
    }
    else {
        rv = NPERR_INCOMPATIBLE_VERSION_ERROR;
    }

    return rv;
}

/*!
    Pushes data into a stream produced by the plug-in and consumed by the browser

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @param len Length in bytes of buf
    @param buffer Buffer of data delivered for the stream.
    @return If successful, the function returns the number of bytes consumed by the browser
            If unsuccessful, the plug-in returns a negative integer.the plug-in
                             should terminate the stream by calling NPN_DestroyStream
*/
int32 NPN_Write(NPP instance, NPStream *stream, int32 len, void *buffer)
{
    int navMinorVersion = NPNFuncs->version & 0xFF;
    int32 rv = -1;

    if ( navMinorVersion >= NPVERS_HAS_STREAMOUTPUT ) {
        rv = NPNFuncs->write(instance, stream, len, buffer);
    }
    return rv;
}

/*!
    Closes and deletes a stream.

    @param instance Pointer to the current plug-in instance
    @param stream Pointer to current stream.
    @param reason Reason the stream was stopped so the application can give the
                  user appropriate feedback. Values:
                  NPRES_DONE (most common): Stream completed normally; all data was
                                            sent by the plug-in to the browser.
                  NPRES_USER_BREAK: Plug-in is terminating the stream due to a user request.
                  NPRES_NETWORK_ERR: Stream failed due to network problems

    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the plug-in is not loaded and the function returns a
                             NPError error code
*/
NPError NPN_DestroyStream(NPP instance, NPStream* stream, NPError reason)
{
    int navMinorVersion = NPNFuncs->version & 0xFF;
    NPError rv = NPERR_NO_ERROR;

    if ( navMinorVersion >= NPVERS_HAS_STREAMOUTPUT ) {
        rv = NPNFuncs->destroystream(instance, stream, reason);
    }
    else {
        rv = NPERR_INCOMPATIBLE_VERSION_ERROR;
    }

    return rv;
}

/*!
    Displays a message on the status line of the browser window.

    @param instance Pointer to the current plug-in instance
    @param message Pointer the buffer that contains the status message string to display.
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
void NPN_Status(NPP instance, const TDesC& message)
{
    NPNFuncs->status(instance, message);
}
#else
void NPN_Status(NPP instance, const char *message)
{
    NPNFuncs->status(instance, message);
}
#endif //__SYMBIAN32__

/*!
    Returns the browser's user agent field

    The user agent is the part of the HTTP header that identifies the browser during
    transfers. You can use this information to verify that the expected browser is
    in use, or you can use it in combination with NPN_Version to supply different
    code for different versions of Netscape browsers

    @param instance Pointer to the current plug-in instance
    @return A pointer to a buffer that contains the user agent field of the browser
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
const TDesC* NP_LOADDS  NPN_UserAgent(NPP instance)
{
    return NPNFuncs->uagent(instance);
}
#else
const char* NPN_UserAgent(NPP instance)
{
    return NPNFuncs->uagent(instance);
}
#endif // __SYMBIAN32__

/*!
    Allocates memory from the browser's memory space.

    @size Size of memory, in bytes, to allocate in the browser's memory space
    @return If successful, the function returns a pointer to the allocated memory, in bytes
            If insufficient memory is available, the plug-in returns null
*/
void* NPN_MemAlloc(uint32 size)
{
    return NPNFuncs->memalloc(size);
}

/*!
    Deallocates a block of allocated memory.

    @ptr Block of memory previously allocated using NPN_MemAlloc
*/
void NPN_MemFree(void* ptr)
{
    NPNFuncs->memfree(ptr);
}

/*!
    Requests that the browser free a specified amount of memory

    @size Size of memory, in bytes, to free in the browser's memory space.
    @return If successful, the function returns the amount of freed memory, in bytes
            If no memory can be freed, the plug-in returns 0
*/
uint32 NPN_MemFlush(uint32 size)
{
    return (uint32)NPNFuncs->memflush(size);
}

/*!
    Reloads all plug-ins in the Plugins directory.

    @reloadPages Whether to reload pages. Values:
                 true: Reload pages.
                 false: Do not reload pages
*/
void NPN_ReloadPlugins(NPBool reloadPages)
{
    NPNFuncs->reloadplugins(reloadPages);
}

/*!
    Get java runtime environment

    @return Java runtime environment.
*/
JRIEnv* NPN_GetJavaEnv(void)
{
    return NPNFuncs->getJavaEnv();
}

/*!
    Get java runtime peer

    @return Java runtime peer
*/
jref NPN_GetJavaPeer(NPP instance)
{
    return NPNFuncs->getJavaPeer(instance);
}

/*!
    Requests creation of a new stream with the contents of the specified URL;
    gets notification of the result.

    @param instance Pointer to the current plug-in instance
    @param url Pointer to the URL of the request
    @param target Name of the target window or frame, or one of several special
                  target names. For values, see @NPN_GetURL
    @param notifData Plug-in-private value for associating the request with the
                     subsequent NPP_URLNotify call
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NP_LOADDS NPN_GetURLNotify(NPP instance, const TDesC& url,
                                   const TDesC* target, void* notifyData)
#else
NPError NPN_GetURLNotify(NPP instance, const char *url,
                         const char *target, void* notifyData)
#endif // __SYMBIAN32__
{
    int navMinorVers = NPNFuncs->version & 0xFF;
    NPError rv = NPERR_NO_ERROR;

    if ( navMinorVers >= NPVERS_HAS_NOTIFICATION ) {
        rv = NPNFuncs->geturlnotify(instance, url, target, notifyData);
    }
    else {
        rv = NPERR_INCOMPATIBLE_VERSION_ERROR;
    }

    return rv;
}

/*!
    Posts data to a URL, and receives notification of the result.

    @param instance Pointer to the current plug-in instance
    @param url Pointer to the URL of the request
    @param window Target window, specified by the plug-in. For values, @see NPN_GetURL.
    @param len Length of the buffer buf.
    @param buf Path to local temporary file or data buffer that contains the data to post
    @param file Whether to post a file
    @return If successful, the function returns NPERR_NO_ERROR
            If unsuccessful, the function returns a NPError error code
*/
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
NPError NP_LOADDS NPN_PostURLNotify(NPP instance, const TDesC& url,
                                    const TDesC* window,
                                    const TDesC& buf, NPBool file,
                                    void* notifyData)
#else
NPError NPN_PostURLNotify(NPP instance, const char* url, const char* window,
                          uint32 len, const char* buf, NPBool file, void* notifyData)
#endif // __SYMBIAN32__
    {
    int navMinorVers = NPNFuncs->version & 0xFF;
    NPError rv = NPERR_NO_ERROR;

    if ( navMinorVers >= NPVERS_HAS_NOTIFICATION ) {
#if defined(__SYMBIAN32__) && !defined(CWRT_S60_WEBKIT)
        rv = NPNFuncs->posturlnotify(instance, url, window, buf, file, notifyData );
#else
        rv = NPNFuncs->posturlnotify(instance, url, window, len, buf, file, notifyData);
#endif // __SYMBIAN32__
    }
    else {
        rv = NPERR_INCOMPATIBLE_VERSION_ERROR;
    }

    return rv;
}

/*!
    Allows the browser to query the plug-in for information.

    @param instance Pointer to the current plug-in instance
    @param variable Plug-in information the call gets.
    @param value Value of the variable
    @return If successful, the function returns the function returns NPERR_NO_ERROR.
            If unsuccessful, the function returns a NPErr error code.
*/
NPError NPN_GetValue(NPP instance, NPNVariable variable, void *value)
{
    return NPNFuncs->getvalue(instance, variable, value);
}

/*!
    Sets various modes of plug-in operation.

    @param instance Pointer to the current plug-in instance
    @param variable Variable to be set
    @param value Value of the variable
    @return If successful, the function returns the function returns NPERR_NO_ERROR.
            If unsuccessful, the function returns a NPErr error code.
*/
NPError NPN_SetValue(NPP instance, NPPVariable variable, void *value)
{
    return NPNFuncs->setvalue(instance, variable, value);
}

/*!
    Invalidates specified drawing area prior to repainting or refreshing a
    windowless plug-in.

    @param instance Pointer to the current plug-in instance
    @param invalidRect The area to invalidate, specified in a coordinate system
           that originates at the top left of the plug-in.
*/
void NPN_InvalidateRect(NPP instance, NPRect *invalidRect)
{
    NPNFuncs->invalidaterect(instance, invalidRect);
}

/*!
    Invalidates specified drawing region prior to repainting or refreshing a
    windowless plug-in

    @param instance Pointer to the current plug-in instance
    @param invalidRegion The area to invalidate, specified in a coordinate system
                         that originates at the top left of the plug-in
*/
void NPN_InvalidateRegion(NPP instance, NPRegion invalidRegion)
{
    NPNFuncs->invalidateregion(instance, invalidRegion);
}

/*!
    Forces a paint message for a windowless plug-in.

    @param instance Pointer to the current plug-in instance
*/
void NPN_ForceRedraw(NPP instance)
{
    NPNFuncs->forceredraw(instance);
}

// The remaining functions are NOT supported in Symbian Avkon browser,
// they are supported in Symbian CWRT browser
#if !defined(__SYMBIAN32__) || defined(CWRT_S60_WEBKIT)

/*!
    Returns an opaque identifier for the string that is passed in.

    @param name The string for which an opaque identifier should be returned
    @return An NPIdentifier identifying the string
*/
NPIdentifier NPN_GetStringIdentifier(const NPUTF8 *name)
{
    return NPNFuncs->getstringidentifier(name);
}

/*!
    Returns an array of opaque identifiers for the names that are passed in

    @param names An array of strings for which opaque identifiers should be returned
    @param nameCount The number of strings in the names array
    @param identifiers A pointer to an array to fill with the identifiers for
                       the strings specified by names.
*/
void NPN_GetStringIdentifiers(const NPUTF8 **names, uint32_t nameCount,
                              NPIdentifier *identifiers)
{
    return NPNFuncs->getstringidentifiers(names, nameCount, identifiers);
}

/*!
    Returns an opaque identifier for the integer that is passed in.

    @param intid The integer for which an opaque identifier should be returned
    @return An NPIdentifier identifying the integer
*/
NPIdentifier NPN_GetIntIdentifier(int32_t intid)
{
    return NPNFuncs->getintidentifier(intid);
}

/*!
    Determines whether or not an identifier is a string.

    @param identifier The identifier whose type is to be examined
    @return true if the identifier is a string identifier, or false otherwise
*/
bool NPN_IdentifierIsString(NPIdentifier identifier)
{
    return NPNFuncs->identifierisstring(identifier);
}

/*!
    Returns the UTF-8 string corresponding to the given string identifier.

    @param identifier The string identifier whose corresponding string should be returned
    @return A UTF-8 string as a sequence of NPUTF8 bytes, or NULL if the specified
            identifier isn't a string identifier.
*/
NPUTF8 *NPN_UTF8FromIdentifier(NPIdentifier identifier)
{
    return NPNFuncs->utf8fromidentifier(identifier);
}

/*!
    Returns the integer value corresponding to the given integer identifier.

    @param identifier The integer identifier whose corresponding integer value should be returned
    @return An integer corresponding to the specified identifier
*/
int32_t NPN_IntFromIdentifier(NPIdentifier identifier)
{
    return NPNFuncs->intfromidentifier(identifier);
}

/*!
    Allocates a new NPObject.

    @param npp The NPP indicating which plugin wants to instantiate the object
    @param aClass The class to instantiate an object of.
    @return The newly-allocated NPObject
*/
NPObject *NPN_CreateObject(NPP npp, NPClass *aClass)
{
    return NPNFuncs->createobject(npp, aClass);
}

/*!
    Increments the reference count of the given NPObject.

    @param obj The NPObject to retain
    @return A pointer back to the same object.
*/
NPObject *NPN_RetainObject(NPObject *obj)
{
    return NPNFuncs->retainobject(obj);
}

/*!
    Decrements the reference count of the given NPObject. If the reference count
    reaches 0, the NPObject is deallocated by calling its deallocate function
    if one is provided; if one is not provided, free() is used.

    @param obj The NPObject whose reference count should be decremented
*/
void NPN_ReleaseObject(NPObject *obj)
{
    return NPNFuncs->releaseobject(obj);
}

/*!
    Invokes a method on the given NPObject.

    @param npp The NPP indicating which plugin wants to call the method on the object
    @param obj The object on which to invoke a method
    @param methodName A string identifier indicating the name of the name of the method to invoke
    @param args An array of arguments to pass to the method
    @param argCount The number of arguments in the args array
    @param result A pointer to an NPVariant to receive the result returned by the method
                 When the caller no longer needs the result, it must call
                 NPN_ReleaseVariantValue() to release it.
    @return true if the method was successfully invoked, otherwise false. If the
            method was invoked successfully, any return value is stored in the
            NPVariant specified by result
*/
bool NPN_Invoke(NPP npp, NPObject* obj, NPIdentifier methodName,
                const NPVariant *args, uint32_t argCount, NPVariant *result)
{
    return NPNFuncs->invoke(npp, obj, methodName, args, argCount, result);
}

/*!
    Invokes the default method, if one exists, on the given NPObject

    @param npp The NPP indicating which plugin wants to call the method on the object
    @param obj The object on which to invoke the default method
    @param args An array of arguments to pass to the method
    @param argCount The number of arguments in the args array
    @param result A pointer to an NPVariant to receive the result returned by the method
                 When the caller no longer needs the result, it must call
                 NPN_ReleaseVariantValue() to release it.
    @return true if the method was successfully invoked, otherwise false. If the
            method was invoked successfully, any return value is stored in the
            NPVariant specified by result
*/
bool NPN_InvokeDefault(NPP npp, NPObject* obj, const NPVariant *args,
                       uint32_t argCount, NPVariant *result)
{
    return NPNFuncs->invokeDefault(npp, obj, args, argCount, result);
}

/*!
    Invokes a method on the given NPObject.

    @param npp The NPP indicating which plugin instance's window to evaluate the script in.
    @param obj The object on which to evaluate the script
    @param script The script to evaluate
    @param result On return, contains the value returned by the script
                  The caller must call NPN_ReleaseVariantValue() to release the
                  returned value when it's no longer needed
    @return true if the script was evaluated successfully, otherwise false
*/
bool NPN_Evaluate(NPP npp, NPObject* obj, NPString *script,
                  NPVariant *result)
{
    return NPNFuncs->evaluate(npp, obj, script, result);
}

/*!
    Gets the value of a property on the specified NPObject.

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object from which a value is to be retrieved.
    @param propertyName A string identifier indicating the name of the property
                        whose value is to be retrieved
    @param result On return, contains the value of the specified property
                  The caller must call NPN_ReleaseVariantValue() to release the
                  returned value when it's no longer needed
    @return true if the script was evaluated successfully, otherwise false
*/
bool NPN_GetProperty(NPP npp, NPObject* obj, NPIdentifier propertyName,
                     NPVariant *result)
{
    return NPNFuncs->getproperty(npp, obj, propertyName, result);
}

/*!
    Sets the value of a property on the specified NPObject.

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object on which a value is to be set.
    @param propertyName A string identifier indicating the name of the property
                        whose value is to be retrieved
    @param value The value to store in the specified property
    @return true if the value was set successfully, otherwise false
*/
bool NPN_SetProperty(NPP npp, NPObject* obj, NPIdentifier propertyName,
                     const NPVariant *value)
{
    return NPNFuncs->setproperty(npp, obj, propertyName, value);
}

/*!
    Removes a property from the specified NPObject

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object on which a property is to be deleted
    @param propertyName A string identifier indicating the name of the property to remove
    @return true if the property was removed successfully, otherwise false
*/
bool NPN_RemoveProperty(NPP npp, NPObject* obj, NPIdentifier propertyName)
{
    return NPNFuncs->removeproperty(npp, obj, propertyName);
}

/*!
    Determines whether or not the specified NPObject has a particular property

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object on which to look for the property
    @param propertyName A string identifier indicating the name of the property to look for
    @return true if the property exists on the specified object, otherwise false
*/
bool NPN_HasProperty(NPP npp, NPObject* obj, NPIdentifier propertyName)
{
    return NPNFuncs->hasproperty(npp, obj, propertyName);
}

/*!
    Determines whether or not the specified NPObject has a particular method

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object on which to look for the method
    @param methodName A string identifier indicating the name of the method to look for
    @return true if the method exists on the specified object, otherwise false.
*/
bool NPN_HasMethod(NPP npp, NPObject* obj, NPIdentifier methodName)
{
    return NPNFuncs->hasmethod(npp, obj, methodName);
}

/*!
    NPN_ReleaseVariantValue() releases the value in the given variant

    This must always be called on result variants and such in this API, i.e.
    any NPVariant whose value comes from a call that passes back an NPVariant
    must be released using this function. Access to the value in an NPVariant
    that has been released will result in undefined behavior

    @param variant The variant whose value is to be released
*/
void NPN_ReleaseVariantValue(NPVariant *variant)
{
    NPNFuncs->releasevariantvalue(variant);
}

/*!
    A plugin can call this function to indicate that a call to one of the
    plugin's NPObjects generated an error

    @param obj The object on which the exception occurred
    @param messge A UTF-8 string describing the error that occurred
*/
void NPN_SetException(NPObject* obj, const NPUTF8 *message)
{
    NPNFuncs->setexception(obj, message);
}

/*!
    Push popup enabled state

    @param npp The NPP indicating which plugin instance's is making the request
    @param enabled true enable popup, false disable popup
*/
void NPN_PushPopupsEnabledState(NPP npp, NPBool enabled)
{
    NPNFuncs->pushpopupsenabledstate(npp, enabled);
}

/*!
    Pop popup enabled state

    @param npp The NPP indicating which plugin instance's is making the request
*/
void NPN_PopPopupsEnabledState(NPP npp)
{
    NPNFuncs->poppopupsenabledstate(npp);
}

/*!
    List all of identifiers in a NPObject

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj The object on which to look for the indeifiers
    @param identifier array of identifiers returned
    @param number of indentifiers
    @return true if the method succeeded, otherwise false.
*/
bool NPN_Enumerate(NPP npp, NPObject *obj, NPIdentifier **identifier, uint32_t *count)
{
    return NPNFuncs->enumerate(npp, obj, identifier, count);
}

/*!
    Thread safe way to request that the browser calls a plugin function on the
    browser or plugin thread (the thread on which the plugin was initiated

    @param npp The NPP indicating which plugin instance's is making the request
    @param func Pointer to the function to call on the correct thread
    @param userData Pointer to the data to pass to the function func once called
*/
void NPN_PluginThreadAsyncCall(NPP instance,void(*func)(void*),void* userData)
{
    NPNFuncs->pluginthreadasynccall(instance,func,userData);
}

/*!
    Create a NPObject

    @param npp The NPP indicating which plugin instance's is making the request
    @param obj an input NPObject
    @param args An array of arguments to pass to the method
    @param argCount number of arguments
    @param result new created NPObject
    @return true if the method succeeded, otherwise false.
*/
bool NPN_Construct(NPP npp, NPObject *obj, const NPVariant *args,
                   uint32_t argCount, NPVariant *result)
{
    return NPNFuncs->construct(npp, obj, args, argCount, result);
}
#endif // __SYMBIAN32__
