/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <QMetaObject>
#include <QMetaClassInfo>
#include <QVector>
#include <QMap>

#ifdef __SYMBIAN32__
#include <string.h>
#endif // __SYMBIAN32__

#include "bindingutility.h"
#include "objectbinding.h"
#include "signalforwarder.h"

namespace WRT
{
/*!
    \class SignalForwarder
    \brief The ObjectBinding class is used to help connect callback to a signal

   It enables script language like JavaScript to conect a callback function to
   the signal in the service object
*/

/*!
    Constructs an object of the SignalForwarder.

    @param npp represents a single instance of a plugin
    @param npObject a NPObject instance
    @param callback callback NPObject
    @param parameterList the list of parameter of the callback
    @param bindingUtility binding utility object
    @param connectionId connection id
*/
SignalForwarder::SignalForwarder(NPP& npp,
                                 NPObject* npObject,
                                 NPObject* callback,
                                 QList<QByteArray>& parameterList,
                                 BindingUtility& bindingUtility,
                                 int connectionId) :
m_npp(npp),
m_cbObject(NPN_RetainObject(npObject)),
m_function(NULL),
m_argNames(parameterList),
m_bindingUtility(bindingUtility),
m_id(connectionId),
m_state(DisConnected)
{
    if ( callback ) {
        m_function = NPN_RetainObject(callback);
    }

    // Setup identifiers for connecting/disconnecting signals
    //
    m_npidCall = NPN_GetStringIdentifier("call");
}

/*!
    Destructor
*/
SignalForwarder::~SignalForwarder()
{
    if (m_cbObject) {
        NPN_ReleaseObject(m_cbObject);
    }
    if (m_function) {
        NPN_ReleaseObject(m_function);
    }
}

/*!
    It is called by qt framework when service object emits a signal.
    It calls back JS callback function saved in the data member

    @param callType QMetaObject::InvokeMetaMethod or others
    @param index dummy index, not used
    @param args arguments of the callback
    @return -1 for done, otherwise internal code
*/

int SignalForwarder::qt_metacall(QMetaObject::Call callType, int index, void **args)
{
    int ret(0);
    m_state = InProgress;
    // Let base object process the meta call first
    int id = QObject::qt_metacall(callType, index, args);
    if (id < 0 || callType != QMetaObject::InvokeMetaMethod) {
        ret = id;
    }
    else {
        //JS callback:
        //1. convert arguments
        QVector<NPVariant> npvArgs;

        if ( m_function ) {
            //add saved object as a first parameter
            NPVariant npobj;
            // convert NPObject to NPVARIANT for callback to JS
            OBJECT_TO_NPVARIANT(m_cbObject,npobj);
            // Object is needed only in the "connect" case
            npvArgs += npobj;
        }

        //add passed arguments
        int n = m_argNames.count();
        QList<NPObject*> objList;
        //Copy signal arguments into JS parameters
        for (int i=0; i<n; ++i) {
            int type = QMetaType::type(m_argNames.at(i).constData());
            if (QMetaType::QObjectStar == type) {
                int* objaddr = reinterpret_cast<int*>(args[i + 1]);
                QObject* object = reinterpret_cast<QObject*>(*objaddr);
                NPVariant variant;
                VOID_TO_NPVARIANT(variant);
                if (object) {
                    NPObject *pObj = NPN_CreateObject(m_npp, GET_NPOBJECT_CLASS(ObjectBinding));
                    if (!pObj) {
                        delete object;
                    }
                    else {
                        objList.append(pObj);
                        static_cast<ObjectBinding*>(pObj)->setObject(
                                            object, m_bindingUtility, true);
                        OBJECT_TO_NPVARIANT(pObj, variant);
                    }
                } else {
                    NULL_TO_NPVARIANT(variant);
                }
                npvArgs += variant;
            }
            else if (m_argNames[i] == "QVariant") {
                // QVariant
                QVariant* qvar = reinterpret_cast<QVariant*>((args[i + 1]));
                if (qvar->isValid()) {
                    NPVariant npvar = m_bindingUtility.fromQVariant(*qvar);
                    npvArgs += npvar;
                } else {
                    NPVariant npvar = m_bindingUtility.fromQVariant(QVariant());
                    npvArgs += npvar;
                }
            }
            else
            {
                QVariant qvar(type, args[i + 1]); //args[0] is for return value
                NPVariant npvar = m_bindingUtility.fromQVariant(qvar);
                npvArgs += npvar;
            } // TODO for void, etc.
        } // end for

        //2. get call function
        NPVariant npvRetVal;
        VOID_TO_NPVARIANT(npvRetVal);
        //in future, implementation should be more robust and allow "function only" callback too
        if ( m_cbObject && m_function ) {
            if (NPN_HasMethod(m_npp,m_function,m_npidCall)) {
                    // call JS API "call"
                    NPN_Invoke(m_npp, m_function, m_npidCall, npvArgs.constData(),
                               npvArgs.count(), &npvRetVal);
             }
        }
        else if ( m_cbObject ) {// other signal
            NPN_InvokeDefault(m_npp, m_cbObject, npvArgs.constData(), npvArgs.count(), &npvRetVal );
        }
        BindingUtility::freeNPVariant(npvRetVal);
        // free objects parameter
        NPObject* theObject;
        foreach (theObject, objList) {
            NPN_ReleaseObject(theObject);
        }
        objList.clear();
        ret = -1; // done
    }
    if (DisConnected == m_state) {
        delete this;
    } else {
        m_state = Completed;
    }
    return ret;
}

/*!
    Return connection id

    @return identifier of the connection
*/
int SignalForwarder::id()
{
    return m_id;
}

/*!
    Return callback state

    @return status of call back
*/
SignalForwarder::CallbackState SignalForwarder::callbackState()
{
    return m_state;
}

/*!
    Set callback state

    @param state callback state
*/
void SignalForwarder::setCallbackState(SignalForwarder::CallbackState state)
{
    m_state = state;
}

} // namespace

// END OF FILE
