/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include <qglobal.h>
#include "scriptableobjectbase.h"

extern NPNetscapeFuncs* NPNFuncs;

namespace WRT
{
/*!
    \class NPScriptableObjectBase
    \brief The NPScriptableObjectBase class is a base class for scriptable object

   It defines static utilities functions for the scriptable object
*/

/*!
    Constructs an object of the NPScriptableObjectBase.

    @param npp represents a single instance of a plugin
*/
NPScriptableObjectBase::NPScriptableObjectBase(NPP npp)
: m_Npp(npp)
{
    // init NPObject
    _class = NULL;
    referenceCount = 0;
}

/*!
    Destructor
*/
NPScriptableObjectBase::~NPScriptableObjectBase()
{
}

/*!
    Invalidate the plugin's drawing area.

*/
void NPScriptableObjectBase::Invalidate()
{
}

/*!
    Check if a method exists in the scriptable object given \a methodId

    Always returns false in this base class, derived class should implement this method.

    @param methodId method identifier
    @return true if the method exists, otherwise false

    @see HasProperty
*/
bool NPScriptableObjectBase::HasMethod(NPIdentifier /*methodId*/)
{
    return false;
}

/*!
    Dispatch a script call to the scriptable object

    Always returns false in this base class, derived class should implement this method.

    @param methodId the identifier of the method
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false
*/
bool NPScriptableObjectBase::Invoke(NPIdentifier /*methodId*/,
                                    const NPVariant* /*args*/,
                                    uint32_t /*argCount*/, NPVariant* /*result*/)
{
    return false;
}

/*!
    Dispatch the default call to the scriptable object

    Always returns false in this base class, derived class should implement this method.

    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false
*/
bool NPScriptableObjectBase::InvokeDefault(const NPVariant* /*args*/,
                                           uint32_t /*argCount*/, NPVariant* /*result*/)
{
    return false;
}

/*!
    Check if a property exists in the scriptable object given \a propertyId

    Always returns false in this base class, derived class should implement this method.

    @param propertyId property identifier
    @return true if the property exists, otherwise false

    @see HasMethod
*/
bool NPScriptableObjectBase::HasProperty(NPIdentifier /*propertyId*/)
{
    return false;
}

/*!
    Get the value of a property of the scriptable object given \a propertyId

    Always returns false in this base class, derived class should implement this method.

    @param propertyId property identifier
    @param[out] result  property value
    @return true if succeed in getting value of the property, otherwise false

    @see HasProperty SetProperty
*/
bool NPScriptableObjectBase::GetProperty(NPIdentifier /*propertyId*/, NPVariant* /*result*/)
{
    return false;
}

/*!
    Set a property of the scriptable object given \a propertyId and \a value

    Always returns false in this base class, derived class should implement this method.

    @param propertyId property identifier
    @param value property value
    @return true if succeed in setting the property, otherwise false

    @see HasProperty GetProperty
*/
bool NPScriptableObjectBase::SetProperty(NPIdentifier /*propertyId*/,
                                         const NPVariant* /*value*/)
{
    return false;
}

/*!
    Remove a property from the scriptable object given \a propertyId

    Always returns false in this base class, derived class should implement this method.

    @param propertyId property identifier
    @return true if succeed, otherwise false

    @see SetProperty
*/
bool NPScriptableObjectBase::RemoveProperty(NPIdentifier /*propertyId*/)
{
    return false;
}

/*!
    List of all identifiers for properties and methods, defined in the scriptable object given

    Always returns false in this base class, derived class should implement this method.

    @param[out] identifier pointer to the array of identifiers
    @param[out] count number of identifiers
    @return true if succeed, otherwise false
*/
bool NPScriptableObjectBase::Enumerate(NPIdentifier** /*identifier*/, uint32_t* /*count*/)
{
    return false;
}

/*!
    Constuct a new scriptable object

    Always returns false in this base class, derived class should implement this method.

    @param args arguments
    @param argCount number of arguments
    @param[out] result consructed object
    @return true if succeed, otherwise false
*/
bool NPScriptableObjectBase::Construct(const NPVariant* /*args*/, uint32_t /*argCount*/,
                                      NPVariant* /*result*/)
{
    return false;
}

/*!
    Static helper function to delete \a npobj

    @param npobj a NPObject instance
*/
void NPScriptableObjectBase::_Deallocate(NPObject *npobj)
{
    delete (NPScriptableObjectBase *)npobj;
    npobj = NULL;
}

/*!
    Static helper function to invalidate \a npobj

    @param npobj a NPObject instance

    @see Invalidate
*/
void NPScriptableObjectBase::_Invalidate(NPObject *npobj)
{
    Q_ASSERT(npobj);
    ((NPScriptableObjectBase *)npobj)->Invalidate();
}

/*!
    Static helper function to check if a method exists in the \a npobj given \a methodId

    @param npobj a NPObject instance
    @param methodId method identifier
    @return true if the method exists, otherwise false

    @see HasMethod
*/
bool NPScriptableObjectBase::_HasMethod(NPObject *npobj, NPIdentifier methodId)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->HasMethod(methodId);
}

/*!
    Static helper function to dispatch a script call to the \a npobj

    @param npobj a NPObject instance
    @param methodId the identifier of the method
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false

    @see Invoke
*/
bool NPScriptableObjectBase::_Invoke(NPObject *npobj, NPIdentifier methodId,
                                     const NPVariant *args, uint32_t argCount,
                                     NPVariant *result)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->Invoke(methodId, args, argCount, result);
}

/*!
    Static helper function to dispatch the default call to the \a npobj

    @param npobj a NPObject instance
    @param args the list of arguments for the method
    @param argCount number of arguments.
    @param[out] result  return value of the method
    @return true if the method call success, otherwise false

    @see InvokeDefault
*/
bool NPScriptableObjectBase::_InvokeDefault(NPObject *npobj,
                                            const NPVariant *args, uint32_t argCount,
                                            NPVariant *result)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->InvokeDefault(args, argCount, result);
}

/*!
    Static helper function to check if a property exists in the \a npobj given \a propertyId

    @param npobj a NPObject instance
    @param propertyId property identifier
    @return true if the property exists, otherwise false

    @see HasProperty
*/
bool NPScriptableObjectBase::_HasProperty(NPObject *npobj, NPIdentifier propertyId)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->HasProperty(propertyId);
}

/*!
    Static helper function to get the value of a property of the \a npobj given \a propertyId

    @param npobj a NPObject instance
    @param propertyId property identifier
    @param[out] result  property value
    @return true if succeed in getting value of the property, otherwise false

    @see GetProperty
*/
bool NPScriptableObjectBase::_GetProperty(NPObject *npobj, NPIdentifier propertyId,
                                          NPVariant *result)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->GetProperty(propertyId, result);
}

/*!
    Static helper function to set a property of the \a npobj given \a propertyId and \a value

    @param npobj a NPObject instance
    @param propertyId property identifier
    @param value property value
    @return true if succeed in setting the property, otherwise false

    @see SetProperty
*/
bool NPScriptableObjectBase::_SetProperty(NPObject *npobj, NPIdentifier propertyId,
                                          const NPVariant *value)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->SetProperty(propertyId, value);
}

/*!
    Static helper function to remove a property from the \a npobj given \a propertyId

    @param npobj a NPObject instance
    @param propertyId property identifier
    @return true if succeed, otherwise false

    @see RemoveProperty
*/
bool NPScriptableObjectBase::_RemoveProperty(NPObject *npobj, NPIdentifier propertyId)
{
    return ((NPScriptableObjectBase *)npobj)->RemoveProperty(propertyId);
}

/*!
    Static helper functio to list of all identifiers for properties and methods, defined in the \a npobj

    @param npobj a NPObject instance
    @param[out] identifier pointer to the array of identifiers
    @param[out] count number of identifiers
    @return true if succeed, otherwise false

    @see Enumerate
*/
bool NPScriptableObjectBase::_Enumerate(NPObject *npobj, NPIdentifier **identifier,
                                        uint32_t *count)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->Enumerate(identifier, count);
}

/*!
    Static helper function to constuct a new scriptable object

    @param npobj a NPObject instance
    @param args arguments
    @param argCount number of arguments
    @param[out] result consructed object
    @return true if succeed, otherwise false

    @see Construct
*/

bool NPScriptableObjectBase::_Construct(NPObject *npobj, const NPVariant *args,
                                        uint32_t argCount, NPVariant *result)
{
    Q_ASSERT(npobj);
    return ((NPScriptableObjectBase *)npobj)->Construct(args, argCount, result);
}

#ifdef __cplusplus
extern "C" {
#endif

/*!
    Static helper function to initialize the NPN functions

    @param npnFuncs NPN functions
*/
Q_DECL_EXPORT void bindingInitialize(NPNetscapeFuncs* npnFuncs)
{
    NPNFuncs = npnFuncs;
}

#ifdef __cplusplus
}
#endif

} // namespace
// End of file
