/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#ifndef service_h
#define service_h

#include <QList>

#include "servicebase.h"

/*!
    The class for service object
*/
template<class Base>
class Service : public Base
{
public:
    Service():m_Ref(0) {};
    virtual ~Service()
    {
        qDeleteAll(m_children);
        m_children.clear();
    };

    /// Begin IServiceBase implementation
    void getInterface(const ServiceInterfaceInfo& aInterfaceInfo, void** aInterface)
    {
        IServiceBase* base = getServiceBase();
        //get base interface
        if (!strcmp(aInterfaceInfo.m_iid, KIServiceBaseInfo.m_iid)) {
            *aInterface = base;
        }
        else {
            *aInterface = this->qt_metacast(aInterfaceInfo.m_iid);
            if (!(*aInterface)) { // try children service objects
                foreach(QObject *serviceObject, m_children) {
                    *aInterface = serviceObject->qt_metacast(aInterfaceInfo.m_iid);
                }
            }
        }
        if (*aInterface) { // increase reference count
            base->addRef();
        }
    }

    //IServiceBase getter increase reference count
    void getInterface(IServiceBase** aServiceBase)
    {
        *aServiceBase = getServiceBase();
        if (*aServiceBase){
            (*aServiceBase)->addRef();
        }
    }
    void addRef() { ++m_Ref; }
    void release() {if(--m_Ref == 0) delete this;}
    QObject* getServiceObject() {return this;}
    /// End IServiceBase implementation
    void addChild(QObject* aObject) {m_children.append(aObject);}
    // IServiceBase getter, no change on reference count
    IServiceBase* getServiceBase() {
        return reinterpret_cast<IServiceBase*>(
            (reinterpret_cast<unsigned char*>(this) + sizeof(QObject))); }
private:
    int m_Ref;
    QList<QObject*> m_children;
};

/*!
    The class for singleton service object, reference count is not used
*/
template<class Base>
class ServiceSingleton : public Base
{
public:
    ServiceSingleton() {};
    virtual ~ServiceSingleton()
    {
        qDeleteAll(m_children);
        m_children.clear();
    };
    /// Begin IServiceBase implementation
    void getInterface(const ServiceInterfaceInfo& aInterfaceInfo, void** aInterface)
    {
        IServiceBase* base = getServiceBase();
        //get base interface
        if (!strcmp(aInterfaceInfo.m_iid, KIServiceBaseInfo.m_iid)) {
            *aInterface = base;
        }
        else {
            *aInterface = this->qt_metacast(aInterfaceInfo.m_iid);
            if (!(*aInterface)) { // try children service objects
                foreach (QObject* serviceObject, m_children)
                    *aInterface = serviceObject->qt_metacast(aInterfaceInfo.m_iid);
            }
        }
        if (*aInterface) { // increase reference count
            base->addRef();
        }
    }
    void addRef() {}
    void release() {}
    QObject* getServiceObject() {return this;}
    /// End IServiceBase implementation
    void addChild(QObject* aObject) {m_children.append(aObject);}
    // IServiceBase getter, no change on reference count
    IServiceBase* getServiceBase() {
        return reinterpret_cast<IServiceBase*>(
            (reinterpret_cast<unsigned char*>(this) + sizeof(QObject))); }
private:
    QList<QObject*> m_children;
};

/*!
    The class for aggregate service object or inner object,
    reference count is not used
*/
template<class Base>
class ServiceAggregate : public Base
{
public:
    ServiceAggregate(IServiceBase* parent)
    : m_parent(parent) {};
    virtual ~ServiceAggregate() {};
    /// Begin IServiceBase implementation
    void getInterface(const ServiceInterfaceInfo& aInterfaceInfo, void** aInterface)
    {
        m_parent->getInterface(aInterfaceInfo, aInterface);
    }
    void addRef() {m_parent->addRef();}
    void release() {m_parent->release();}
    QObject* getServiceObject() {return this;}
    /// End IServiceBase implementation
private:
    IServiceBase* m_parent;
};

#define CREATE_SERVICE_OBJECT(ServiceClass, pInterface, refCounted) \
    Service<ServiceClass>* serviceObject(NULL); \
    ServiceSingleton<ServiceClass>* singletonServiceObject(NULL); \
    if (refCounted){ \
        serviceObject = new Service<ServiceClass>(); \
        pInterface = serviceObject ? serviceObject->getServiceBase() : NULL; \
    } else { \
        singletonServiceObject = new ServiceSingleton<ServiceClass>(); \
        pInterface = singletonServiceObject ? singletonServiceObject->getServiceBase() : NULL; \
    }

#endif //service_h
