/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



#include "usercondition.h"
#include "userconditionhandler.h"
#include "userconditionserializer.h"

namespace WRT
{

/*!
 * \class UserCondition
 * This class represents a user condition and the current grant state
 */

/**
 * Constructor
 * \param[in] capability - Capability for this condition
 * \param[in] flags - Current flags for this condition (see enum #UserGrant)
 * \param[in] handler - User Condition Handler
 * \param[in] serializer - Condition Serializer
 */
UserCondition::UserCondition(const QString &capability,
                             unsigned int flags,
                             const UserConditionHandler * handler,
                             UserConditionSerializer * serializer) :
    m_flags(flags),
    m_userConditionHandler(handler),
    m_userConditionSerializer(serializer),
    m_granted(0)
{
    add(capability);
}

/*!
 * From Condition interface
 * \see Condition::isMet()
 */
bool UserCondition::isMet(const QString &serviceName)
{
    bool result(false);
    if (((m_granted & EUserGrantBlanket) == EUserGrantBlanket) || ((m_granted
                    & EUserGrantSession) == EUserGrantSession)) {
        result = true;
    } else if (m_userConditionHandler) {
        result = m_userConditionHandler->allow(serviceName,m_capabilities, m_flags, m_granted);
    }
    return result;
}

/*!
 * From Condition interface
 * \see Condition::reset()
 */
bool UserCondition::persist()
{
    bool ret(false);
    if (m_userConditionSerializer) {
        ret = m_userConditionSerializer->persist(m_capabilities,
                                   m_flags & EUserGrantPersistMask,
                                   m_granted & EUserGrantBlanket);
    }
    return ret;
}

/*!
 * From Condition interface
 * \see Condition::persist()
 */
void UserCondition::reset()
{
    m_granted = 0;
}

/*!
 * Adds a capability to this condition
 * \param[in] capability - Capability to add
 * \return void
 */
void UserCondition::add(const QString &capability)
{
    m_capabilities.append(capability);
}

/*!
 * Updates the flag and grant status by doing a bitwise OR
 * \param[in] flags - Flags to update
 * \param[in] grants - Grants to update
 * \return void
 */
void UserCondition::update(unsigned int flags, unsigned int grants)
{
    m_flags |= flags;
    m_granted |= grants;
}

} // end namespace WRT
