/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "trustsession.h"
#include "trustsession_p.h"
#include "trustparser.h"
#include "originhandler.h"
#include "processhandler.h"
#include "certificatehandler.h"
#include "apppathhandler.h"

namespace WRT
{

/* TrustSessionPrivate implementation */
TrustSessionPrivate::TrustSessionPrivate() :
    m_parser(0)
{
    m_trustHandlers.insert(ETrustCriteriaOrigin, new OriginHandler());
    m_trustHandlers.insert(ETrustCriteriaAppPath, new AppPathHandler());
    m_trustHandlers.insert(ETrustCriteriaProcessId, new ProcessHandler());
    m_trustHandlers.insert(ETrustCriteriaCertificate, new CertificateHandler());
}

  TrustSessionPrivate::~TrustSessionPrivate()
  {
    for (int i = 0; i < m_trustHandlers.count(); i++) {
      delete m_trustHandlers[i];
    }
    if (m_parser) {
      delete (m_parser);
    }
  }

/* TrustSession implementation */

/*!
 * \class TrustSession
 * This class represents a trust manager client session
 */

/*!
 * Default Constructor
 */
  TrustSession::TrustSession()
{
    m_privateTrustSession = new TrustSessionPrivate();
}


/*!
 * Constructor with a policy file
 */
TrustSession::TrustSession(const QString &policy)
{
    m_privateTrustSession = new TrustSessionPrivate();
    setPolicyFile(policy);
}

/*!
 * Destructor
 */
TrustSession::~TrustSession()
{
    delete m_privateTrustSession;
}

/*!
 * Sets the policy file
 * \param[in] policyFile Filename of the policy file
 * \return TRUE if the policy file is valid, else FALSE
 */
bool TrustSession::setPolicyFile(const QString &policyFile)
{
    if (m_privateTrustSession->m_parser)
      delete m_privateTrustSession->m_parser;
    m_privateTrustSession->m_parser = new TrustParser(policyFile);
    if (!m_privateTrustSession->m_parser->parse()) {
      delete m_privateTrustSession->m_parser;
      m_privateTrustSession->m_parser = NULL;
      return false;
    }
    else
      return true;
}

/*!
 * Sets the policy file in memory
 * \param[in] policyFile The policy file read into memory as a QByteArray
 * \return TRUE if the policy file is valid, else FALSE
*/
bool TrustSession::setPolicyFile(const QByteArray &policyFile)
{
    if (m_privateTrustSession->m_parser)
      delete m_privateTrustSession->m_parser;
    m_privateTrustSession->m_parser = new TrustParser(policyFile);
    if (!m_privateTrustSession->m_parser->parse()) {
      delete m_privateTrustSession->m_parser;
      m_privateTrustSession->m_parser = NULL;
      return false;
    }
    else
      return true;
}

/*!
 * Gets the domain for a specified value
 * \param[in] type - Type of domain
 * \param[in] value - Value to get the domain for
 * \return The requested domain
 */
QString TrustSession::domainFor(TrustCriteriaType type, const QString &value)
{
    // TODO: Perhaps using an enum is not that good because it has header
    // dependencies for clients.
    QString domain;
    if (m_privateTrustSession->m_parser) {
      TrustPropHandler *handler = m_privateTrustSession->m_trustHandlers[type];
      handler->setParser(m_privateTrustSession->m_parser);
      domain = handler->domainFor(value);
    }
    return domain;
}

/*!
 * Gets the domain for a specified value
 * \param[in] name - Name of the type
 * \param[in] value - Value to get the domain for
 * \return The requested domain
 */
QString TrustSession::domainFor(const QString &name, const QString &value)
{
    TrustCriteriaType type;
    if (name == QString(KPROCESS)) {
      type = ETrustCriteriaProcessId;
    }
    else if (name == QString(KAPPPATH)) {
      type = ETrustCriteriaAppPath;
    }
    else if (name == QString(KCERTIFICATE)) {
      type = ETrustCriteriaCertificate;
    }
    else {
      type = ETrustCriteriaOrigin;
    }
    return domainFor(type, value);
}

}// end namespace WRT
