/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



// INCLUDE FILES
#include <QUuid>
#include <QFileInfo>
#include <QDir>
#include <QDebug>
#include "secmgrdefs.h"
#include "capability.h"
#include "secsession.h"
#include "secsession_p.h"
#include "secpolicy.h"
#include "usercondition.h"

namespace WRT
{

static const char SECSESSION_FILE[] = "secsession";

  /*
   * SecSessionPrivate implementation
   */

SecSessionPrivate::SecSessionPrivate(const QByteArray &policy,
                     const QString &domain,
                     const QString &store) :
    m_policy(NULL),
    m_domain(domain),
    m_store(store),
    m_allowedCaps(NULL),
    m_key(QUuid::createUuid().toString())
{
    if (!policy.isEmpty()) {
        m_policy = new SecPolicy(policy);
    }
    // If no storage location is provided, use a temporary path
    if (m_store.isNull() || m_store.isEmpty()) {
        m_store = QDir::tempPath();
    }
}

SecSessionPrivate::SecSessionPrivate(const QString &key,
                     const QString &store,
                     UserConditionHandler *handler /*=NULL*/) :
    m_store(store),
    m_allowedCaps(NULL),
    m_key(key)
{
    QString filestore(store + key);
    m_policy = new SecPolicy(filestore);
    m_policy->setUserHandler(handler);
    m_allowedCaps = m_policy->evaluate(); //Evaluate on the default domain
    m_domain = m_policy->defaultDomain();
}

SecSessionPrivate::SecSessionPrivate(const QByteArray &blob,
                                     UserConditionHandler *handler /*=NULL*/) :
    m_allowedCaps(NULL)
{
    m_policy = new SecPolicy(blob);
    m_policy->setUserHandler(handler);
    m_allowedCaps = m_policy->evaluate(); //Evaluate on the default domain
    m_domain = m_policy->defaultDomain();
}

SecSessionPrivate::SecSessionPrivate(const QString &policyFileName,
                   const QString &trustDomain,
                   const QString &secureStoreLocation,
                   UserConditionHandler *handler) :
    m_policy(NULL),
    m_domain(trustDomain),
    m_store(secureStoreLocation),
    m_allowedCaps(NULL),
    m_key(QUuid::createUuid().toString())
{
    m_policy = new SecPolicy(policyFileName);
    m_policy->setUserHandler(handler);

    // If no storage location is provided, use a temporary path
    if (m_store.isNull() || m_store.isEmpty()) {
        m_store = QDir::tempPath();
    }
}

SecSessionPrivate::~SecSessionPrivate()
{
    delete m_policy;
    m_clientInfo.clear();
}

/*
 * SecSession implementation
 */

/*!
 * \class SecSession
 * This class represents a security manager client session
 * Exported.
 */

/*!
 * Constructor
 * \param[in] policy - Policy file in memory
 * \param[in] domain - Domain name
 * \param[in] store - Storage location for persistence
 */
SecSession::SecSession(const QByteArray &policy,
                       const QString &domain,
                       const QString &store)
{
    m_privSecSession = new SecSessionPrivate(policy, domain, store);
}

/*!
 * Constructor using the key of a previously persisted session
 * \pre A previous session must have been stored
 * \param[in] key - Key of previously persisted session
 * \param[in] store - Storage location for persisted sessions
 * \param[in] handler - User condition handler
 */
SecSession::SecSession(const QString &key,
                       const QString &store,
                       UserConditionHandler *handler /*=NULL*/)
{
    m_privSecSession = new SecSessionPrivate(key, store, handler);
}

SecSession::SecSession(const QByteArray& blob,
                       UserConditionHandler *handler /*=NULL*/)

{
    m_privSecSession = new SecSessionPrivate(blob, handler);
}

SecSession::SecSession(const QString &policyFileName,
                       const QString &trustDomain,
                       const QString &secureStoreLocation,
                       UserConditionHandler *handler)
{
    m_privSecSession = new SecSessionPrivate(policyFileName, trustDomain, secureStoreLocation, handler);
}

/*!
 * Destructor
 */
SecSession::~SecSession()
{
    delete m_privSecSession;
}

/*!
 * Checks whether the list of capabilities are allowed
 * \param[in] neededCaps - This list of capabilities needed by the caller
 * \return TRUE if allowed, else FALSE
 */


bool SecSession::isAllowed(const CapabilityList &neededCaps)
{
    bool result(true);
    if (neededCaps.size() == 0)
        return true;
    if (m_privSecSession->m_policy) {
        if (!m_privSecSession->m_allowedCaps) {
            m_privSecSession->m_allowedCaps = m_privSecSession->m_policy->evaluateOn(m_privSecSession->m_domain);

        }

        if (m_privSecSession->m_allowedCaps) {
            for (int i = 0; i < neededCaps.size() && result; i++) {
                // Check if the capability is in the allowed list
                // ignore empty strings
                QString cap(neededCaps.at(i));
                QStringList capList = m_privSecSession->m_allowedCaps->keys();
                bool isFound = false;

                for (int j=0; j<capList.count(); j++) {
                    QRegExp regExp(capList[j],Qt::CaseInsensitive,QRegExp::Wildcard);
                    if (regExp.exactMatch(cap)) {
                        isFound = true;
                        break;
                    }
                }
                if (!cap.isEmpty() && !isFound) {

                    // Check if the capability has an alias
                    if (m_privSecSession->m_policy->hasAlias(cap)) {
                        if (!atLeastOneAllowed(m_privSecSession->m_policy->aliasesFor(cap))) {
                            result = false;
                        }
                    } else {
                        result = false;
                    }
                }
                if (result) {
                    // Check for conditional capability
                    Condition * cond = m_privSecSession->m_allowedCaps->value(neededCaps.at(i));
                    if (cond) {
                        if (!cond->isMet(displayText())) {
                            result = false;
                        }
                    }
                }
            }
        }
        else {
      result = false; // No allowed capabilities
    }
    }
    else {
      result = true; // No policy
    }
    return result;
}

/*!
 * Persists the current session
 * \return void
 */
bool SecSession::persist()
{
    bool result(false);
    if (m_privSecSession->m_policy) {
        if (!m_privSecSession->m_allowedCaps)
            m_privSecSession->m_allowedCaps = m_privSecSession->m_policy->evaluateOn(m_privSecSession->m_domain);
        result = m_privSecSession->m_policy->persist(m_privSecSession->m_key, m_privSecSession->m_store, m_privSecSession->m_domain);
    }
    return result;
}

bool SecSession::persist(QByteArray &blob)
{
    bool result(false);
    if (m_privSecSession->m_policy) {
        if (!m_privSecSession->m_allowedCaps)
            m_privSecSession->m_allowedCaps = m_privSecSession->m_policy->evaluateOn(m_privSecSession->m_domain);
        result = m_privSecSession->m_policy->persist(blob, m_privSecSession->m_domain);
    }
    return result;
}

bool SecSession::saveSecsession(QString resourcesDir, QString secureSessionString)
{
    QDir dir;
    dir.mkpath(resourcesDir);
    QString secSessionFileName(QDir::toNativeSeparators(resourcesDir));
    secSessionFileName += QDir::separator();
    secSessionFileName += SECSESSION_FILE;
    QFile secSessionFile(secSessionFileName);
    if (secSessionFile.open(QIODevice::WriteOnly)) {
        QTextStream stream(&secSessionFile);
        stream << secureSessionString;
        secSessionFile.close();
        return true;
    }
    return false;
}

/*!
 * Sets the user condition handler to evaulate permissions
 * \param handler - User Condition Handler
 * \return void
 */
void SecSession::setUserHandler(UserConditionHandler *handler) const
{
    if (m_privSecSession->m_policy) {
        m_privSecSession->m_policy->setUserHandler(handler);
    }
}

/*!
 * Generates a key for this session, to be used when sharing sessions
 * \see SecSession(const QString &aKey, const QString &aStore)
 * \return The key for this policy
 */
QString SecSession::key() const
{
    return m_privSecSession->m_key;
}

/*!
 * Sets the display text of client
 * \param text Display text
 */
void SecSession::setDisplayText(const QString &text)
{
    setClientInfo(KDISPLAYTEXT, text);
}

/*!
 * Retrieves the display text
 * \return Display text
 */
QString SecSession::displayText() const
{
    return clientInfo().value(KDISPLAYTEXT, QString());
}

/*!
 * Sets the name of client
 * \param clientName Client Name
 */
void SecSession::setClientName(const QString &clientName)
{
    setClientInfo(KCLIENTNAME, clientName);
}

/*!
 * Retrieves the Client Name
 * \return Client Name
 */
QString SecSession::clientName() const
{
    return clientInfo().value(KCLIENTNAME, QString());
}

/*!
 * Gets the client info hashlist
 * \return The Client Info hash list
 */
const QHash<QString, QString> &SecSession::clientInfo() const
{
    return m_privSecSession->m_clientInfo;
}

/*!
 * Generic method to set any value in the client info list
 * \param aKey Key to set
 * \param aValue Value for the key
 */
void SecSession::setClientInfo(const QString &key, const QString &value)
{
    m_privSecSession->m_clientInfo.insert(key, value);
}

// Private method used for alias checking
bool SecSession::atLeastOneAllowed(const CapabilityList &neededCaps)
{
    bool result(false);
    if (m_privSecSession->m_policy) {
        for (int i = 0; i < neededCaps.size() && !result; i++) {
            QString cap(neededCaps.at(i));
            QStringList capList = m_privSecSession->m_allowedCaps->keys();
            bool isFound = false;

            for (int j=0; j<capList.count(); j++) {
                QRegExp regExp(capList[j],Qt::CaseInsensitive,QRegExp::Wildcard);
                if (regExp.exactMatch(cap)) {
                    isFound = true;
                    break;
                }
            }

            if (isFound) {
                Condition *cond = m_privSecSession->m_allowedCaps->value(cap);
                if (cond) {
                    if (cond->isMet(displayText())) {
                        result = true;
                    }
                } else {
                    result = true;
                }
            }
        }
    }
    return result;
}

} // end namespace WRT
