/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */



#include <QUrl>
#include "originhandler.h"
#include "trustparser.h"

//TBD: replace this with WebKit same-origin check

namespace WRT
{

/*!
 * \class OriginHandler
 * Implementation class of TrustPropHandler interface.
 * This class uses the origin to find corresponding domain
 * Gets domain for the the policy url that is either equal to the
 * input value or the longest parent url for the input value
 */

/*!
 * Constructor
 * \pre Must have already parsed the policy file.
 */
OriginHandler::OriginHandler()
{
    m_parser = NULL;
}

/*!
 * Destructor
 */
OriginHandler::~OriginHandler()
{
}

/*!
 * Gets the domain for a specified value
 * \param[in] value - Value to get the domain for
 * \return The requested domain
 */
QString OriginHandler::domainFor(const QString &value)
{
    QString result;
    if (m_parser) {
        int i = bestMatchFor(value);
        if (i < 0) {
            result = m_parser->defaultOriginDomain();
        } else {
            result = m_parser->domainFor(ETrustCriteriaOrigin,
                                        (m_parser->values(ETrustCriteriaOrigin))[i]);
        }
    }
    return result;
}

/*!
 * Finds the best matching index for the specified value
 * \param[in] value - Value to find the match
 * \return Index within the parser's list of domains, or -1 if not found.
 */
int OriginHandler::bestMatchFor(const QString &value)
{
    int ret = -1;
    if (m_parser) {
        QStringList originList = m_parser->values(ETrustCriteriaOrigin);
        int index = 0;
        int comp = 0;

        //TBD check validity
        QUrl url(value);
        //qDebug("=========> Original url: %s\n", url.toString().toLatin1().data());
        if (url.isRelative()) // No scheme: this is a hack for WebAppPath which contains no scheme
            url.setScheme("file");

        //qDebug("=======> Transformed url: %s\n", url.toString().toLatin1().data());
        QString domain = url.toString(QUrl::RemovePath);
        //qDebug("========> Domain: %s\n", domain.toLatin1().data());
        //Matches with same domain
        QStringList matches = originList.filter(domain);

        //Get longest match that is parent of value
        for (int i = 0; i < matches.size(); i++) {
            QUrl match(matches[i]);
            //qDebug("=========> Match against: %s\n", match.toString().toLatin1().data());
            if (match == url) {
                //qDebug("=======> Exact match\n");
                comp = matches[i].size();
                index = i;
                ret = originList.indexOf(matches[index]);
                break;
            }
            if (match.isParentOf(url)) {
                //qDebug("=======> Parent match\n");
                if (matches[i].size() > comp) {
                    //qDebug("========> Best match\n");
                    comp = matches[i].size();
                    index = i;
                }
            }
            if (comp > 0) {
                ret = originList.indexOf(matches[index]);
            }
        }
    }
    return ret;
}

} // end namespace WRT
