/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef __WRT_STORAGE_HH
#define __WRT_STORAGE_HH

#include <QObject>
#include <QString>
#include <QStringList>

class QFile;
class QFileInfo;
class QBuffer;

/**
* Class export macro
*/
#if defined(QT_MAKE_SECURESTORAGE_DLL)
#define SECURESTORAGE_API Q_DECL_EXPORT
#else
#define SECURESTORAGE_API Q_DECL_IMPORT
#endif

namespace WRT {

/** Storage interface
 *
 */
class SECURESTORAGE_API Storage: public QObject {
    Q_OBJECT

public:

    /**
     * The visibility of a storage. Defined
     * when a storage is created and cannot be changed
     * afterwards.
     */
    typedef enum {vis_shared, vis_private}visibility;

    /**
     * The protection level of a storage. Defined
     * when a storage is created and cannot be changed
     * afterwards.
     */
    typedef enum {prot_signed, prot_encrypted}protection;

    /**
     * Factory method for creating new instance of WRT::Storage class.
     */
    static Storage * createInstance(const QString & aStorageName, WRT::Storage::visibility = vis_shared, WRT::Storage::protection = prot_signed, QObject* aParent = 0);
    virtual ~Storage();

    /**
     * Returns a list of files stored in the storage
     * @return a list of file name stored in the storage.
     */
    virtual QStringList getFileList() const = 0;

    /**
     * Adds a file to storage. Calculates checksum for the file based on the file name and size. Basic support based on MD5 hash.
     * @param aFile the file to be added to storage, can be in either opened or closed state
     */
    virtual void add(const QFile & aFile) = 0;

    /**
     * Adds an in memory file to storage. Calculates checksum for the file based on the file name and size. Basic support based on MD5 hash.
     * @param aFilePath the the file path including filename that is to be added to the storage
     */
    virtual void add(const QString & aFilePath, const QBuffer & aData) = 0;

    /**
     * Removes a file from storage
     * @param aFile to be removed from storage, can be in either opened or closed state
     */
    virtual void remove(const QFile & aFile) = 0;

    /**
     * Verifies that the checksum of a file matches to the stored file.
     * Checksum is calculated from the file name, size, and last modification date of the file. Basic support based on MD5 hash.
     * @param aFile to be verified, can be in either opened or closed state
     */
    virtual bool verify(const QFile & aFile) const = 0;

    /**
     * Returns a file stored in the storage. Caller becomes the owner of the QIODevice. Hence, the caller must delete the QIODevice instance.
     * @param aFile the name of file to be returned, including full path.
     * @return a QIODevice instance when file exists in storage, null is returned otherwise. The returned QIODevice is not opened
     */
    virtual QIODevice * getFile(const QString & aPathFile) = 0;

    /**
     * Returns the number of files stored in the the storage.
     */
    virtual qint32 count() const = 0;

    /**
     * Returns the storage name
     */
    virtual QString storageName() const;

public:
    // from base
public slots:

protected:
    // Constructing through factory method
    explicit Storage(const QString & aStorageName, QObject* aParent = 0);
    QString m_storageName;


private:
    WRT::Storage&  operator = (const WRT::Storage&);
    Storage(const WRT::Storage &);
};

}
#endif
