/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


#include "securestoragedefs.h"
#include "securestorageclient.h"
#include "serviceipc.h"
#include "securestoragemetadata.h"
#include "sslog.h"

namespace WRT
{
/*!
 * \class WRT::SecureStorageClient
 * This class is the client interface to secure storage
 */


/*!
 * Constructor with initialization ID
 * \param[in] aId - ID of this client to identify the session
 * \param[in] aParent - QObject parent
 */
SecureStorageClient::SecureStorageClient(const QString &aStorageName,
                                         Storage::visibility aVisibility /* = vis_shared */,
                                         Storage::protection aProtection /* = prot_signed */,
                                         QObject* aParent /* = 0 */)
    : SecureStorage(aStorageName, aVisibility, aProtection, aParent),
      m_IPCClient(NULL)
{
    initStorage();
}

/*!
 * Destructor
 */
SecureStorageClient::~SecureStorageClient()
{
    if (m_IPCClient) {
        m_IPCClient->disconnect();
        delete m_IPCClient;
        m_IPCClient = NULL;
    }
}


//------------------------------------------------------------------------
// Protected Methods
//------------------------------------------------------------------------

/*
 * Initializes the storage
 */
void SecureStorageClient::initStorage()
{
    SS_FUNC("SecureStorageClient::initStorage()");
    m_IPCClient = new ServiceFwIPC( this );
    if (m_IPCClient) {
        bool connected = m_IPCClient->connect(KSECURESTORAGESERVER);
        if (!connected) {
            // Start server if we cannot connect
            if (m_IPCClient->startServer(KSECURESTORAGESERVER, KSECURESTORAGESERVEREXE)) {
                connected = m_IPCClient->connect(KSECURESTORAGESERVER);
            }
        }

        // Check again if connected
        if (!connected) {
            delete m_IPCClient;
            m_IPCClient = NULL;
            // maybe throw exception instead?
        } else {
            // Send storage name, visibility and protection level parameters to
            // server for initialization
            QByteArray data;
            data += m_storageName.toUtf8();
            data += QString(KSECURESTORAGEDELIMITER).toUtf8();
            data += (QString::number(m_visibility)).toUtf8();
            data += QString(KSECURESTORAGEDELIMITER).toUtf8();
            data += (QString::number(m_protection)).toUtf8();
            if (!m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPINIT), data)) {
                // TODO: Error handling
                m_IPCClient->disconnect();
                delete m_IPCClient;
                m_IPCClient = NULL;
            } else {
                // Get file list
                m_fileList = updateFileList();
            }
        }
    }
}

/*!
 * From SecureStorage
 * Updates the current file list
 */
QHash<QString, QString> SecureStorageClient::updateFileList() const
{
    SS_FUNC("SecureStorageClient::updateFileList()");
    QHash<QString, QString> fileList;
    if (m_IPCClient) {
        QByteArray data = m_storageName.toUtf8();
        if (m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPGETFILELIST), data)) {
            data = m_IPCClient->readAll();
            if (data.size() > 0) {
                QDataStream bufferStream(data);
                bufferStream >> fileList;
            }
        }
    } else {
        fileList = m_fileList;
    }
    SS_DEBUG("SecureStorageClient::updateFileList(): m_fileList.count() = " << m_fileList.count());
    return fileList;
}

/*!
 * From SecureStorage
 * Saves the current file list
 */
void SecureStorageClient::saveFileList()
{
    SS_FUNC("SecureStorageClient::saveFileList()");
    // Send file list to server to be saved
    if (m_IPCClient) {
        QByteArray data;
        data += m_storageName.toUtf8();
        data += QString(KSECURESTORAGEDELIMITER).toUtf8();
        QBuffer buffer;
        if (buffer.open(QIODevice::ReadWrite)) {
            QDataStream bufferStream(&buffer);
            bufferStream << m_fileList;
            data += buffer.data();
            buffer.close();
        }
        if (m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPSAVEFILELIST), data)) {
            // TODO: No error handling?
        }
    }
}

/*!
 * From SecureStorage
 * Saves the given metadata
 * \param[in] aMetadata - metadata to be saved
 */
void SecureStorageClient::saveMetadata(const SecureStorageMetadata& aMetadata)
{
    SS_FUNC("SecureStorageClient::saveMetadata()");
    // Sends metadata to the server to be saved
    if (m_IPCClient) {
        QBuffer buffer;
        aMetadata.externalize(buffer);
        QByteArray data;
        data += m_storageName.toUtf8();
        data += QString(KSECURESTORAGEDELIMITER).toUtf8();
        data += buffer.data();
        if (m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPSAVEMETADATA), data)) {
            // TODO: No error handling?
        }
    }
}

/*!
 * From SecureStorage
 * Removes the metadata from storage
 * \param[in] aFileName - File to remove the metadata for.
 */
void SecureStorageClient::removeMetadata(const QString &aFileName)
{
    SS_FUNC("SecureStorageClient::removeMetadata()");
    // Send filename to server to remove metadata
    if (m_IPCClient) {
        QByteArray data;
        data += m_storageName.toUtf8();
        data += QString(KSECURESTORAGEDELIMITER).toUtf8();
        data += aFileName.toUtf8();
        if (m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPREMOVEMETADATA), data)) {
            // TODO: No error handling?
        }
    }
}

/*!
 * From SecureStorage
 * Gets the metadata for the given file
 * \param[in] aFileName - File to retrieve metadata for
 */
SecureStorageMetadata * SecureStorageClient::getMetadata(const QString &aFileName) const
{
    SS_FUNC("SecureStorageClient::getMetadata()");
    SecureStorageMetadata *metadata(NULL);
    if (m_IPCClient) {
        QByteArray data;
        data += m_storageName.toUtf8();
        data += QString(KSECURESTORAGEDELIMITER).toUtf8();
        data += aFileName.toUtf8();
        if (m_IPCClient->sendSync(QString::number(KSECURESTORAGEOPGETMETADATA), data)) {
            data = m_IPCClient->readAll();
            if (!data.isNull() && !data.isEmpty()) {
                QBuffer buffer(&data);
                metadata = new SecureStorageMetadata();
                metadata->internalize(buffer);
            }
        }
    }
    return metadata;
}

}   // end namespace WRT
