/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "wrtpage.h"
#include "qwebhistory.h"
#include "wrtcontroller.h"
#include "wrtpage_p.h"
#include "wrtsettings.h"
#include "../widgetmanager/src/widgetmanagerconstants.h"

#ifdef CWRT_BUILDING_TENONE
#include "hbmessagebox.h"
#include "hbinputdialog.h"
#include "hbeffect.h"
#include "hbaction.h"
#include "qwebframe.h"
#endif

#include <QMessageBox>
#include <QTextStream>
#include <QFile>
#include <QDebug>

#include <QDesktopServices>
#include <QDir>

#ifdef Q_OS_SYMBIAN
#include "useragent.h"
#endif // Q_OS_SYMBIAN

void QWEBKIT_EXPORT qt_webpage_setGroupName(QWebPage* page, const QString& groupName);
static const qint64 OFFLINE_WEBAPP_DEFAULT_QUOTA  = 5000000;

namespace WRT {

WrtPagePrivate::WrtPagePrivate()
{
}

WrtPagePrivate::~WrtPagePrivate()
{
}

WrtPage::WrtPage(QWidget *parent) : QWebPage(parent), d(new WrtPagePrivate())
{
    settings()->setAttribute(QWebSettings::PluginsEnabled, true);
    // Initialization below can only be done once per Webkit instance (and not once per page)
    // otherwise it hits an ASSERT in WebKit in debug mode
    static bool initialized = false;
    // Debug mode can be dealt with in debug - we need these settings while opening a database
    // while another widget is running in the backgound. That is why: if (!initialized) is commented.
 //   if (!initialized) {
        QString storagePath = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
        settings()->setAttribute(QWebSettings::OfflineStorageDatabaseEnabled, true);
        settings()->setAttribute(QWebSettings::OfflineWebApplicationCacheEnabled, true);
        settings()->setAttribute(QWebSettings::LocalStorageEnabled, true);
        QWebSettings::setOfflineStoragePath(QDir::toNativeSeparators(QDir(storagePath).filePath("Databases")));
        QWebSettings::setOfflineWebApplicationCachePath(DATA_PATH);
        QWebSettings::setOfflineWebApplicationCacheQuota(OFFLINE_WEBAPP_DEFAULT_QUOTA);
        initialized = true;
 //   }
}

WrtPage::~WrtPage()
{
    delete d;
}

void WrtPage::setPageGroupName(const QString& groupName)
{
    qt_webpage_setGroupName(this, groupName);
}

QWebPage* WrtPage::createWindow(QWebPage::WebWindowType webWindowType)
{

    Q_UNUSED(webWindowType);
    if (WrtSettings::createWrtSettings()->value("PopupBlocking").toInt())
        return 0;
    WrtPage* wrtPage = new WrtPage();
    emit createWindow(wrtPage);
    return wrtPage;
}

void WrtPage::enableLocalStorage(const QString & path) {
    settings()->setAttribute(QWebSettings::LocalStorageEnabled, true);
    settings()->setLocalStoragePath(path);
 }

qint64 WrtPage::enableOfflineWebApplicationCache(const QString & path, const qint64 & quota) {
    if (!QFile::exists(path)) {
        QDir dir(path);
        if (!dir.mkdir(path)) {
            qCritical()<<"FAIL: Can't create missing application cache directory" << path;
            return -1;
        }
    }
    settings()->setAttribute(QWebSettings::OfflineWebApplicationCacheEnabled, true);
    settings()->setOfflineWebApplicationCachePath(path);
    if (quota) { // If not set, it is by default OFFLINE_WEBAPP_DEFAULT_QUOTA
        settings()->setOfflineWebApplicationCacheQuota(quota);
    }

    return settings()->offlineWebApplicationCacheQuota();
}

/*!
// implementation of javaScriptConsoleMessage defined in WebKit qwebpage.h
// User settings are provided through WrtSettingsUI.
// The following choices are offered:
// Logging Off;Log to file;Show pop-up notes;Log to file and show pop-up notes
*/
void WrtPage::javaScriptConsoleMessage(const QString& messageSource, const QString& messageLevel, const QString& message, int lineNumber, const QString& sourceID)
{
    Q_UNUSED(messageSource);
    Q_UNUSED(messageLevel);
    javaScriptConsoleMessage(message, lineNumber, sourceID);
}

void WrtPage::javaScriptConsoleMessage(const QString& message, int lineNumber, const QString& sourceID)
{

    int jsSetting = WrtSettings::createWrtSettings()->value ( "JavaScriptConsoleLog" ).toInt();
    QString logFileName = WrtSettings::createWrtSettings()->value ( "JavaScriptConsoleLogFilePath" ).toString();
    QString msg = ( "WRT Console Log: "+ ( QDate::currentDate() ).toString ( Qt::SystemLocaleShortDate ) +" "+ ( QTime::currentTime() ).toString ( "hh:mm:ss ap" ) + \
            "\nMessage: " + message + "\nSource: "+sourceID+" : "+QString::number(lineNumber) );

    if ( jsSetting == 2 || jsSetting == 3 ){
#if defined(Q_OS_MAEMO5) || defined(Q_OS_MAEMO6)
        jsSetting = 1;
#else
        //display this simple dialog box for now.
#ifdef Q_OS_SYMBIAN
        // This is a S60 specific for forcing softkeys visible when popup opens, if softkeys are 
        // set to be invisible before opening. This is the only case where this signal is used.
        emit popupMessageVisible(true);
        QMessageBox::about(view(), "JavaScript Console", msg);
        emit popupMessageVisible(false);
#else
        QMessageBox::about(view(), "JavaScript Console", msg);
#endif
#endif
    }
    if ( jsSetting == 1 || jsSetting == 3 ){
    // message is logged to file
        QFile jsFile ( logFileName );
        if ( !jsFile.open ( QIODevice::Append ) ) return;

        QTextStream jsLog ( &jsFile );
        jsLog << msg << "\n";
        jsFile.close();
    }
    // Emit message in case some view manager wants to implement a better GUI to display console messages
    emit jsConsoleMessage("messageSource: N/A in QT46", "messageLevel: N/A in QT46", message, lineNumber, sourceID);
}


QString WrtPage::userAgentForUrl(const QUrl& url) const
{
#ifdef Q_OS_SYMBIAN
    Q_UNUSED(url);
    UserAgent* ua = new UserAgent();
    QString str(ua->userAgentString());
    delete ua;
    return str;
#else
    return QWebPage::userAgentForUrl(url);
#endif //Q_OS_SYMBIAN
}




#ifdef Q_OS_MAEMO6

   void WrtPage::javaScriptAlert ( QWebFrame * frame, const QString & msg ){
       //No return value for javascript. Just emit the signal, which hopefully is with
       //Qt::DirectConnection (if it is connected, and if not then nothing happens).
       //Execution will resume then the slot has handeled the alert box
       emit showJavaScriptAlert(frame, msg);
   }
   bool WrtPage::javaScriptConfirm ( QWebFrame * frame, const QString & msg ){
       //Here we can not directly emit the signal since we would loose the
       //return value from function. Instead invoke the signal in wrtpage's manually
       //with return value argument. This way the receiving slots return value
       //will end up here and can be returned to qwebpage

       if (receivers(SIGNAL(showJavaScriptConfirm(QWebFrame*, const QString&))) == 0){
           //Nobody is listenening to the signal
           return false;
       }
       bool retval = false;
       if (!QMetaObject::invokeMethod(this,"showJavaScriptConfirm",
                                     Qt::DirectConnection,
                                     Q_RETURN_ARG(bool,retval),
                                     Q_ARG(QWebFrame*,frame),
                                     Q_ARG(const QString&, msg))){
           //this should never fail if the header contains the signal
           qWarning()<<"Could not invoke showJavaScriptConfirm signal in"<<Q_FUNC_INFO;
           return false;
       }
       return retval;
   }
   bool WrtPage::javaScriptPrompt ( QWebFrame * frame, const QString & msg, const QString & defaultValue, QString * result ){
       //Here we can not directly emit the signal since we would loose the
       //return value from function. Instead invoke the signal in wrtpage's manually
       //with return value argument. This way the receiving slots return value
       //will end up here and can be returned to qwebpage
       if (receivers(SIGNAL(showJavaScriptPrompt(QWebFrame*, const QString&, const QString&, QString*))) == 0){
           //Nobody is listenening to the signal
           return false;
       }
       bool retval = false;
       if (!QMetaObject::invokeMethod(this,"showJavaScriptPrompt",
                                     Qt::DirectConnection,
                                     Q_RETURN_ARG(bool, retval),
                                     Q_ARG(QWebFrame*, frame),
                                     Q_ARG(const QString &, msg),
                                     Q_ARG(const QString &, defaultValue),
                                     Q_ARG(QString *, result))){
           //this should never fail if the header contains the signal
           qWarning()<<"Could not invoke showJavaScriptPrompt signal in"<<Q_FUNC_INFO;
           return false;
       }
       return retval;
   }

#else
#ifdef CWRT_BUILDING_TENONE
void WrtPage::javaScriptAlert(QWebFrame *frame, const QString& msg)
{
    HbMessageBox* dlg = new HbMessageBox(Qt::escape(msg));
    HbMessageBox *messageBox = new HbMessageBox(HbMessageBox::MessageTypeInformation, 0);
    messageBox->exec();
}

bool WrtPage::javaScriptConfirm(QWebFrame *frame, const QString& msg)
{
    HbMessageBox *messageBox = new HbMessageBox(HbMessageBox::MessageTypeQuestion, 0);
    messageBox->setText(Qt::escape(msg));
    HbAction *primaryAction = new HbAction("yes");
    messageBox->setPrimaryAction(primaryAction);
    HbAction *secondaryAction = new HbAction("no");
    messageBox->setSecondaryAction(secondaryAction);
    HbAction *action = messageBox->exec();
    if (action == messageBox->primaryAction())
        return true;
    else
        return false;
}

bool WrtPage::javaScriptPrompt(QWebFrame *frame, const QString& msg, const QString& defaultValue, QString* result)
{
    bool ok = false;
    QString res =  HbInputDialog::getText(Qt::escape(msg), defaultValue, &ok);
    if (ok)
        *result = res;
    return ok;
}

QString WrtPage::chooseFile(QWebFrame *parentFrame, const QString& suggestedFile)
{
/*
    HbFileDialog dlg;
    dlg.setDirectory(QDesktopServices::storageLocation(QDesktopServices::DataLocation));
    return (dlg.getFileName());
*/
    return QWebPage::chooseFile(parentFrame, suggestedFile);
}
#endif
#endif

}   // end of namespace WRT



