/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#ifndef WEBTOUCHNAVIGATION_H
#define WEBTOUCHNAVIGATION_H

#include <QObject>
#include <QPoint>
#include <QEvent>
#include <QMouseEvent>
#include <QGraphicsSceneMouseEvent>
#include <QTime>
#include <QTimer>
#include <QTimeLine>
#include <qwebelement.h>

class QWebFrame;
class QWebPage;

namespace WRT {

    class WebTouchEvent {
    public:
        QPoint m_pos;
        QEvent::Type m_type;
        Qt::MouseButton m_button;
        Qt::MouseButtons m_buttons;
        bool m_graphicsSceneEvent;
        bool m_fired;
        bool m_editable;
        Qt::KeyboardModifiers m_modifier;

        //Graphics scene event members
        QPointF m_scenePos;
        QPoint m_screenPos;
        QPointF m_buttonDownPos;
        QPointF m_buttonDownScenePos;
        QPoint m_buttonDownScreenPos;
        QPointF m_lastPos;
        QPointF m_lastScenePos;
        QPoint m_lastScreenPos;

        WebTouchEvent() : m_type(QEvent::None)
        , m_button(Qt::NoButton)
        , m_buttons(Qt::NoButton)
        , m_graphicsSceneEvent(false)
        , m_fired(false)
        , m_editable(false)
        , m_modifier(Qt::NoModifier)
        {}

        WebTouchEvent(const QMouseEvent *mouseEvent)
        {
            Q_ASSERT(mouseEvent != 0);
            m_type = mouseEvent->type();
            m_pos = mouseEvent->pos();
            m_button = mouseEvent->button();
            m_buttons = mouseEvent->buttons();
            m_modifier = mouseEvent->modifiers();
            m_fired = false;
            m_editable = false;
            m_graphicsSceneEvent = false;
        }

        WebTouchEvent(const QGraphicsSceneMouseEvent *graphicsSceneMouseEvent)
        {
            Q_ASSERT(graphicsSceneMouseEvent != 0);
            m_type = graphicsSceneMouseEvent->type();
            m_pos = graphicsSceneMouseEvent->pos().toPoint();
            m_button = graphicsSceneMouseEvent->button();
            m_buttons = graphicsSceneMouseEvent->buttons();
            m_modifier = graphicsSceneMouseEvent->modifiers();
            m_scenePos = graphicsSceneMouseEvent->scenePos();
            m_screenPos = graphicsSceneMouseEvent->screenPos();
            m_buttonDownPos = graphicsSceneMouseEvent->buttonDownPos(graphicsSceneMouseEvent->button());
            m_buttonDownScenePos = graphicsSceneMouseEvent->buttonDownScenePos(graphicsSceneMouseEvent->button());
            m_buttonDownScreenPos = graphicsSceneMouseEvent->buttonDownScreenPos(graphicsSceneMouseEvent->button());
            m_lastPos = graphicsSceneMouseEvent->lastPos();
            m_lastScenePos = graphicsSceneMouseEvent->lastScenePos();
            m_lastScreenPos = graphicsSceneMouseEvent->lastScreenPos();
            m_fired = false;
            m_editable = false;
            m_graphicsSceneEvent = true;
        }

    };


    class WebTouchNavigation : public QObject
    {
    Q_OBJECT
    public:
        WebTouchNavigation(QObject* object, QWebPage* webPage);
        virtual ~WebTouchNavigation();

    protected:
        bool eventFilter(QObject *object, QEvent *event);
        void handleDownEvent(const WebTouchEvent &event);
        void handleMoveEvent(const WebTouchEvent &event);
        void handleReleaseEvent(const WebTouchEvent &event);

    public slots:
        void hoverTimerFired();
        void downTimerFired();
        void quickDownTimerFired();
        void quickUpTimerFired();
        void decelerationTimerEvent(int count);

    private:
        void generateMouseEvent(const WebTouchEvent& touchEvent);

    private:
        WebTouchEvent m_touchDown;
        WebTouchEvent m_prevMouseMove;

        QObject* m_viewObject;
        QWebPage* m_webPage;
        QWebFrame* m_webFrame;
        QTimer m_downTimer;
        QTimer m_hoverTimer;
        QTimer m_quickDownTimer;
        QTimer m_quickUpTimer;
        bool m_scrolling;

        QPoint m_decelerationSpeed;
        QTimeLine m_decelerationTimer;
        QPoint m_cumlativeDistance;
    };

}

#endif
