//--------------------------------------------------------------------------------------------------
function CTestTree(title, container)
//--------------------------------------------------------------------------------------------------
{
////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                //
//  This JavaScript class creates a framework where browser and widget testing can be automated.  //
//  When this class is instantiated, a user interface is created where tests can be launched and  //
//  results recorded.                                                                             //
//                                                                                                //
//  To use this class, simply include this JavaScript module in your markup and create a          //
//  CTestTree variable.  No global objects or functions are defined here, so the code can be      //
//  safely included in another application without stepping on the toes of any existing objects.  //
//                                                                                                //
////////////////////////////////////////////////////////////////////////////////////////////////////

//--------------------------------------------------------------------------------------------------
// Constants
//--------------------------------------------------------------------------------------------------

	var _CLASS_NAME_ =                    "CTestTree JavaScript Test Framework";
	var _VERSION_ =                       "v1.9 14-Jul-2008";

	var CHAR_AMP =                        "&";
	var CHAR_BRACE =                      "{";
	var CHAR_COLON =                      ":";
	var CHAR_DASH =                       "-";
	var CHAR_DOT =                        ".";
	var CHAR_LPAREN =                     "(";
	var CHAR_Q =                          "Q";
	var CHAR_RPAREN =                     ")";
	var CHAR_SEMI =                       ";";
	var CHAR_SPACE =                      " ";
	var CHAR_STAR =                       "*";
	var CHAR_ZERO =                       "0";

	var CLASS_BODY =                      "body";
	var CLASS_BUTTON =                    "button";
	var CLASS_BUTTON_ROW =                "button_row";
	var CLASS_CAPTION =                   "caption";
	var CLASS_CATEGORY =                  "category";
	var CLASS_CHECK =                     "check";
	var CLASS_COUNT =                     "count";
	var CLASS_ERROR =                     "error";
	var CLASS_FAIL =                      "fail";
	var CLASS_INPUT =                     "input";
	var CLASS_PASS =                      "pass";
	var CLASS_RESULTS =                   "results";
	var CLASS_RUN =                       "run";
	var CLASS_SKIP =                      "skip";
	var CLASS_STOP =                      "stop";
	var CLASS_TABLE =                     "table";
	var CLASS_TD =                        "td";
	var CLASS_TEST =                      "test";

	var CONTENT_HEADER =                  "Content-Type";
	var CONTENT_TYPE =                    "application/x-www-form-urlencoded";

	var DEFAULT_CAPTION =                 "Automated Test";
	var DEFAULT_TIMEOUT =                 100;
	var DEFAULT_URL =                     "http://waplabdc.nokia-boston.com/browser/users/TestResults/PostTestResults.php";

	var ELEMENT_BUTTON =                  "button";
	var ELEMENT_CELL =                    "td";
	var ELEMENT_CHECK =                   "checkbox";
	var ELEMENT_DIV =                     "div";
	var ELEMENT_INPUT =                   "input";
	var ELEMENT_LABEL =                   "label";
	var ELEMENT_ROW =                     "tr";
	var ELEMENT_TABLE =                   "table";

	var EMPTY_STRING =                    "";

	var ERROR_BAD_DELAY =                 "The delay value (in milliseconds) must be an integer greater than zero";
	var ERROR_CANT_DO_AJAX =              "Cannot create an XMLHTTP object";
	var ERROR_OPERATOR_CANCEL =           "Cancelled by operator";

	var EVENT_CLICK =                     "click";

	var ID_BUTTONS =                      "buttons";
	var ID_BUTTON_RUN =                   "runButton";
	var ID_BUTTON_STOP =                  "stopButton";
	var ID_CLASS =                        "CTestTree";
	var ID_ERROR =                        "error";
	var ID_FAIL_COUNT =                   "failCount";
	var ID_HEADER =                       "header";
	var ID_PASS_COUNT =                   "passCount";
	var ID_RESULTS =                      "_results";
	var ID_SKIP_COUNT =                   "skipCount";

	var LABEL_FAIL =                      "FAIL";
	var LABEL_PASS =                      "PASS";
	var LABEL_RUN =                       "Run";
	var LABEL_SKIP =                      "SKIP";
	var LABEL_STOP =                      "Stop";

	var LINK_HEADER =                     "Log=";
	var LINK_METHOD =                     "POST";
	var LINK_NAME =                       "Name=";

	var LOG_EXECUTED =                    "      Tests Executed: ";
	var LOG_FAILED =                      "        Tests Failed: ";
	var LOG_MARGIN =                      6;
	var LOG_PASSED =                      "        Tests Passed: ";
	var LOG_SKIPPED =                     "       Tests Skipped: ";
	var LOG_TEST_COUNT =                  "     Number of Tests: ";
	var LOG_TEST_END =                    "End of ";
	var LOG_TEST_START =                  "Start of ";
	var LOG_WIDTH =                       80;

	var MENU_ITEM_ABOUT =                 "About Test Tree...";
	var MENU_ITEM_ALL =                   "All";
	var MENU_ITEM_ASC =                   "Ascending";
	var MENU_ITEM_CLEAR =                 "Clear results";
	var MENU_ITEM_DELAY =                 "Set test delay...";
	var MENU_ITEM_DESC =                  "Descending";
	var MENU_ITEM_HIDE =                  "Hide UI";
	var MENU_ITEM_NONE =                  "None";
	var MENU_ITEM_ROTATE =                "Rotate screen";
	var MENU_ITEM_RUN =                   "Run tests";
	var MENU_ITEM_SELECT =                "Select tests";
	var MENU_ITEM_SEND =                  "Send results to server...";
	var MENU_ITEM_SHOW =                  "Show UI";
	var MENU_ITEM_SORT =                  "Sort tests";
	var MENU_ITEM_STOP =                  "Stop tests";
	var MENU_ITEM_TOP =                   "Test Tree";

	var MENU_NUMBER_ABOUT =               124100;
	var MENU_NUMBER_ALL =                 123320;
	var MENU_NUMBER_ASC =                 123720;
	var MENU_NUMBER_CLEAR =               123500;
	var MENU_NUMBER_DELAY =               124000;
	var MENU_NUMBER_DESC =                123740;
	var MENU_NUMBER_HIDE =                123800;
	var MENU_NUMBER_NONE =                123340;
	var MENU_NUMBER_ROTATE =              123600;
	var MENU_NUMBER_RUN =                 123100;
	var MENU_NUMBER_SELECT =              123300;
	var MENU_NUMBER_SEND =                123400;
	var MENU_NUMBER_SHOW =                123900;
	var MENU_NUMBER_SORT =                123700;
	var MENU_NUMBER_STOP =                123200;
	var MENU_NUMBER_TOP =                 123000;

	var MIME_TYPE =                       "text/plain";

	var MS_NEW_OBJECT =                   "Msxml2.XMLHTTP";
	var MS_OLD_OBJECT =                   "Microsoft.XMLHTTP";

	var PRIVILEGE_TYPE =                  "UniversalBrowserRead";

	var READY_STATE_COMPLETE =            4;

	var RESULT_MAXIMUM =                  2;
	var RESULT_MINIMUM =                  0;

	var STATE_PAUSED =                    2;
	var STATE_RUNNING =                   1;
	var STATE_STOPPED =                   0;

	var STRING_AM =                       "am";
	var STRING_BREAK =                    "<br/>";
	var STRING_CLASSNAME =                "CTestTree_";
	var STRING_COLON_DELIM =              ": ";
	var STRING_CSS =                      "text/css";
	var STRING_DASHES =                   " -- ";
	var STRING_ENTER_DELAY =              "Please enter a delay between tests (in milliseconds):";
	var STRING_ENTER_URL =                "Server where log results will be posted:";
	var STRING_EVENT_PREFIX =             "on";
	var STRING_FORMAT_DATE =              "_dd _mmm _yyyy";
	var STRING_FORMAT_TIME =              "_HH:_nn:_ss";
	var STRING_FUNCTION =                 "function";
	var STRING_HEAD =                     "head";
	var STRING_INLINE =                   "inline";
	var STRING_LINEFEED =                 "\n";
	var STRING_NONE =                     "none";
	var STRING_NUMBER =                   "number";
	var STRING_PARENS =                   "()";
	var STRING_PERCENTAGE =               " (?%)";
	var STRING_PM =                       "pm";
	var STRING_RUNNING =                  "Running...";
	var STRING_STARS =                    "**** ";
	var STRING_STRING =                   "string";
	var STRING_STYLE =                    "style";
	var STRING_STYLESHEET =               "TestTreeStylesheet";
	var STRING_UNKNOWN =                  "Results Unknown";

	var TEST_FAILED =                     0;
	var TEST_PASSED =                     1;
	var TEST_PENDING =                    4;
	var TEST_SKIPPED =                    2;
	var TEST_STOPPED =                    3;

	var TIMEOUT_QUICK =                   10;

// Constant Arrays

	var ARRAY_CLASSES = new Array(        CLASS_FAIL,
                                        CLASS_PASS,
                                        CLASS_SKIP,
                                        CLASS_CAPTION,
                                        CLASS_TEST);

	var ARRAY_RESULTS = new Array(        "Fail ",
                                        "Pass ",
                                        "Skip ",
                                        "Tests cancelled by operator",
                                        " Pending");

	var ARRAY_WEEKDAYS = new Array (      "Sunday",
                                        "Monday",
                                        "Tuesday",
                                        "Wednesday",
                                        "Thursday",
                                        "Friday",
                                        "Saturday");

	var ARRAY_MONTHS = new Array (        "January",
                                        "February",
                                        "March",
                                        "April",
                                        "May",
                                        "June",
                                        "July",
                                        "August",
                                        "September",
                                        "October",
                                        "November",
                                        "December");
										
										
// Variables defined for creating the logs
	var logDir = "/data/others/cjse_logs";
	var logEncoding = "UTF-8";
	var xmlExtn = ".xml";
	var logfileObj = null;
	var logFile = null;
	var logSo = null;
	var logContinue = new Boolean(false);
					
    var autoExitTimeout = 20000; //default timeout is 20s						

// Create a two-dimensional array of class names and style definitions.

		var ARRAY_CSS = new Array(
      ["body",       " { font-family: verdana, arial, helvetica; display: inline; white-space: nowrap; }"],
      ["button",     " { font-family: verdana, arial, helvetica; margin-right: .5em; }"],
      ["button_row", " { background-color: #AAAAAA; border-top: 1px solid #000000; text-align: right; }"],
      ["caption",    " { border-bottom: 0px; background-color: #0000FF; color: #FFFFFF; }"],
      ["category",   " { background-color: #AAAAFF; color: #000000; }"],
      ["check",      " { margin-right: .5em; }"],
      ["count",      " { border-bottom: 0px; text-align: right; }"],
      ["error",      " { background-color: #0000FF; border-bottom: 0px; color: #FFFF00; font-weight: bold; }"],
      ["fail",       " { background-color: #FF2222; color: #FFFFFF; }"],
      ["input",      " { font-family: verdana, arial, helvetica; }"],
      ["pass",       " { background-color: #00AA00; color: #FFFFFF; }"],
      ["run",        " { background-color: #00AA00; color: #FFFFFF; }"],
      ["skip",       " { background-color: #FFFF00; color: #000000; }"],
      ["stop",       " { background-color: #FF2222; color: #FFFFFF; }"],
      ["table",      " { border: 1px solid #000000; border-bottom: 0px; border-spacing: 0px; font-family: verdana, arial, helvetica; white-space: nowrap; }"],
      ["td",         " { border-bottom: 1px solid #000000; margin: 0em; padding: .25em .5em; vertical-align: middle; }"],
      ["test",       " { background-color: #CCCCCC; color: #000000; }"]
		);

//--------------------------------------------------------------------------------------------------
// Document object with get, add and remove methods.
//--------------------------------------------------------------------------------------------------

	var Dom = {
		get: function(el) {
			if(typeof el === STRING_STRING) {
				return document.getElementById(el);
			} else {
				return el;
			}
		},
		add: function(el, dest) {
			var el = this.get(el);
			var dest = this.get(dest);
			dest.appendChild(el);
		},
		remove: function(el) {
			var el = this.get(el);
			el.parentNode.removeChild(el);
		}
	};

//--------------------------------------------------------------------------------------------------
// Event object with an add method.
//--------------------------------------------------------------------------------------------------

	var Event = {
		add: function() {
			if(window.addEventListener) {
				return function(el, type, fn) {
					Dom.get(el).addEventListener(type, fn, false);
				};
			} else if(window.attachEvent) {
				return function(el, type, fn) {
					var f = function() {
						fn.call(Dom.get(el), window.event);
					};
					Dom.get(el).attachEvent(STRING_EVENT_PREFIX + type, f);
				};
			}
		}()
	};

//--------------------------------------------------------------------------------------------------
// String prototype functions
//--------------------------------------------------------------------------------------------------

	String.prototype.pad = function(size, fill) {
		var t = this.toString();
		var p = fill ? fill.toString().repeat(size) : CHAR_SPACE.repeat(size);

		if(fill === 0) p = String(CHAR_ZERO).repeat(size);

		if(t.length >= size) return t;
		return (p + t).substr((p + t).length - size);
	}

	String.prototype.repeat = function(n) {
		var s = EMPTY_STRING;
		var t = this.toString();

		while (--n >= 0) s += t;
		return s;
	}

	String.prototype.trim = function() {
		return this.replace(/^\s\s*/, EMPTY_STRING).replace(/\s\s*$/, EMPTY_STRING);
	}

	String.prototype.trimL = function() {
		return this.replace(/^\s+/, EMPTY_STRING);
	}

	String.prototype.trimR = function() {
		return this.replace(/\s+$/, EMPTY_STRING);
	}

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                //
//                                        Class Definition                                        //
//                                                                                                //
////////////////////////////////////////////////////////////////////////////////////////////////////

	var callback = null;                                      // optional user function
	var caption = title ? title : DEFAULT_CAPTION;            // caption of the Test Tree
	var countFail = 0;                                        // number of failed tests
	var countPass = 0;                                        // number of passed tests
	var countSkip = 0;                                        // number of skipped tests
	var div = null;                                           // container for this class
	var logEnabled = true;                                    // whether or not logging is enabled
	var logFileName = EMPTY_STRING;                           // optional name of log file
	var pending = new Object;                                 // list of tests with pending results
	var portrait = true;                                      // current orientation
	var record = EMPTY_STRING;                                // record of all log entries
	var resultsCell = EMPTY_STRING;                           // cell containing test results
	var state = STATE_STOPPED;                                // state of the test run
	var startTime = null;                                     // start time of test run
	var table = null;                                         // HTML table that holds the data
	var testCount = 0;                                        // total number of tests
	var testDelay = DEFAULT_TIMEOUT;                          // inter-test delay in milliseconds
	var testNumber = -1;                                      // current test number
	var timer = 0;                                            // test loop timer
	var userCancelled = false;                                // flag to indicate user cancellation

// Set a pointer to this for use by other functions.

	var self = this;

// Create a new stylesheet for items that will be added to the document.

	addStylesheet();

// Create entries for the widget options menu.

	if(window.menu) menuCreate();

// Add the visible elements to the document.

	addTable();

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                //
//                                         Public Methods                                         //
//                                                                                                //
////////////////////////////////////////////////////////////////////////////////////////////////////

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.initLogging = function()
//--------------------------------------------------------------------------------------------------
{
//After adding all the testcase now preparing the service object and file for result logging
	try{
		logSo = nokia.device.load("filesystem",null);
	}
	catch(e)
	{
		this.debug("File system load failed: Logging disabled");
		logContinue = false;
		return;
	}
	if( navigator.appVersion.toLowerCase().indexOf("linux") != -1 )
	  {
	  	//maemo platform
	  	var mountdir = "file:///home/user/MyDocs";
			var createdirpath = mountdir+logDir;

      try{
			  logSo.createDir(createdirpath);
			  logFile = createdirpath + "/" + title + xmlExtn;
			  logfileObj = logSo.openFile(logFile, "w", logEncoding);
			}
			catch(e)
			{
				logContinue = false;
				return;
			}
	  }
	else
	  {
	  	//S60 platform
			var mountpointdata = logSo.getMountPoints();
			var mountdir = mountpointdata[0].uri
			var createdirpath = mountdir+logDir;
		
		  try{
			  logSo.createDir(createdirpath);
			  logFile = createdirpath + "/" + title + xmlExtn;
			  logfileObj = logSo.openFile(logFile, "w", logEncoding);
			}
			catch(e)
			{
				logContinue = false;
				return;
			}
		}
	logfileObj.write("<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?>"+"\n");
	logfileObj.write("<testset id=\""+title+"\">"+"\n");		
	logfileObj.close();
	logContinue = true;
}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.debug = function(msg)
//--------------------------------------------------------------------------------------------------
	{
		//alert(msg);
		if(logSo && logContinue){
			try {
			    logfileObj = logSo.openFile(logFile, "a", logEncoding);
			  }
			  catch(e)
			  {
			  	logContinue = false;
			  	return;
			  }
			//alert("Inside DisplayResults");
			logfileObj.write("<debug msg=\""+msg+"\"/>"+"\n");
			logfileObj.close();
		}
  }
  
//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.setExitTimeout = function(timeout)
//--------------------------------------------------------------------------------------------------
{
	if (timeout != null && typeof timeout != "undefined") {
		autoExitTimeout = timeout;
	}
}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.addTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
// Either a string or an array can be passed in, so force a single name into an array.

		if(typeof(names) == STRING_STRING) names = new Array(names);
		if(typeof(names) == STRING_FUNCTION) names = new Array(getFunctionName(names));

		for(var i = 0; i < names.length; i++) {
			testCount++;

// Get both the function name (for display purposes) and the full calling syntax (for execution).

			var testName = getFunctionName(names[i]);
			var testCall = getFunctionCall(names[i]);

			var row = addRow();
			row.id = ELEMENT_ROW + testCall;
			row.className = CLASS_TEST;

			var cell = addTest(row, testName, testCall);

			var results = document.createElement(ELEMENT_CELL);
			results.colSpan = 3;
			results.id = testCall + ID_RESULTS;
			results.className = CLASS_TD + CHAR_SPACE + CLASS_RESULTS;
			Dom.add(results, row);
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.addTestsByPrefix = function(prefix)
//--------------------------------------------------------------------------------------------------
	{
		for(var i in window) {
			try {
				if(typeof(window[i]) == STRING_FUNCTION) {
					if(i.substr(0,prefix.length) == prefix) {
						this.addTests(i);
					}
				}
			} catch(er) {
				// Do nothing
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.clearResults = function()
//--------------------------------------------------------------------------------------------------
	{
		record = EMPTY_STRING;
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.continueTests = function(name)
//--------------------------------------------------------------------------------------------------
	{
		if(state == STATE_PAUSED) loopTests();
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.deselectTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
// Either a string or an array can be passed in, so force a single name into an array.

		if(names) {
			if(typeof(names) == STRING_STRING) names = new Array(names);

			for(var i in names) if(l(getFunctionCall(names[i]))) l(getFunctionCall(names[i])).checked = false;

// If no names are passed in, deselect all tests.

		} else {
			var checks = getElementsByClassName(CLASS_CHECK);

			for(var i in checks) {
				if(checks[i].type == ELEMENT_CHECK) checks[i].checked = false;
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.displayResults = function(results, display, id)
//--------------------------------------------------------------------------------------------------
	{
		var logString = STRING_UNKNOWN;
		var newClass = EMPTY_STRING;
		var thisCell = resultsCell;

// This function should only be called once for a given test, unless results were pending.

		if(id) {
			if(pending[id]) {
				thisCell = pending[id];
				logString = STRING_STARS;
			}
		} else {
			if((l(thisCell).innerHTML) && (l(thisCell).innerHTML != STRING_RUNNING)) return;
		}

		var testCall = thisCell.replace(ID_RESULTS, EMPTY_STRING);

		if(results === false) results = TEST_FAILED;
		if(results === true) results = TEST_PASSED;

		if(typeof(results) == STRING_NUMBER) {
			results = parseInt(results);

			if((results >= RESULT_MINIMUM) || (results <= RESULT_MAXIMUM)) {
				logString = ARRAY_RESULTS[results];
				newClass = ARRAY_CLASSES[results];
			}
		}

		if(l(thisCell)) {
			l(thisCell).innerHTML = display ? display : logString;
			
//Changed By Javed:

	if(logSo && logContinue){
		if(results ==0 || results ==1)
		{
			var tcresult = null;
			try{
			   logfileObj = logSo.openFile(logFile, "a", logEncoding);
			 }
			 catch(e)
			 {
			 	logContinue = false;
			 	return;
			 }
			//alert("Inside DisplayResults");
			//alert(testCall);
			if(results ==0)	tcresult = "FAIL";
			if(results ==1)	tcresult = "PASS";
			//alert(display);
			logfileObj.write("<test name=\""+testCall+"\" "+"result=\""+tcresult+"\" "+"msg=\""+display+"\"/>"+"\n");
			logfileObj.close();
		}
}
				
			

// Log the results.

			if(results != TEST_STOPPED) logString += testCall;
			if(display) logString = logString.trim() + STRING_COLON_DELIM + display;
			recordResults(logString);

// Change the color of this row and update the totals.

			if(newClass != EMPTY_STRING) l(ELEMENT_ROW + testCall).className = newClass;

			if(results == TEST_FAILED) {
				if(isNumeric(l(ID_FAIL_COUNT).innerHTML)) l(ID_FAIL_COUNT).innerHTML++;
			} else if(results == TEST_PASSED) {
				if(isNumeric(l(ID_PASS_COUNT).innerHTML)) l(ID_PASS_COUNT).innerHTML++;
			} else if(results == TEST_SKIPPED) {
				if(isNumeric(l(ID_SKIP_COUNT).innerHTML)) l(ID_SKIP_COUNT).innerHTML++;
			}
		}

		if(id) {
			if(pending[id]) {
				delete pending[id];
				if(!timer) loopTests();
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.hideUI = function()
//--------------------------------------------------------------------------------------------------
	{
		getElementsByClassName(CLASS_BODY)[0].style.display = STRING_NONE;

		if(window.menu) {
			var menuParent = window.menu.getMenuItemById(MENU_NUMBER_HIDE);
			menuParent.setDimmed(true);
			menuParent = window.menu.getMenuItemById(MENU_NUMBER_SHOW);
			menuParent.setDimmed(false);
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.logRename = function(newName)
//--------------------------------------------------------------------------------------------------
	{
		logFileName = newName;
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.logStart = function()
//--------------------------------------------------------------------------------------------------
	{
		logEnabled = true;
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.logStop = function()
//--------------------------------------------------------------------------------------------------
	{
		logEnabled = false;
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.logTrace = function(item)
//--------------------------------------------------------------------------------------------------
	{
		recordResults(item);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.pauseTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
		if(state == STATE_RUNNING) {
			if(timer) clearTimeout(timer);
			timer = 0;
			state = STATE_PAUSED;
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.readResults = function()
//--------------------------------------------------------------------------------------------------
	{
		return(record);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.removeTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
// Either a string or an array can be passed in, so force a single name into an array.

		if(names) {
			if(typeof(names) == STRING_STRING) names = new Array(names);

// If no names are passed in, remove all tests.

		} else {
			Dom.remove(div);
			addTable();
			testCount = 0;
			return;
		}

		for(var i = 0; i < names.length; i++) {
			try {
				var target = ELEMENT_ROW + names[i].id;
				Dom.remove(target);
				testCount--;
			} catch(er) {
				// Do nothing
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.rotateScreen = function()
//--------------------------------------------------------------------------------------------------
	{
		if(window.widget) {
			if(portrait) {
				portrait = false;
				widget.setDisplayLandscape();
			} else {
				portrait = true;
				widget.setDisplayPortrait();
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.runTests = function(callbackFunction)
//--------------------------------------------------------------------------------------------------
	{
		if(state != STATE_STOPPED) return;

// If callbackFunction is passed in, store it to call it at the end of the run.

		callback = callbackFunction ? callbackFunction: null;
		countFail = 0;
		countPass = 0;
		countSkip = 0;
		testNumber = -1;
		timer = 0;
		userCancelled = false;

		startTime = new Date();

		l(ID_BUTTON_RUN).style.display = STRING_NONE;
		l(ID_BUTTON_STOP).style.display = STRING_INLINE;

		if(window.menu) {
			var menuParent = window.menu.getMenuItemById(MENU_NUMBER_RUN);
			menuParent.setDimmed(true);
			menuParent = window.menu.getMenuItemById(MENU_NUMBER_STOP);
			menuParent.setDimmed(false);
		}

// Log a console entry to indicate the start of the test.

		var header = LOG_TEST_START + caption + STRING_DASHES + fmtDateTime(STRING_FORMAT_DATE);
		var headerWidth = header.length + LOG_MARGIN;

		if(headerWidth < LOG_WIDTH) headerWidth = LOG_WIDTH;

		var margin = parseInt((headerWidth - header.length) / 2);

		recordResults(CHAR_SPACE);
		recordResults(CHAR_DASH.repeat(headerWidth));
		recordResults(CHAR_SPACE.repeat(margin) + header);
		recordResults(CHAR_DASH.repeat(headerWidth));
		recordResults(CHAR_SPACE);

// Clear the results from any previous run.

		l(ID_PASS_COUNT).innerHTML = CHAR_ZERO;
		l(ID_FAIL_COUNT).innerHTML = CHAR_ZERO;
		l(ID_SKIP_COUNT).innerHTML = CHAR_ZERO;
		l(ID_ERROR).innerHTML = EMPTY_STRING;

// Remove any previous results and reset the color of each row.

		var results = getElementsByClassName(CLASS_RESULTS);

		for(var i = 0; i < results.length; i++) {
			results[i].innerHTML = EMPTY_STRING;
		}

		var rows = document.getElementsByTagName(ELEMENT_ROW);

		for(i = 0; i < rows.length; i++) {
			if(rows[i].parentNode.id == table.id) {
				if((rows[i].id != ID_HEADER) && (rows[i].id != ID_BUTTONS)) {
					rows[i].className = CLASS_TEST;
				}
			}
		}

		loopTests();
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.selectTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
// Either a string or an array can be passed in, so force a single name into an array.

		if(names) {
			if(typeof(names) == STRING_STRING) names = new Array(names);

			for(var i in names) if(l(getFunctionCall(names[i]))) l(getFunctionCall(names[i])).checked = true;

		} else {
			var checks = getElementsByClassName(CLASS_CHECK);

			for(var i in checks) {
				if(checks[i].type == ELEMENT_CHECK) checks[i].checked = true;
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.sendResults = function(url)
//--------------------------------------------------------------------------------------------------
	{
		var httpRequest = false;
		var sendAsynchronously = true;

// Send the log contents to an external server.

		if (url) {
			var serverURL = url;
		} else {
			var serverURL = DEFAULT_URL;
		}

// Mozilla needs to be told to allow cross-domain scripting.

			try {
				netscape.security.PrivilegeManager.enablePrivilege(PRIVILEGE_TYPE);
			} catch (err) {
				// Do nothing
	 		}

// Set up the communications based on the browser type.

		if (!httpRequest) {
			if (window.XMLHttpRequest) {

// We've found a Mozilla-type browser.

				httpRequest = new XMLHttpRequest();

				if (httpRequest.overrideMimeType) {
					httpRequest.overrideMimeType(MIME_TYPE);
					}
			} else if (window.ActiveXObject) {

// If this is IE, try to create the newest object version first.

				try {
					httpRequest = new ActiveXObject(MS_NEW_OBJECT);
				} catch (err) {
					try {
						httpRequest = new ActiveXObject(MS_OLD_OBJECT);
					} catch (err) {
						self.logTrace(err);
						return false;
					}
				}
			}
		}

// Quit here if the request failed.

		if (!httpRequest) {
			self.logTrace(ERROR_CANT_DO_AJAX);
			return false;
		}

// Set up a callback that the server will use to send us the results.

		try {
			httpRequest.onreadystatechange = sendResultsCallback;
		} catch(err) {
			self.logTrace(err);
			return false;
		}

// Open the link.

		try {
			httpRequest.open(LINK_METHOD, serverURL, sendAsynchronously);
		} catch(err) {
				self.logTrace(err);
				return false;
		}

// This content type works for text-based data.

		try {
			httpRequest.setRequestHeader(CONTENT_HEADER, CONTENT_TYPE);
		} catch(err) {
			self.logTrace(err);
			return false;
		}

// If the log file name has been specified, send it along with the log data.

		var logData = "";

		if(logFileName) logData = LINK_NAME + logFileName + CHAR_AMP;
		logData += LINK_HEADER + record;

// Now go ahead and send the data that we want to log.

		try {
			httpRequest.send(encodeURI(logData));
		} catch(err) {
			self.logTrace(err);
			return false;
		}

		return true;
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.setDelay = function(delay)
//--------------------------------------------------------------------------------------------------
	{
		if(parseInt(delay) == delay) {
			if(parseInt(delay) > 0) {
				testDelay = delay;
				return(true);
			}
		}

		return(false);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.showUI = function()
//--------------------------------------------------------------------------------------------------
	{
		getElementsByClassName(CLASS_BODY)[0].style.display = STRING_INLINE;

		if(window.menu) {
			var menuParent = window.menu.getMenuItemById(MENU_NUMBER_SHOW);
			menuParent.setDimmed(true);
			menuParent = window.menu.getMenuItemById(MENU_NUMBER_HIDE);
			menuParent.setDimmed(false);
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.skipTests = function(names)
//--------------------------------------------------------------------------------------------------
	{
// Either a string or an array can be passed in, so force a single name into an array.

		if(typeof(names) == STRING_STRING) names = new Array(names);

		for(var i = 0; i < names.length; i++) {

// The id for the checkbox is the full calling syntax of the function.

			var testCall = getFunctionCall(names[i]);

			try {
				l(testCall).checked = false;
			} catch(er) {
				// Do nothing
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.skipTo = function(name)
//--------------------------------------------------------------------------------------------------
	{
		if(state == STATE_STOPPED) return;
		if(timer) clearTimeout(timer);
		timer = 0;

		var testCall = getFunctionCall(name);
		var checks = getElementsByClassName(CLASS_CHECK);

		for(var i = 0; i < checks.length; i++) {
			testNumber = i;

// The id for the checkbox is the full calling syntax of the function.

			if(checks[i].id == testCall) {
				testNumber = i - 1;
				loopTests();
				return;
			} else {

// If we haven't recorded any results for this test already, record a Skip now.

				resultsCell = checks[i].id + ID_RESULTS;
				var currentRow = ELEMENT_ROW + checks[i].id;
				if(l(currentRow).className.indexOf(CLASS_TEST) != -1) self.displayResults(TEST_SKIPPED);
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.sortTests = function(desc)
//--------------------------------------------------------------------------------------------------
	{
		if(state != STATE_STOPPED) return;
		if(testCount < 2) return;

		var checks = getElementsByClassName(CLASS_CHECK);
		var names = new Array();

		for(var i = 0; i < checks.length; i++) {
			names.push(getFunctionName(checks[i].id));
		}

		Dom.remove(div);
		addTable();

		callback = null;
		countFail = 0;
		countPass = 0;
		countSkip = 0;
		state = STATE_STOPPED;
		testCount = 0;
		testNumber = -1;
		timer = 0;

		if(desc) {
			names = names.sort(sortDescending);
		} else {
			names = names.sort();
		}

		self.addTests(names);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.stopTests = function(runCallback)
//--------------------------------------------------------------------------------------------------
	{
		if(state == STATE_STOPPED) return;
		l(ID_ERROR).innerHTML = ARRAY_RESULTS[TEST_STOPPED];
		endTests(runCallback);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.version = function()
//--------------------------------------------------------------------------------------------------
	{
		return(_VERSION_);
	}

//--------------------------------------------------------------------------------------------------
	CTestTree.prototype.waitForResults = function(id, description)
//--------------------------------------------------------------------------------------------------
	{
		if(state == STATE_STOPPED) return;

		var testCall = resultsCell.replace(ID_RESULTS, EMPTY_STRING);

		l(resultsCell).innerHTML = description ? description : ARRAY_RESULTS[TEST_PENDING];
		l(ELEMENT_ROW + testCall).className = ARRAY_CLASSES[TEST_PENDING];

		var logString = STRING_STARS + getFunctionName(testCall) + ARRAY_RESULTS[TEST_PENDING];
		if(description) logString += STRING_COLON_DELIM + description;
		recordResults(logString);

// Store the value of resultsCell under the user's id so we can display results when they come.

		pending[id] = resultsCell;
	}

////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                //
//                                         Private Methods                                        //
//                                                                                                //
////////////////////////////////////////////////////////////////////////////////////////////////////

//--------------------------------------------------------------------------------------------------
	function addButtons(row)
//--------------------------------------------------------------------------------------------------
	{
		var cell = document.createElement(ELEMENT_CELL);
		cell.colSpan = 4;
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_BUTTON_ROW;
		Dom.add(cell, row);

// Add the run button.

		var button = document.createElement(ELEMENT_INPUT);
		button.id = ID_BUTTON_RUN;
		button.type = ELEMENT_BUTTON;
		button.value = LABEL_RUN;
		button.className = CLASS_INPUT + CHAR_SPACE + CLASS_RUN;
		Dom.add(button, cell);

		Event.add(button, EVENT_CLICK, function() {
			self.runTests();
		});

// Add the stop button.

		button = document.createElement(ELEMENT_INPUT);
		button.id = ID_BUTTON_STOP;
		button.type = ELEMENT_BUTTON;
		button.value = LABEL_STOP;
		button.className = CLASS_INPUT + CHAR_SPACE + CLASS_STOP;
		button.style.display = STRING_NONE;
		Dom.add(button, cell);

		Event.add(button, EVENT_CLICK, function() {
			userCancelled = true;
			self.stopTests();
		});
	}

//--------------------------------------------------------------------------------------------------
	function addCell(row, caption, id)
//--------------------------------------------------------------------------------------------------
	{
		var cell = document.createElement(ELEMENT_CELL);
		if(id) cell.id = id;
		Dom.add(cell, row);

		if(caption) {
			var text = document.createTextNode(caption);
			Dom.add(text, cell);
		}

		return cell;
	}

//--------------------------------------------------------------------------------------------------
	function addRow()
//--------------------------------------------------------------------------------------------------
	{
		var row = document.createElement(ELEMENT_ROW);
		Dom.add(row, table);

		return row;
	}

//--------------------------------------------------------------------------------------------------
	function addStylesheet()
//--------------------------------------------------------------------------------------------------
	{
// This function adds styles for the objects in this class while ensuring unique class names.

		var counter = 0;
		var currentRules = new Array();
		var prefix = STRING_CLASSNAME;
		var sheets = document.styleSheets;

// Create an array of the currently-defined stylesheet rule names.

		for(var i in sheets) {

			if(sheets[i].cssRules) {
				var rules = sheets[i].cssRules;
			} else {
				var rules = sheets[i].rules;
			}

			for(var j in rules) {
				if(rules.selectorText) currentRules.push(rules.selectorText);
			}
		}

// Now add our list of styles to the document.

		var newStyle = document.createElement(STRING_STYLE);
		newStyle.type = STRING_CSS;
		newStyle.title = uniqueId(STRING_STYLESHEET);
		document.getElementsByTagName(STRING_HEAD)[0].appendChild(newStyle);
		newStyle = document.getElementsByTagName(STRING_HEAD)[0].lastChild.sheet;

		for(i in ARRAY_CSS) {
			var newClass = CHAR_DOT + prefix + ARRAY_CSS[i][0];

// Increment the name until we find one that is unique.

			while(currentRules[newClass]) newClass = CHAR_DOT + prefix + ++counter + ARRAY_CSS[i][0];

// JavaScript doesn't really have constants, so we can update these with the new values.

			ARRAY_CSS[i][0] = newClass;
		}

		for(i in ARRAY_CSS) {

			try {
				if(newStyle.insertRule) {
					newStyle.insertRule(ARRAY_CSS[i][0] + ARRAY_CSS[i][1], 0);
				}
			} catch(er) {
				try {
					if(newStyle.addRule) {
						newStyle.addRule(ARRAY_CSS[i][0], ARRAY_CSS[i][1]);
					}
				} catch(er) {}
			}
		}

// Assign the newly-created class names to the class variables.

		CLASS_BODY =       ARRAY_CSS[0][0].substr(1);
		CLASS_BUTTON =     ARRAY_CSS[1][0].substr(1);
		CLASS_BUTTON_ROW = ARRAY_CSS[2][0].substr(1);
		CLASS_CAPTION =    ARRAY_CSS[3][0].substr(1);
		CLASS_CATEGORY =   ARRAY_CSS[4][0].substr(1);
		CLASS_CHECK =      ARRAY_CSS[5][0].substr(1);
		CLASS_COUNT =      ARRAY_CSS[6][0].substr(1);
		CLASS_ERROR =      ARRAY_CSS[7][0].substr(1);
		CLASS_FAIL =       ARRAY_CSS[8][0].substr(1);
		CLASS_INPUT =      ARRAY_CSS[9][0].substr(1);
		CLASS_PASS =       ARRAY_CSS[10][0].substr(1);
		CLASS_RUN =        ARRAY_CSS[11][0].substr(1);
		CLASS_SKIP =       ARRAY_CSS[12][0].substr(1);
		CLASS_STOP =       ARRAY_CSS[13][0].substr(1);
		CLASS_TABLE =      ARRAY_CSS[14][0].substr(1);
		CLASS_TD =         ARRAY_CSS[15][0].substr(1);
		CLASS_TEST =       ARRAY_CSS[16][0].substr(1);

// Note that CLASS_RESULTS is a dummy class just used to identify the results column.

		ARRAY_CLASSES =    new Array(
		                   CLASS_FAIL,
                       CLASS_PASS,
                       CLASS_SKIP,
                       CLASS_CAPTION,
                       CLASS_TEST);
	}

//--------------------------------------------------------------------------------------------------
	function addTable()
//--------------------------------------------------------------------------------------------------
	{
// The div contains all of the UI elements of this class.

		div = document.createElement(ELEMENT_DIV);
		div.id = uniqueId(ID_CLASS);
		div.className = CLASS_BODY;
		Dom.add(div, container ? container : document.body);

// The table displays the visible interface.

		table = document.createElement(ELEMENT_TABLE);
		table.id = uniqueId(ELEMENT_TABLE);
		table.className = CLASS_TABLE;
		Dom.add(table, div);

// Set up the header rows to contain pass/fail/skip totals.

		var row = addRow();
		row.id = ID_HEADER;
		var cell = addCell(row, caption);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_CAPTION;

		cell = addCell(row, LABEL_PASS);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_PASS + CHAR_SPACE + CLASS_COUNT;
		cell = addCell(row, LABEL_FAIL);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_FAIL + CHAR_SPACE + CLASS_COUNT;
		cell = addCell(row, LABEL_SKIP);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_SKIP + CHAR_SPACE + CLASS_COUNT;

		row = addRow();
		row.className = CLASS_CAPTION;
		cell = addCell(row, EMPTY_STRING, ID_ERROR);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_ERROR;

		cell = addCell(row, CHAR_ZERO, ID_PASS_COUNT);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_PASS + CHAR_SPACE + CLASS_COUNT;
		cell = addCell(row, CHAR_ZERO, ID_FAIL_COUNT);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_FAIL + CHAR_SPACE + CLASS_COUNT;
		cell = addCell(row, CHAR_ZERO, ID_SKIP_COUNT);
		cell.className = CLASS_TD + CHAR_SPACE + CLASS_SKIP + CHAR_SPACE + CLASS_COUNT;

		row = addRow();
		row.id = ID_BUTTONS;
		row.className = CLASS_BUTTON_ROW;
		addButtons(row);
	}

//--------------------------------------------------------------------------------------------------
	function addTest(row, caption, id)
//--------------------------------------------------------------------------------------------------
	{
		var cell = document.createElement(ELEMENT_CELL);
		cell.className = CLASS_TD;
		Dom.add(cell, row);

// Include a checkbox with each test name.

		var check = document.createElement(ELEMENT_INPUT);

		if(id) check.id = id;
		check.name = caption;
		check.type = ELEMENT_CHECK;
		check.className = CLASS_INPUT + CHAR_SPACE + CLASS_CHECK;
		Dom.add(check, cell);

// Setting the text as a label allows the checkbox to be selected by clicking the text.

		var label = document.createElement(ELEMENT_LABEL);
		label.htmlFor = check.id;
		Dom.add(label, cell);

		if(id != caption) {
			var text = document.createTextNode(id);
		} else {
			var text = document.createTextNode(caption);
		}

		Dom.add(text, label);

		return cell;
	}

//--------------------------------------------------------------------------------------------------
	function calculateTotals()
//--------------------------------------------------------------------------------------------------
	{
		var endTime = new Date();
		var elapsed = endTime.getTime() - startTime.getTime();

		var hrs = parseInt(elapsed / 3600000);
		elapsed = elapsed - (hrs * 3600000);
		var mns = parseInt(elapsed / 60000);
		elapsed = elapsed - (mns * 60000);
		var sec = parseInt(elapsed / 1000);
		elapsed = zeroPad(hrs, 2) + CHAR_COLON + zeroPad(mns, 2) + CHAR_COLON + zeroPad(sec, 2);

		var passCount = parseInt(l(ID_PASS_COUNT).innerHTML);
		var failCount = parseInt(l(ID_FAIL_COUNT).innerHTML);
		var skipCount = parseInt(l(ID_SKIP_COUNT).innerHTML);

		var total = passCount + failCount + skipCount;
		var passPct = 0;
		var failPct = 0;
		var skipPct = 0;
		var runPct = 0;

		if(total < testCount) {
			skipCount += (testCount - total);
			total = testCount;
		}

		if(total != 0) {
			passPct = Math.round((passCount * 100) / total);
			failPct = Math.round((failCount * 100) / total);
			skipPct = Math.round((skipCount * 100) / total);
			runPct = Math.round(((passCount + failCount) * 100) / total);
		}

		l(ID_PASS_COUNT).innerHTML = passCount + STRING_PERCENTAGE.replace(/\?/, passPct);
		l(ID_FAIL_COUNT).innerHTML = failCount + STRING_PERCENTAGE.replace(/\?/, failPct);
		l(ID_SKIP_COUNT).innerHTML = skipCount + STRING_PERCENTAGE.replace(/\?/, skipPct);

		recordResults(CHAR_SPACE);
		recordResults(LOG_TEST_COUNT + (testCount.toString()).pad(4));
		recordResults(LOG_EXECUTED + ((passCount + failCount).toString()).pad(4) + (STRING_PERCENTAGE.replace(/\?/, runPct)).pad(7));
		recordResults(CHAR_SPACE);
		recordResults(LOG_PASSED + passCount.toString().pad(4) + (STRING_PERCENTAGE.replace(/\?/, passPct)).pad(7));
		recordResults(LOG_FAILED + failCount.toString().pad(4) + (STRING_PERCENTAGE.replace(/\?/, failPct)).pad(7));
		recordResults(LOG_SKIPPED + skipCount.toString().pad(4) + (STRING_PERCENTAGE.replace(/\?/, skipPct)).pad(7));
		recordResults(CHAR_SPACE);

		if(userCancelled) recordResults(ERROR_OPERATOR_CANCEL);

		var header = LOG_TEST_END + caption + STRING_DASHES + elapsed;
		var headerWidth = header.length + LOG_MARGIN;

		if(headerWidth < LOG_WIDTH) headerWidth = LOG_WIDTH;

		var margin = parseInt((headerWidth - header.length) / 2);

		recordResults(CHAR_DASH.repeat(headerWidth));
		recordResults(CHAR_SPACE.repeat(margin) + header);
		recordResults(CHAR_DASH.repeat(headerWidth));
		recordResults(CHAR_SPACE);
	}

//--------------------------------------------------------------------------------------------------
	function countPending()
//--------------------------------------------------------------------------------------------------
	{
		var count = 0;
		for(var props in pending)
		{
			//alert("Pending:"+pending[props]);
			 count++;
		}
		return(count);
	}

//--------------------------------------------------------------------------------------------------
	function endTests(runCallback)
//--------------------------------------------------------------------------------------------------
	{
//		alert("inside endTests");
		if(timer) clearTimeout(timer);
		timer = 0;

		state = STATE_STOPPED;
		testNumber = getElementsByClassName(CLASS_CHECK).length;
		l(ID_BUTTON_RUN).style.display = STRING_INLINE;
		l(ID_BUTTON_STOP).style.display = STRING_NONE;

		if(window.menu) {
			var menuParent = window.menu.getMenuItemById(MENU_NUMBER_STOP);
			menuParent.setDimmed(true);
			menuParent = window.menu.getMenuItemById(MENU_NUMBER_RUN);
			menuParent.setDimmed(false);
		}

		calculateTotals();

// If the caller requests that we run a function, execute any callback that was passed to runTests().

		if(runCallback) {
			if(callback) {

				try {
					eval(getFunctionCall(callback));
				} catch(er) {
					// Do nothing
				}
			}
		}
//closing the xml log file with the correct closing tags
	logfileObj = logSo.openFile(logFile, "a", logEncoding);
	for (var idx in pending)
	{
	  logfileObj.write("<test name=\""+pending[idx]+"\" "+"result=\""+"Unknown"+"\" "+"msg=\""+"Possibly hanging.."+"\"/>"+"\n");
	}
	var passCount = parseInt(l(ID_PASS_COUNT).innerHTML);
	var failCount = parseInt(l(ID_FAIL_COUNT).innerHTML);
	var skipCount = parseInt(l(ID_SKIP_COUNT).innerHTML);

	var total = passCount + failCount + skipCount;
	logfileObj.write("<summary total="+"\""+total+"\""+" pass="+"\""+passCount+"\""+" fail="+"\""+failCount+"\""+" incomplete="+"\""+skipCount+"\""+"/>"+"\n");
	logfileObj.write("</testset>");
	logfileObj.close();
    window.close();
	}

//--------------------------------------------------------------------------------------------------
	function fmtDateTime()
//--------------------------------------------------------------------------------------------------
	{
/*
	Returns the date and/or time in the specified format.

	Usage:	var returnValue = fmtDateTime(formatString[, timeString]);

	Where:	formatString is any combination of values listed below; this string
						may include other characters, which are returned unaltered.
					timestring is an optional date/time string; if not given, it defaults
						to the current date and time.

	Format Values (case-sensitive)
	==============================
	_h      hour (12-hour clock, 1 through 12)
	_H      hour (24-hour clock, 0 through 23)
	_hh     hour (12-hour clock, zero-padded, 01 through 12)
	_HH     hour (24-hour clock, zero-padded, 00 through 23)
	_hhhh   hour and minute (military time, 0000 through 2359)
	_n      minute (0 through 59)
	_nn     minute (zero-padded, 00 through 59)
	_s      second (0 through 59)
	_ss     second (zero-padded, 00 through 59)
	_a      12-hour clock abbreviation ("a" or "p")
	_A      upper case 12-hour clock abbreviation ("A" or "P")
	_ap     12-hour clock abbreviation ("am" or "pm")
	_AP     upper case 12-hour clock abbreviation ("AM" or "PM")
	_d      day (1 through 31)
	_dd     day (zero-padded, 01 through 31)
	_m      month number (1 through 12)
	_mm     month number (zero-padded, 01 through 12)
	_mmm    month abbreviation ("Jan" through "Dec")
	_MMM    upper case month abbreviation ("JAN" through "DEC")
	_mmmm   month ("January" through "December")
	_MMMM   upper case month ("JANUARY through "DECEMBER")
	_y      year (0 through 99)
	_yy     year (zero-padded, 00 through 99)
	_yyyy   year (four-digit, 0000 thru 9999)
	_w      weekday (1 through 7; 1 = Sunday)
	_www    weekday abbreviation ("Sun" through "Sat")
	_WWW    upper case weekday abbreviation ("SUN" through "SAT")
	_wwww   weekday ("Sunday" through "Saturday")
	_WWWW   upper case weekday ("SUNDAY" through "SATURDAY")
	_k      week number (1 through 52)
	_kk     week number (zero-padded, 01 through 52)
	_q      quarter (1 throgh 4)
	_qq     quarter (Q-prefixed, "Q1" through "Q4")
	_o      ordinal (suffixes the preceeding number: 1st, 2nd, 3rd, etc.)
	_O      upper case ordinal (suffixes the preceeding number: 1ST, 2ND, 3RD, etc.)
	\_      escapes the underscore character

	Examples
	--------
	fmtDateTime("_wwww, _mmmm _d_o") returns "Monday, April 11th"
	fmtDateTime("_hhhh _WWW _dd _MMM _yyyy") returns "1345 MON 11 APR 2007"
	fmtDateTime("Today is _mmmm") returns "Today is Monday"
	fmtDateTime("Today is _mm/_dd/_yy") returns "Today is 04/11/07"
	fmtDateTime("Today is \_ _wwww \_") returns "Today is _ Monday _"
*/

		var today = new Date();
		var format = EMPTY_STRING;

		switch (arguments.length) {
			case 2:
				today = new Date(arguments[1]);
			case 1:
				format = new String(arguments[0]);
				break;
			default:
				return;
		}

		try {
			var hour_h = today.getHours();
			var hour_H = hour_h;
			var hour_hh = zeroPad(hour_h, 2);
			var hour_HH = hour_hh;
			var minute_n = today.getMinutes();
			var minute_nn = zeroPad(minute_n, 2);
			var second_s = today.getSeconds();
			var second_ss = zeroPad(second_s, 2);
			var hour_hhhh = EMPTY_STRING + hour_hh + minute_nn;
			var ampm_ap = STRING_AM;

			if (hour_h > 11) {
				hour_h -= 12;

				if (hour_h == 0) {
					hour_h = 12;
				}

				hour_hh = zeroPad(hour_h, 2);
				ampm_ap = STRING_PM;
			}

			var ampm_AP = ampm_ap.toUpperCase();
			var month_m = today.getMonth() + 1;
			var month_mm = zeroPad(month_m, 2);
			var month_mmmm = ARRAY_MONTHS[month_m - 1];
			var month_MMMM = month_mmmm.toUpperCase();
			var month_mmm = month_mmmm.substr(0, 3);
			var month_MMM = month_mmm.toUpperCase();

			var day_d = today.getDate();
			var day_dd = zeroPad(day_d, 2);

			var year_yyyy = today.getFullYear();
			var year_y = year_yyyy.toString().substr(2, 2) / 1;
			var year_yy = zeroPad(year_y, 2);

			var weekday_w = today.getDay() + 1;
			var weekday_wwww = ARRAY_WEEKDAYS[weekday_w - 1];
			var weekday_WWWW = weekday_wwww.toUpperCase();
			var weekday_www = weekday_wwww.substr(0, 3);
			var weekday_WWW = weekday_www.toUpperCase();

			var oneDay = 1000 * 60 * 60 * 24;
			var jan1 = new Date(year_yyyy, 0, 1);
			var week_k = Math.floor((((today.getTime() - jan1.getTime()) / oneDay) + 6) / 7);
			var week_kk = zeroPad(week_k, 2);

			var quarter_q = Math.ceil((month_m) / 4);
			var quarter_qq = CHAR_Q + quarter_q;

		} catch (err) {
			if (err) return today;
		}

		format = format.replace(/\\_/g, "~�");
		format = format.replace(/_hhhh/g, hour_hhhh);
		format = format.replace(/_hh/g, hour_hh);
		format = format.replace(/_HH/g, hour_HH);
		format = format.replace(/_h/g, hour_h);
		format = format.replace(/_H/g, hour_H);
		format = format.replace(/_nn/g, minute_nn);
		format = format.replace(/_n/g, minute_n);
		format = format.replace(/_ss/g, second_ss);
		format = format.replace(/_s/g, second_s);
		format = format.replace(/_ap/g, ampm_ap);
		format = format.replace(/_AP/g, ampm_AP);
		format = format.replace(/_a/g, ampm_ap.substr(0, 1));
		format = format.replace(/_A/g, ampm_AP.substr(0, 1));
		format = format.replace(/_mmmm/g, month_mmmm);
		format = format.replace(/_MMMM/g, month_MMMM);
		format = format.replace(/_mmm/g, month_mmm);
		format = format.replace(/_MMM/g, month_MMM);
		format = format.replace(/_mm/g, month_mm);
		format = format.replace(/_m/g, month_m);
		format = format.replace(/_dd/g, day_dd);
		format = format.replace(/_d/g, day_d);
		format = format.replace(/_yyyy/g, year_yyyy);
		format = format.replace(/_yy/g, year_yy);
		format = format.replace(/_y/g, year_y);
		format = format.replace(/_wwww/g, weekday_wwww);
		format = format.replace(/_WWWW/g, weekday_WWWW);
		format = format.replace(/_www/g, weekday_www);
		format = format.replace(/_WWW/g, weekday_WWW);
		format = format.replace(/_w/g, weekday_w);
		format = format.replace(/_kk/g, week_kk);
		format = format.replace(/_k/g, week_k);
		format = format.replace(/_qq/g, quarter_qq);
		format = format.replace(/_q/g, quarter_q);

		var num = EMPTY_STRING;
		var ord = EMPTY_STRING;
		var suf = EMPTY_STRING;
		var ordinal_o = EMPTY_STRING;
		var ordinal_O = EMPTY_STRING;
		var temp = format.toUpperCase();

		while (temp.indexOf("_O") != -1) {
			ord = temp.indexOf("_O")
			num = EMPTY_STRING;
			suf = EMPTY_STRING;

			if (ord > 0) {
				if (!isNaN(format.substr(ord - 1, 1))) {
					num = format.substr(ord - 1, 1);

					if (ord > 1) {
						if (!isNaN(format.substr(ord - 2, 1))) {
							num = EMPTY_STRING + format.substr(ord - 2, 1) + num;
						}
					}
				}

				if (num != EMPTY_STRING) {
					num = "00" + num;

					switch (num.substr(num.length - 1, 1)) {
						case "1":
							suf = "st";
							break;
						case "2":
							suf = "nd";
							break;
						case "3":
							suf = "rd";
							break;
						default:
							suf = "th";
					}

					if (num.substr(num.length - 2, 2) > 10 && num.substr(num.length - 2, 2) < 20) {
						suf = "th";
					}
				}
			}

			ordinal_o = suf;
			ordinal_O = ordinal_o.toUpperCase();

			format = format.replace(/_o/, ordinal_o);
			format = format.replace(/_O/, ordinal_O);
			temp = format.toUpperCase();
		}

		format = format.replace(/~�/g, "_");
		return format;
	}

//--------------------------------------------------------------------------------------------------
	function getElementsByClassName(targetClass)
//--------------------------------------------------------------------------------------------------
	{
// Function for browsers that don't support getElementsByClassName natively
/*
		if (document.getElementsByClassName)
			return document.getElementsByClassName(targetClass);
		else {*/
			var classArray = new Array();
			var searchClass = new RegExp("\\b" + targetClass + "\\b");
			var allElements = document.getElementsByTagName("*");

			for (var i = 0; i < allElements.length; i++) {
				var classes = allElements[i].className;
				if (searchClass.test(classes)) classArray.push(allElements[i]);
			}

			return classArray;
		//}
	}

//--------------------------------------------------------------------------------------------------
	function getFunctionCall(func)
//--------------------------------------------------------------------------------------------------
	{
// Return the calling syntax (including arguments) for a given function.

		var signature = STRING_FUNCTION + CHAR_SPACE;
		var testName = getFunctionName(func);

		try {
			var syntax = window[testName].toString();
		} catch(er) {
			return(testName + "()");
		}

		syntax = syntax.substring(syntax.indexOf(testName), syntax.indexOf(CHAR_BRACE) - 1);

// Get the actual arguments, since multiple tests can run a single function, just with different parameters.

		var argsSyntax = syntax.match(/\(.+\)$/);
		var argsActual = func.match(/\(.+\)$/);

// If this function has no arguments, return the original declaration.

		if((argsSyntax == null) || (argsActual == null)) return(syntax);
		return(syntax.replace(argsSyntax, argsActual));
	}

//--------------------------------------------------------------------------------------------------
	function getFunctionName(func)
//--------------------------------------------------------------------------------------------------
	{
// Return only the name for a given function.

		var testName = func.toString();
		var signature = STRING_FUNCTION + CHAR_SPACE;

// Check if an actual function object was passed in.

		if(testName.indexOf(signature) == 0) testName = testName.substring(signature.length, testName.indexOf(CHAR_BRACE));

// Strip off any trailing parentheses, parameters, or semi-colons.

		if(testName.indexOf(CHAR_LPAREN) != -1) testName = testName.substr(0, testName.indexOf(CHAR_LPAREN));
		if(testName.indexOf(CHAR_SEMI) != -1) testName = testName.substr(0, testName.indexOf(CHAR_SEMI));

		return(testName);
	}

//--------------------------------------------------------------------------------------------------
	function getUniqueMenuId(menuNumber)
//--------------------------------------------------------------------------------------------------
	{
		while(window.menu.getMenuItemById(menuNumber)) menuNumber++;
		return menuNumber;
	}

//--------------------------------------------------------------------------------------------------
	function isNumeric(string)
//--------------------------------------------------------------------------------------------------
	{
		return(string.match(/^[0-9]+$/));
	}

//--------------------------------------------------------------------------------------------------
	function l(item)
//--------------------------------------------------------------------------------------------------
	{
		if(typeof item === STRING_STRING) {
			return document.getElementById(item);
		} else {
			return item;
		}
	}

//--------------------------------------------------------------------------------------------------
	function loopTests()
//--------------------------------------------------------------------------------------------------
	{

		var checks = getElementsByClassName(CLASS_CHECK);

		while(true) {
			timer = 0;
			testNumber++;

			if(testNumber > checks.length - 1) {

// Check for pending results, and don't quit if we've still got some coming.

				if(countPending()) {
					//alert("still running");
					setTimeout(endTests, autoExitTimeout);
				    return;
			}
				endTests();
				return;

// Get the id of the cell that will contain the results.

			} else {
				state = STATE_RUNNING;
				resultsCell = checks[testNumber].id + ID_RESULTS;

				if(checks[testNumber].checked) {
					if(checks[testNumber].name != EMPTY_STRING) {
						l(resultsCell).innerHTML = STRING_RUNNING;

						try {
							var testResults = eval(checks[testNumber].id);	
							self.displayResults(testResults); // to be removed 
						} catch(er) {
							l(resultsCell).innerHTML = EMPTY_STRING;
							self.displayResults(TEST_FAILED, er);
						}
					}

// Wait a bit, then call this function to run the next test.

					timer = setTimeout(loopTests, testDelay);
					return;

				} else {
					self.displayResults(TEST_SKIPPED);
					timer = setTimeout(loopTests, TIMEOUT_QUICK);
					return;
				}
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	function menuCreate()
//--------------------------------------------------------------------------------------------------
	{
// Take each menu number (in numeric order) and ensure that we have unique menu IDs.

		MENU_NUMBER_TOP = getUniqueMenuId(MENU_NUMBER_TOP);
		MENU_NUMBER_RUN = getUniqueMenuId(MENU_NUMBER_RUN);
		MENU_NUMBER_STOP = getUniqueMenuId(MENU_NUMBER_STOP);
		MENU_NUMBER_SELECT = getUniqueMenuId(MENU_NUMBER_SELECT);
		MENU_NUMBER_ALL = getUniqueMenuId(MENU_NUMBER_ALL);
		MENU_NUMBER_NONE = getUniqueMenuId(MENU_NUMBER_NONE);
		MENU_NUMBER_SEND = getUniqueMenuId(MENU_NUMBER_SEND);
		MENU_NUMBER_CLEAR = getUniqueMenuId(MENU_NUMBER_CLEAR);
		MENU_NUMBER_ROTATE = getUniqueMenuId(MENU_NUMBER_ROTATE);
		MENU_NUMBER_SORT = getUniqueMenuId(MENU_NUMBER_SORT);
		MENU_NUMBER_ASC = getUniqueMenuId(MENU_NUMBER_ASC);
		MENU_NUMBER_DESC = getUniqueMenuId(MENU_NUMBER_DESC);
		MENU_NUMBER_HIDE = getUniqueMenuId(MENU_NUMBER_HIDE);
		MENU_NUMBER_SHOW = getUniqueMenuId(MENU_NUMBER_SHOW);
		MENU_NUMBER_DELAY = getUniqueMenuId(MENU_NUMBER_DELAY);
		MENU_NUMBER_ABOUT = getUniqueMenuId(MENU_NUMBER_ABOUT);

// Display the top-level menu item.

		var menuTop = new MenuItem(MENU_ITEM_TOP, MENU_NUMBER_TOP);
		window.menu.append(menuTop);

// Display the run tests menu item.

		menuParent = new MenuItem(MENU_ITEM_RUN, MENU_NUMBER_RUN);
		menuParent.onSelect = menuRunTests;
		menuTop.append(menuParent);

		menuParent = new MenuItem(MENU_ITEM_STOP, MENU_NUMBER_STOP);
		menuParent.onSelect = menuStopTests;
		menuTop.append(menuParent);
		menuParent.setDimmed(true);

// Display the select all/none menu items.

		var menuParent = new MenuItem(MENU_ITEM_SELECT, MENU_NUMBER_SELECT);
		menuTop.append(menuParent);

		var menuChild = new MenuItem(MENU_ITEM_ALL, MENU_NUMBER_ALL);
		menuChild.onSelect = menuSelectTests;
		menuParent.append(menuChild);

		menuChild = new MenuItem(MENU_ITEM_NONE, MENU_NUMBER_NONE);
		menuChild.onSelect = menuDeselectTests;
		menuParent.append(menuChild);

// Display the send results to server menu item.

		menuParent = new MenuItem(MENU_ITEM_SEND, MENU_NUMBER_SEND);
		menuParent.onSelect = menuSendResults;
		menuTop.append(menuParent);

// Display the clear results menu item.

		menuParent = new MenuItem(MENU_ITEM_CLEAR, MENU_NUMBER_CLEAR);
		menuParent.onSelect = menuClearResults;
		menuTop.append(menuParent);

// Display the rotate screen menu item.

		menuParent = new MenuItem(MENU_ITEM_ROTATE, MENU_NUMBER_ROTATE);
		menuParent.onSelect = menuRotate;
		menuTop.append(menuParent);

// Display the sort menu items.

		menuParent = new MenuItem(MENU_ITEM_SORT, MENU_NUMBER_SORT);
		menuTop.append(menuParent);

		menuChild = new MenuItem(MENU_ITEM_ASC, MENU_NUMBER_ASC);
		menuChild.onSelect = menuSortAsc;
		menuParent.append(menuChild);

		menuChild = new MenuItem(MENU_ITEM_DESC, MENU_NUMBER_DESC);
		menuChild.onSelect = menuSortDesc;
		menuParent.append(menuChild);

// Display the show/hide menu items.

		menuParent = new MenuItem(MENU_ITEM_HIDE, MENU_NUMBER_HIDE);
		menuParent.onSelect = menuHideUI;
		menuTop.append(menuParent);

		menuParent = new MenuItem(MENU_ITEM_SHOW, MENU_NUMBER_SHOW);
		menuParent.onSelect = menuShowUI;
		menuTop.append(menuParent);
		menuParent.setDimmed(true);

// Display the set delay menu item.

		menuParent = new MenuItem(MENU_ITEM_DELAY, MENU_NUMBER_DELAY);
		menuParent.onSelect = menuDelay;
		menuTop.append(menuParent);

// Display the about menu item.

		menuParent = new MenuItem(MENU_ITEM_ABOUT, MENU_NUMBER_ABOUT);
		menuParent.onSelect = menuAbout;
		menuTop.append(menuParent);
	}

//--------------------------------------------------------------------------------------------------
	function menuAbout()
//--------------------------------------------------------------------------------------------------
	{
		alert(_CLASS_NAME_ + STRING_LINEFEED + _VERSION_);
	}

//--------------------------------------------------------------------------------------------------
	function menuClearResults()
//--------------------------------------------------------------------------------------------------
	{
		self.clearResults();
	}

//--------------------------------------------------------------------------------------------------
	function menuDelay()
//--------------------------------------------------------------------------------------------------
	{
		while(true) {
			var delay = prompt(STRING_ENTER_DELAY, testDelay);
			if(delay == null) break;

			if(parseInt(delay) == delay) {
				if(parseInt(delay) > 0) {
					testDelay = delay;
					break;
				}
			}

			alert(ERROR_BAD_DELAY);
		}
	}

//--------------------------------------------------------------------------------------------------
	function menuDeselectTests()
//--------------------------------------------------------------------------------------------------
	{
		self.deselectTests();
	}

//--------------------------------------------------------------------------------------------------
	function menuHideUI()
//--------------------------------------------------------------------------------------------------
	{
		self.hideUI();
	}

//--------------------------------------------------------------------------------------------------
	function menuRotate()
//--------------------------------------------------------------------------------------------------
	{
		self.rotateScreen();
	}

//--------------------------------------------------------------------------------------------------
	function menuRunTests()
//--------------------------------------------------------------------------------------------------
	{
		self.runTests();
	}

//--------------------------------------------------------------------------------------------------
	function menuSelectTests()
//--------------------------------------------------------------------------------------------------
	{
		self.selectTests();
	}

//--------------------------------------------------------------------------------------------------
	function menuSendResults()
//--------------------------------------------------------------------------------------------------
	{
		var newURL = prompt(STRING_ENTER_URL, DEFAULT_URL);
		if(newURL == null) return;
		DEFAULT_URL = newURL;

/// ToDo: Save newURL for future sessions

		self.sendResults();
	}

//--------------------------------------------------------------------------------------------------
	function menuShowUI()
//--------------------------------------------------------------------------------------------------
	{
		self.showUI();
	}

//--------------------------------------------------------------------------------------------------
	function menuSortAsc()
//--------------------------------------------------------------------------------------------------
	{
		self.sortTests();
	}

//--------------------------------------------------------------------------------------------------
	function menuSortDesc()
//--------------------------------------------------------------------------------------------------
	{
		self.sortTests(true);
	}

//--------------------------------------------------------------------------------------------------
	function menuStopTests()
//--------------------------------------------------------------------------------------------------
	{
		self.stopTests();
	}

//--------------------------------------------------------------------------------------------------
	function recordResults(item)
//--------------------------------------------------------------------------------------------------
	{
		if(logEnabled) {
			var stamp = fmtDateTime(STRING_FORMAT_TIME);

			item = item.replace(/\\n/g, ", ");
			item = item.replace(/\x0A/g, ", ");

			record += stamp + CHAR_SPACE + item + STRING_LINEFEED;

			if(console) {
				console.log(stamp + CHAR_SPACE + item);
			}
		}
	}

//--------------------------------------------------------------------------------------------------
	function sendResultsCallback()
//--------------------------------------------------------------------------------------------------
	{
// Do not call this function directly; let the server do it.

		try {
	 		var serverReadyState = httpRequest.readyState;
	 	} catch(err) {}

		try {
		 	var serverResponse = httpRequest.responseText;
	 	} catch(err) {}

		if (serverReadyState == READY_STATE_COMPLETE) {
			l(ID_ERROR).innerHTML = serverResponse;
		}
	}

//--------------------------------------------------------------------------------------------------
	function sortDescending(a, b)
//--------------------------------------------------------------------------------------------------
	{
		return ((a < b) ? 1 : ((a > b) ? -1 : 0));
	}

//--------------------------------------------------------------------------------------------------
	function uniqueId(prefix)
//--------------------------------------------------------------------------------------------------
	{
// Returns an ID that begins with the given prefix and that is unique in this document.

		var suffix = 0;
		var el = prefix;

		do {
			el = prefix + ++suffix;
		} while(l(el));

		return(el);
	}

//--------------------------------------------------------------------------------------------------
	function zeroPad(item, len)
//--------------------------------------------------------------------------------------------------
	{
// Returns item padded with zeroes to a length of len.
// A len < 0 pads on the right; otherwise, padding occurs on the left.

		var pad = item;
		var size = Math.abs(len);

		if (size == 0 || isNaN(size)) return pad;

		for (var i = 0; i < size; i++) {
			if (len < 0) {
				pad += CHAR_ZERO;
			} else {
				pad = CHAR_ZERO + pad;
			}
		}

		if (len < 0) {
			return pad.substr(0, size);
		} else {
			return pad.substr(pad.length - size, size);
		}

		return pad;
	}
}
