/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "WebAppRegistry.h"
#include "AttributeMap.h"
#include "w3cxmlplugin.h"
#include "w3csettingskeys.h"
#include "wrtsettings.h"
#include "trustsession.h"
#include "configw3xmlparser.h"
#include "secsession.h"
#include "widgetmanagerconstants.h"
#include "proprietarysettingskeys.h"
#include <QCoreApplication>
#include <QTextStream>

#define UNTRUSTED_WIDGET QString("UntrustedWidgets");

static unsigned long size;

static void
setAttributeMap(ConfigXmlParser &parser, AttributeMap &contents)
{
    QVariant version = parser.version();
    QVariant height = parser.height();
    QVariant width = parser.width();
    QVariant id = parser.id();
    QVariant lang = parser.language();
    QVariant viewmodes = parser.viewmodes();
    QStringList w3ckeys = parser.allKeys();
    int keycount = w3ckeys.count() ;

    if (!version.isNull())
        contents.insert(W3CSettingsKey::WIDGET_VERSION, version);
    if (!height.isNull())
        contents.insert(W3CSettingsKey::WIDGET_HEIGHT, height);
    if (!width.isNull())
        contents.insert(W3CSettingsKey::WIDGET_WIDTH, width);
    if (!id.isNull())
        contents.insert(W3CSettingsKey::WIDGET_ID, id);
    if (!lang.isNull())
        contents.insert(W3CSettingsKey::WIDGET_LANG, lang);
    if (!viewmodes.isNull())
        contents.insert(W3CSettingsKey::WIDGET_VIEWMODES, viewmodes);

    for (int iter = 0; iter < keycount; ++iter) {
        QString key = w3ckeys.at(iter) ;
        W3CElement *elm = (W3CElement *) parser.getElement(key) ;

        // Insert Element text, Attribute name and Attribute value
        QVariant w3celemText = elm->readElementText();
        contents.insert(key, w3celemText);

        QXmlStreamAttributes attributes = elm->attributes();
        for (int i = 0; i < attributes.size(); ++i) {
            QString name = attributes[i].name().toString();
            QString attributeKey = key + QString(":") + name;
            QVariant attributeValue = attributes[i].value().toString();
            contents.insert(attributeKey, attributeValue);
        }
    }
}

static void
get_widget_size(const QString& root)
{
    QDir dir(root);
    dir.setFilter(QDir::Files | QDir::Hidden | QDir::NoSymLinks | QDir::Dirs);
    QFileInfoList list = dir.entryInfoList();
    QFileInfo f;
    QString filename;

    foreach (f, list) {
        filename = f.fileName();
        if (!(f.isDir() && filename.compare(".") && filename.compare("..")))
            size += f.size();
        else
            get_widget_size(f.filePath());
    }
}

static bool
create_security_section(const QString& id, const QString& root, const QString& name,
                        AttributeMap *manifestinfo)
{
    WidgetProperties *widgetProps = new WidgetProperties;
    if (!widgetProps)
        return false;

    QString startFile = QString(root).append("/index.html");

    size = 0;
    get_widget_size(root);
    widgetProps->setId(id);
    widgetProps->setSize(size);
    widgetProps->setTitle(name);
    widgetProps->setResourcePath(root);
    widgetProps->setAuthorDir(manifestinfo->value(W3CSettingsKey::WIDGET_AUTHOR).toString());
    widgetProps->setDescriptionDir(manifestinfo->value(W3CSettingsKey::WIDGET_DESCRIPTION).toString());

    widgetProps->setSharedLibrary(!manifestinfo->value(ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_FOLDER).
                                  toString().isEmpty());
    widgetProps->setSharedLibraryWidget(manifestinfo->value(ProprietarySettingsKey::WIDGET_SHARED_LIBRARY_WIDGET).
                                        toString().startsWith("T",Qt::CaseInsensitive));

    if (widgetProps->isSharedLibrary() && !widgetProps->isSharedLibraryWidget())
        widgetProps->setType(WIDGET_PACKAGE_FORMAT_SHARED_LIBRARY);
    else
        widgetProps->setType(WIDGET_PACKAGE_FORMAT_WGT);

    QString resourcesDir = widgetProps->resourcePath();

    if (!resourcesDir.isNull() && !resourcesDir.isEmpty()) {
        QDir dir;
        dir.mkpath(resourcesDir);
        QString secSessionFileName(QDir::toNativeSeparators(resourcesDir));
        secSessionFileName += QDir::separator();
        secSessionFileName += SECSESSION_FILE;
        QFile secSessionFile(secSessionFileName);

        if (secSessionFile.open(QIODevice::WriteOnly)) {
            QTextStream stream(&secSessionFile);
            stream << widgetProps->secureSessionString();
            secSessionFile.close();
        }
    }

    delete widgetProps;
    return true;
}

static int
register_widget(const QString appPath, const QString appId)
{
    QString appTitle, iconPath, type, startPath;
    bool ret = false;
    AttributeMap contents;
    ConfigXmlParser parser;
    QString filename = QString(appPath).append("/config.xml");
    size = 0;
    parser.setFile(filename);

    WebAppRegistry *registry = WebAppRegistry::instance();
    if (!registry)
        return 2;

    if (!parser.parseFile()) {
        delete registry;
        return 3;
    }

    startPath = QString(appPath).append("/index.html");
    get_widget_size(appPath);
    setAttributeMap(parser, contents);
    type = QString(WIDGET_PACKAGE_FORMAT_WGT);
    iconPath = contents.value(W3CSettingsKey::WIDGET_ICON).toString();
    appTitle = contents.value(W3CSettingsKey::WIDGET_NAME).toString();
    if (appTitle.isEmpty())
        appTitle = appId;
     ret = registry->registerApp(appId, appTitle, appPath, iconPath,
                                 contents, type, size, startPath);

    delete registry;

    create_security_section(appId, appPath, appTitle, &contents);
    return (ret) ? 0 : 4;
}

static int
unregister_widget(const QString appId)
{
    bool ret = false;
    WebAppRegistry *registry = WebAppRegistry::instance();
    if (!registry)
        return 2;
    ret = registry->unregister(appId);

    delete registry;
    return (ret) ? 0 : 4;
}

int main(int argc, char *argv[])
{
    QCoreApplication app(argc, argv);
    QStringList args = app.arguments();
    if(args.size() == 3) {
        if(args.at(2) == "unregister")
            return unregister_widget(args.at(1));
        else
            return 6;
    }

    if(args.size() == 4) {
#ifdef POST_REGISTER_WEBWIDGET
        if(args.at(3) == "register")
            return register_widget(args.at(1), args.at(2));
        else
            return 6;
#else
        return 0;
#endif
    }

    return 7;
}
