/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FoodInput.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Array.h>
#include <Lum/Button.h>
#include <Lum/DateSelect.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TimeSelect.h>

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Values.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include <Lum/OS/Theme.h>

#include "Trend.h"
#include "Util.h"

#include <iostream>

class FoodEdit : public Lum::Dlg::ValuesInput
{
public:
  FoodEdit(Lum::Model::String* name,
           Lum::Model::String* unit,
           Lum::Model::String* value)
  {
    Lum::String  *string;
    Value        v;
    std::wstring tmp;

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,20);

    v.label=_(L"FOOD_EDIT_LABEL_NAME",L"Name:");
    v.model=name;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,8);

    v.label=_(L"FOOD_EDIT_LABEL_UNIT",L"Unit:");
    v.model=unit;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"FOOD_EDIT_LABEL_VALUE",L"Value ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
    v.label=tmp;
    v.model=value;
    v.control=string;
    v.validator=new ValueValidator();
    AddValue(v);
  }

  static bool EditFood(Lum::OS::Window* parent,
                       const std::wstring& caption,
                       const std::wstring& text,
                       Lum::Model::String* name,
                       Lum::Model::String* unit,
                       Lum::Model::String* value)
  {
    assert(name!=NULL && unit!=NULL && value!=NULL);

    FoodEdit *dlg;
    bool     res;

    dlg=new FoodEdit(name,unit,value);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodRegister : public Lum::Dlg::ValuesInput
{
public:
  FoodRegister(Lum::Model::Time* time,
          Lum::Model::String* amount,
          const std::wstring& unit)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"FOOD_REGISTER_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));

    tmp=_(L"FOOD_REGISTER_LABEL_AMOUNT",L"Amount ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",unit);
    v.label=tmp;
    v.model=amount;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool RegisterFood(Lum::OS::Window* parent,
                           const std::wstring& caption,
                           const std::wstring& text,
                           Lum::Model::Time* time,
                           Lum::Model::String* amount,
                           const std::wstring& unit)
  {
    assert(time!=NULL && amount!=NULL);

    FoodRegister *dlg;
    bool    res;

    dlg=new FoodRegister(time,amount,unit);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodDataProvider : public FoodModel::DataProvider
{
public:
  std::wstring GetString(const FoodModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return (*iter)->name+L" ("+(*iter)->unit+L")";
    case 2:
      return ValueToString((*iter)->value)+L" "+valueUnit->Get();
    default:
      return L"";
    }
  }
};

class FoodComparator : public FoodModel::Comparator
{
public:
  bool operator()(const FoodModel::ElementType& a,
                  const FoodModel::ElementType& b,
                  size_t column, bool down) const
  {
    if (down) {
      return a->name>b->name;
    }
    else {
      return a->name<b->name;
    }
  }
};

class FoodFormatProvider : public Lum::TableView::FormatProvider
{
private:
  FoodModelRef      table;
  Lum::OS::ColorRef goodFoodColor;
  Lum::OS::ColorRef badFoodColor;
  Lum::OS::FillRef  goodFoodBackground;
  Lum::OS::FillRef  badFoodBackground;

public:
  FoodFormatProvider(FoodModel* table)
   : table(table),
     goodFoodColor(0.82,1.0,0.82,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     badFoodColor(1.0,0.82,0.82,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     goodFoodBackground(new Lum::OS::PlainFill(goodFoodColor)),
     badFoodBackground(new Lum::OS::PlainFill(badFoodColor))
  {
    // no code
  }

  Lum::OS::Fill* GetBackground(size_t column, size_t row) const
  {
    const Food* entry=table->GetEntry(row);

    if (entry->value<=(Value)goodValueLimit->Get()) {
      return goodFoodBackground.Get();
    }
    else if (entry->value>=(Value)badValueLimit->Get()) {
      return badFoodBackground.Get();
    }
    else {
      return Lum::TableView::FormatProvider::GetBackground(column,row);
    }
  }
};

FoodInput::FoodInput()
 : addFoodEntryAction(new Lum::Model::Action()),
   removeFoodEntryAction(new Lum::Model::Action()),
   editFoodEntryAction(new Lum::Model::Action()),
   registerFoodAction(new Lum::Model::Action()),

   foodSearch(new Lum::Model::String(L"")),
   foods(new FoodModel(new FoodDataProvider(),new FoodComparator())),
   foodSelection(new Lum::Model::SingleLineSelection()),

   searchTimerAction(new Lum::Model::Action())
{
  Observe(foodSearch);

  Observe(addFoodEntryAction);
  Observe(removeFoodEntryAction);
  Observe(editFoodEntryAction);
  Observe(registerFoodAction);

  Observe(foodSelection);

  Observe(searchTimerAction);

  Observe(goodValueLimit);
  Observe(badValueLimit);
}

void FoodInput::CalcSize()
{
  Lum::Menu             *menu;
  Lum::Panel            *ft;
  Lum::Panel            *sb;
  Lum::Panel            *horiz;
  Lum::Panel            *vert;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  // Search box

  sb=Lum::VPanel::Create(true,false);
  sb->Add(Lum::HPanel::Create(true,false)
          ->Add(Lum::Text::Create(L"Search:"))
          ->AddSpace()
          ->Add(Lum::String::Create(foodSearch,true,false)->ShowClearButton(true))
          ->AddSpace()
          ->Add(Lum::Button::Create(L"+",
                                    Lum::OS::Theme::imageAdd,
                                    addFoodEntryAction,
                                    Lum::Button::typeIcon))
          ->Add(new Lum::HSpace(Lum::Space::sizeSmall))
          ->Add(Lum::Button::Create(L"-",
                                    Lum::OS::Theme::imageRemove,
                                    removeFoodEntryAction,
                                    Lum::Button::typeIcon))
          ->Add(new Lum::HSpace(Lum::Space::sizeSmall))
          ->Add(Lum::Button::Create(L"~",
                                    Lum::OS::Theme::imageEdit,
                                    editFoodEntryAction,
                                    Lum::Button::typeIcon))
          ->Add(new Lum::HSpace(Lum::Space::sizeSmall))
          ->Add(Lum::Button::Create(L"->",
                                    Lum::OS::Theme::imagePositive,
                                    registerFoodAction,
                                    Lum::Button::typeIcon))
          );

  // Food table

  ft=Lum::HPanel::Create(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_FOOD",L"Food"),Lum::Base::Size::stdCharWidth,20,true);
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_VALUE",L"Value"),Lum::Base::Size::stdCharWidth,8);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetHeaderModel(headerModel);
  table->SetModel(foods);
  table->SetSelection(foodSelection);
  table->SetDoubleClickAction(registerFoodAction);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoFitColumns(true);
  table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetFormatProvider(new FoodFormatProvider(foods));
  ft->Add(table);

  menu=new Lum::Menu();
  menu->SetParent(GetWindow());
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_ADD",L"_Add entry..."),addFoodEntryAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_EDIT",L"_Edit entry..."),editFoodEntryAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_REMOVE",L"_Remove entry"),removeFoodEntryAction);
  table->SetMenu(menu->GetWindow());

  // Layout comonents

  horiz=Lum::HPanel::Create();

  vert=Lum::VPanel::Create(true,true);

  vert->Add(sb);
  vert->AddSpace();
  vert->Add(ft);

  horiz->Add(vert);

  container=horiz;

  Component::CalcSize();
}

void FoodInput::UpdateFoods()
{
  editFoodEntryAction->Disable();
  removeFoodEntryAction->Disable();
  registerFoodAction->Disable();

  std::wstring filter=foodSearch->Get();

  Lum::Base::ToLower(filter);

  // Food
  foods->Off();

  foods->Clear();
  for (std::set<Food>::iterator iter=data.foods.begin(); iter!=data.foods.end(); ++iter) {
    if (filter.empty()) {
      foods->Append(&*iter);
    }
    else {
      std::wstring tmp(iter->name);

      Lum::Base::ToLower(tmp);

      if (tmp.find(filter)!=std::wstring::npos) {
        foods->Append(&*iter);
      }
    }
  }

  foods->Sort(1);

  foods->On();
}

void FoodInput::AddFoodEntry()
{
  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;
  std::wstring          tmp;

  name=new Lum::Model::String(L"");
  unit=new Lum::Model::String(L"");
  value=new Lum::Model::String(L"");

  tmp=_(L"FOOD_ADD_TEXT",L"Please enter the name of the new food component together\n"
        L"with its standard portion unit and its value in $unit!");
  Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
  if (!FoodEdit::EditFood(this->GetWindow(),
                          _(L"FOOD_ADD_TITLE",L"Please enter a new food component..."),
                          tmp,
                          name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));
    return;
  }

  foods->Append(result);

  foods->Sort(1);
}

void FoodInput::EditFoodEntry()
{
  Food       oldFood;
  const Food *entry=foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;

  oldFood=*entry;

  name=new Lum::Model::String(entry->name);
  unit=new Lum::Model::String(entry->unit);
  value=new Lum::Model::String(ValueToString(entry->value));

  if (!FoodEdit::EditFood(this->GetWindow(),
                              L"Editing food component...",
                              L"Please enter the name of the food component together\n"
                              L"with its standard portion unit and its value!",
                              name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  foods->Off();

  data.DeleteFood(*foods->GetEntry(foodSelection->GetLine()));
  foods->Delete(foodSelection->GetLine());

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));

    result=data.AddFood(oldFood);
  }

  foods->Append(result);
  foods->Sort(1);

  foods->On();
}

void FoodInput::RemoveFoodEntry()
{
  data.DeleteFood(*foods->GetEntry(foodSelection->GetLine()));
  foods->Delete(foodSelection->GetLine());
}

void FoodInput::RegisterFood()
{
  const Food *entry=foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef value;
  size_t                amount;
  std::wstring          t1,t2;

  value=new Lum::Model::String(L"1");

  t1=_(L"FOOD_REGISTER_TITLE",L"I have consumed $food...");
  Lum::Base::L10N::Substitute(t1,L"$food",entry->name);
  t2=_(L"FOOD_REGISTER_TEXT",L"Please enter the amount of $food\nyou have consumed!");
  Lum::Base::L10N::Substitute(t2,L"$food",entry->name);
  if (!FoodRegister::RegisterFood(this->GetWindow(),
                                  t1,t2,timestamp,value,entry->unit)) {
    return;
  }

  if (value->IsNull() || value->Empty()) {
    return;
  }

  if (!StringToAmount(value->Get(),amount)) {
    return;
  }

  Entry e;

  e.time=timestamp->Get();
  e.item=entry->name;
  e.unit=entry->unit;
  e.value=entry->value;
  e.amount=amount;

  data.AddEntry(date->Get(),e);
  // We trigger the date model to initiate refresh of data table
  date->Notify();
}

void FoodInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==addFoodEntryAction && addFoodEntryAction->IsFinished()) {
    AddFoodEntry();
  }
  else if (model==editFoodEntryAction && editFoodEntryAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      EditFoodEntry();
    }
  }
  else if (model==removeFoodEntryAction && removeFoodEntryAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RemoveFoodEntry();
    }
  }
  else if (model==registerFoodAction && registerFoodAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RegisterFood();
    }
  }
  else if (model==goodValueLimit || model==badValueLimit) {
    UpdateFoods();
  }
  else if (model==foodSearch) {
    if (foodSearch->Empty()) {
      UpdateFoods();
      lastFoodSearchFilter=foodSearch->Get();
    }
    else {
      Lum::OS::display->RemoveTimer(searchTimerAction);
      Lum::OS::display->AddTimer(1,150000,searchTimerAction);
    }
  }
  else if (model==searchTimerAction && searchTimerAction->IsFinished()) {
    if (foodSearch->Get()!=lastFoodSearchFilter) {
      UpdateFoods();
      lastFoodSearchFilter=foodSearch->Get();
    }
  }
  else if (model==foodSelection) {
    if (foodSelection->HasSelection()) {
      editFoodEntryAction->Enable();
      removeFoodEntryAction->Enable();
      registerFoodAction->Enable();
    }
    else {
      editFoodEntryAction->Disable();
      removeFoodEntryAction->Disable();
      registerFoodAction->Disable();
    }
  }

  Lum::Component::Resync(model,msg);
}

void FoodInput::InitializeGUI()
{
  UpdateFoods();
}
