/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>
#include <Lum/Def/MultiView.h>

#include <Lum/Dialog.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Tab.h>
#include <Lum/View.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Help.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Main.h>

#include "config.h"

#include "Configuration.h"
#include "Diagram.h"
#include "DataInput.h"
#include "FoodInput.h"
#include "ImportExport.h"
#include "Preferences.h"
#include "Statistics.h"

static Lum::Def::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef importAction;
  Lum::Model::ActionRef exportAction;
  Lum::Model::ActionRef backupAction;
  Lum::Model::ActionRef restoreAction;
  Lum::Model::ActionRef statisticsAction;
  Lum::Model::ActionRef configAction;
  Lum::Model::ActionRef helpAction;
  Lum::Model::ActionRef aboutAction;

  FoodInput             *foodInput;
  DataInput             *dataInput;
  Diagram               *diagram;

public:
  MainDialog()
  : importAction(new Lum::Model::Action()),
    exportAction(new Lum::Model::Action()),
    backupAction(new Lum::Model::Action()),
    restoreAction(new Lum::Model::Action()),
    statisticsAction(new Lum::Model::Action()),
    configAction(new Lum::Model::Action()),
    helpAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action())
  {
    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(importAction);
    Observe(exportAction);
    Observe(backupAction);
    Observe(restoreAction);
    Observe(statisticsAction);
    Observe(configAction);
    Observe(helpAction);
    Observe(aboutAction);
  }

  void PreInit()
  {
    Lum::Def::MultiView multiView(Lum::Def::Desc(L"WeightJinni"));

    multiView.SetValue(currentTab);

    foodInput=new FoodInput();
    foodInput->SetFlex(true,true);

    multiView.AddView(Lum::Def::MultiView::View(0,
                                                Lum::Def::Desc(_(L"MAIN_TAB_FOODINPUT",L"Food")),
                                                foodInput));

    dataInput=new DataInput();
    dataInput->SetFlex(true,true);

    multiView.AddView(Lum::Def::MultiView::View(1,
                                                Lum::Def::Desc(_(L"MAIN_TAB_INPUT",L"Day view")),
                                                dataInput));

    diagram=new Diagram();
    diagram->SetFlex(true,true);

    multiView.AddView(Lum::Def::MultiView::View(2,
                                                Lum::Def::Desc(_(L"MAIN_TAB_DIAGRAM",L"Diagram")),
                                                Lum::View::Create(diagram,true,true)));

    Lum::OS::display->GetBehaviour()->ApplyMultiViewDlg(this,multiView);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_IMPORT_FOOD_COMPONENTS",L"_Import food..."))
                                  .SetShortcut(Lum::OS::qualifierControl,L"i"),
                                  importAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_EXPORT_FOOD_COMPONENTS",L"_Export food..."))
                                  .SetShortcut(Lum::OS::qualifierControl,L"e"),
                                  exportAction))
        ->Separator()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_BACKUP",L"_Backup...")),backupAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_RESTORE",L"_Restore...")),restoreAction))
        ->Separator()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupEdit()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_STATISTICS",L"_Statistics...")),statisticsAction))
        ->Separator()
        ->ActionSettings(configAction)
      ->End()
      ->GroupHelp()
        ->ActionHelp(helpAction)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (!LoadConfig(Lum::Base::Path::GetApplicationConfigPath())) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Cannot load configuration!",
                              L"Cannot load configuration!");
      }

      foodInput->InitializeGUI();
      dataInput->InitializeGUI();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (!SaveConfig(Lum::Base::Path::GetApplicationConfigPath())) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Cannot save configuration!",
                              L"Cannot save configuration!");
      }
    }
    else if (model==importAction && importAction->IsFinished()) {
      Import(this);
      foodInput->UpdateFoods();
    }
    else if (model==exportAction && exportAction->IsFinished()) {
      Export(this);
    }
    else if (model==backupAction && backupAction->IsFinished()) {
      Lum::Base::Path path;
      std::wstring    file;

      path.SetNativeDir(Lum::Base::Path::GetHome());
      path.SetBaseName(L"WeightJinni.bak");

      file=path.GetPath();

      if (Lum::Dlg::File::SaveFile(GetWindow(),
                                   L"Store backup to...",
                                   file)) {
        if (!SaveConfig(file)) {
          Lum::Dlg::Msg::ShowOk(this,
                                L"Cannot save configuration!",
                                L"Cannot save configuration!");
        }
      }
    }
    else if (model==restoreAction && restoreAction->IsFinished()) {
      Lum::Base::Path path;
      std::wstring    file;

      path.SetNativeDir(Lum::Base::Path::GetHome());
      path.SetBaseName(L"WeightJinni.bak");

      file=path.GetPath();

      if (Lum::Dlg::File::GetFile(GetWindow(),
                                   L"Restore from...",
                                   file)) {
        if (!LoadConfig(file)) {
          Lum::Dlg::Msg::ShowOk(this,
                                L"Cannot load configuration!",
                                L"Cannot load configuration!");
        }
        else {
          dataInput->InitializeGUI();
          diagram->Redraw();
        }
      }
    }
    else if (model==statisticsAction && statisticsAction->IsFinished()) {
      Statistics *dialog;

      dialog=new Statistics();
      dialog->SetParent(this);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==configAction && configAction->IsFinished()) {
      Preferences *dialog;

      dialog=new Preferences();
      dialog->SetParent(this);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==helpAction && helpAction->IsFinished()) {
      Lum::Dlg::Help::Show(this,_(L"HELP_CONTENT",L"No help available"));
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    Lum::Base::L10N::SetFallbackLocale(L"en");

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Control your weight..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License (GPL)");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

