/*
 * mainwindow.cpp - main window class implementation.
 *
 * Copyright (C) 2010 - Aleksandr A. Lomov.
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include <QtWebKit>

#include "mainwindow.h"
#include "quseragentwebpage.h"



/**
 * @brief Authenticating result.
 */
namespace AuthResult {
    /** User closed window. */
    const QString ClosedByUser = "closed";

    /** Page not load. */
    const QString NoAccess = "no_access";
}



/** 
 * @brief Constructor.
 */
MainWindow::MainWindow(const int ID) : windowID (ID)
{
    this->setAttribute(Qt::WA_DeleteOnClose, true);

    progress = 0;

    isAuthSimple = true;
    isClosedByUser = true;

    view = new QWebView(this);

    view->setPage(new QUserAgentWebPage());

    connect(view, SIGNAL(loadFinished(bool)), SLOT(adjustLocation()));
    connect(view, SIGNAL(titleChanged(QString)), SLOT(adjustTitle()));
    connect(view, SIGNAL(loadProgress(int)), SLOT(setProgress(int)));
    connect(view, SIGNAL(loadFinished(bool)), SLOT(finishLoading(bool)));

    locationEdit = new QLineEdit(this);
    locationEdit->setSizePolicy(QSizePolicy::Expanding, locationEdit->sizePolicy().verticalPolicy());
    connect(locationEdit, SIGNAL(returnPressed()), SLOT(changeLocation()));

    authTitle = new QLabel(this);
    authTitle->setAlignment(Qt::AlignCenter);
    authTitle->setSizePolicy(QSizePolicy::Expanding, locationEdit->sizePolicy().verticalPolicy());
    authTitle->setText("");


    setCentralWidget(view);
    setUnifiedTitleAndToolBarOnMac(true);
    locationEdit->setReadOnly(true);
    locationEdit->setCursorPosition(0);
}


/**
 * @brief Destructor
 */
MainWindow::~MainWindow()
{
    qDebug() << "MainWindow destructor START";
    authParams.clear();
    qDebug() << "MainWindow destructor END";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Set new url.
 *
 * @param url new url.
 */
void MainWindow::setURL(const QString &url)
{
    locationEdit->setText(url);
    locationEdit->setCursorPosition(0);
    this->url = url;
    view->load(QUrl(url));
}


/**
 * @brief Start simple authentication.
 *
 * @param url url for user athentication.
 */
void MainWindow::startAuth(const QString &url)
{
    isAuthSimple = true;

    setToolBars(false);
    setURL(url);
    this->show();
}


/**
 * @brief Start extended authentication.
 *
 * @param url url for user athentication.
 * @param params cliet list witn parameters(regexps, title, ID...).
 */
void MainWindow::startAuth(const QString &url, const QStringList &params)
{
    isAuthSimple = false;

    authParams.parseParams(params);

    authTitle->setText(authParams.getTitle());

    bool withAuthTitle = !authTitle->text().isEmpty();

    setToolBars(withAuthTitle);

    setURL(url);
    this->show();
}


/**
 * @brief Getter for wndows ID.
 *
 * @return windows identifier.
 */
int MainWindow::getWindowID()
{
    return windowID;
}



/********************************************/
/********** Protected functios *************/
/**
 * a window close request for a top-level widget from the window system.
 *
 * @param event close event.
 */
void MainWindow::closeEvent(QCloseEvent *)
{
    if (isClosedByUser == true) {
        sendAuthResult(AuthResult::ClosedByUser, !isAuthSimple);
    }

    emit closed(getWindowID());
}



/********************************************/
/************ Protected slots ***************/
/**
 * @brief Adjust url windows location line.
 */
void MainWindow::adjustLocation()
{
    locationEdit->setText(view->url().toString());
    locationEdit->setCursorPosition(0);
}


/**
 * @brief Change location.
 *  Get new url and load it.
 */
void MainWindow::changeLocation()
{
    view->load(QUrl(locationEdit->text()));
    view->setFocus();
}


/**
 * @brief Adjust window title, and load progress.
 */
void MainWindow::adjustTitle()
{
    if (progress <= 0 || progress >= 100) {
        setWindowTitle(view->title());
    } else {
        setWindowTitle(QString("%1 (%2%)").arg(view->title()).arg(progress));
    }
}


/**
 * @brief Set load progress.
 *
 * @param p current load progress.
 */
void MainWindow::setProgress(int p)
{
    progress = p;
    adjustTitle();
}


/**
 * @brief Called when page is load.
 *  Setup window elements, and check authentication.
 */
void MainWindow::finishLoading(bool)
{
    progress = 100;
    adjustTitle();

    // Location empty - webkit can't load page.
    if (locationEdit->text().isEmpty() != true && isAuthComplete() == false) {
        return;
    }    

    if (locationEdit->text().isEmpty() == true) {
        sendAuthResult(AuthResult::NoAccess, !isAuthSimple);
    } else {
        sendAuthResult(locationEdit->text(), !isAuthSimple);
    }

    isClosedByUser = false;
    this->close();
}



/********************************************/
/*********** Private functions **************/
/**
 * @brief Add toolbars.
 *  Add toolbar with location edit and with authentication title.
 *
 * @param withAuthTitle true if auth title toolbar needed, false otherwise.
 */
void MainWindow::setToolBars(bool withAuthTitle)
{
    if (withAuthTitle == true) {
        QToolBar *authTitleBar = addToolBar(tr("AuthTitle"));
        authTitleBar->addWidget(authTitle);

        addToolBarBreak();
    }

    QToolBar *toolBar = addToolBar(tr("Navigation"));
    toolBar->addWidget(locationEdit);
}


/**
 * @brief Check authentication.
 *  Simple auth: check only different URLs (given and current).
 *  Extendet auth: result based on regexp checking.
 * (A rich Perl-based matching sintax and Case insensinsitive using).
 *
 * @return true if has different URLs(simple), or regexp find(extendet),
 * otherwise false.
 */
bool MainWindow::isAuthComplete()
{
    // Check simple auth. Auth complete if location URL and given are different.
    if (isAuthSimple == true && locationEdit->text() != url) {
        qDebug() << "Simple auth complete.";
        return true;
    } else if(isAuthSimple == true) {
        return false;
    }

    const QList<QRegExp *> &regexpList = authParams.getRegExps();
//    qDebug() << __FILE__ << ":" << __LINE__ << "agent=" << view->page()->userAgentForUrl(QUrl(locationEdit->text()));

    // Check extended auth: try to find regexp in location URL.
    for(int i = 0; i < regexpList.length(); ++i) {
        qDebug() << "Checking regular expression: " << regexpList[i]->pattern();

        if (regexpList.at(i)->exactMatch(locationEdit->text()) == true) {
            qDebug() << "Extenden auth complete (" << regexpList[i]->pattern() << ").";

            return true;
        }
    }

    return false;
}


/**
 * @brief Send signal with authentication result.
 *  Prepare parameters and send signal.
 *
 * @param authResult result of authentication.
 * @param withReturnParams include return parameters to signal.
 */
void MainWindow::sendAuthResult(const QString &authResult, bool withReturnParams)
{
    QList<QVariant> signalParams = authParams.prepareSignalParams(authResult, withReturnParams);
    const QString &clientID = authParams.getClientID();

    if (AuthDBusAdaptor::sendSignal(clientID, signalParams, true) == false) {
        qDebug() << "\nMessage not send.";
    } else {
        qDebug() << "\nMessage send (" << signalParams << ").";
    }
}
