/*
 * authparams.h - parameters authentication implementation.
 *
 * Copyright (C) 2010 - Aleksandr A. Lomov.
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include <QStringList>
#include <QtDebug>
#include <QRegExp>
#include <QVariant>

#include "authparams.h"



/**
 * @brief Types for authenticating parameters.
 */
namespace ParamsType {
    /** "Regular expression" type name. */
    const QString RegExp = "regexp";
    /** "Authenticating title" type name. */
    const QString AuthTitle = "title";
    /** "Authenticating result" type name. */
    const QString AuthResult = "authResult";

    const QString CaptchaResult = "captchaResult";
    const QString LogInResult = "loginResult";
    /** "Authenticating client ID" type name. */
    const QString ClientID = "clientID";

    /** Type modification: returned parameter. */
    const QString ReturnPostfix = "_RET";
    /** Type modification: context parameter. */
    const QString ContextPostfix = "_CXT";
}



/**
 * @brief Constructor.
 */
AuthParams::AuthParams()
{
}


/**
 * @brief Destructor
 */
AuthParams::~AuthParams()
{
    qDebug() << "AuthParams destructor START";
    clear();
    qDebug() << "AuthParams destructor END";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Parsing parameters.
 *
 * @param outerParams parameters list.
 */
void AuthParams::parseParams(const QStringList &outerParams)
{
    if (outerParams.isEmpty() == true) {
        return;
    }

    parseParametersByPostfixes(outerParams);

    title = getParamsAuthTitle(params);
    clientID = getParamsClientID(params);
    regexpList = getParamsRegExp(params);
}


/**
 * @brief Getter for client ID.
 *
 * @return client identifier.
 */
const QString& AuthParams::getClientID()
{
    return clientID;
}


/**
 * @brief Getter for title.
 *
 * @return authentiction title.
 */
const QString& AuthParams::getTitle()
{
    return title;
}


/**
 * @brief Getter for parameters for return.
 *
 * @return return parameters list.
 */
const QList<QString *>& AuthParams::getReturnParams()
{
    return returnParams;
}


/**
 * @brief Getter for context parameters.
 *
 * @return context parameters list.
 */
const QList<QString *>& AuthParams::getContextParams()
{
    return contextParams;
}


/**
 * @brief Getter for regular expressions parameters.
 *
 * @return regular parameters list.
 */
const QList<QRegExp *>& AuthParams::getRegExps()
{
    return regexpList;
}


/**
 * @brief Prepare parameters for dbus signal.
 * Build list with variant object, created from parameters.
 *
 * @param authResult authentication resul.
 * @param withRetParams flag - insert to signal parameters return parameters.
 *
 * @return list with parameters prepared for sending by signal.
 */
QList<QVariant> AuthParams::prepareSignalParams(const QString &authResult, bool withRetParams)
{
    QList<QVariant> signalParams;
    QStringList params;

    params << ParamsType::AuthResult << authResult;

    if (withRetParams == false) {
        signalParams.append(QVariant(params));
        return signalParams;
    }

    int len = returnParams.length() - 1;

    for (int i = 0; i < len; i += 2) {
        params << *returnParams[i] << *returnParams[i + 1];
    }

    signalParams.append(QVariant(params));

    return signalParams;
}


QList<QVariant> AuthParams::prepareCaptchaSignalParams(const QString &captchaResult, bool withRetParams)
{
    QList<QVariant> signalParams;
    QStringList params;

    params << ParamsType::CaptchaResult << captchaResult;

    if (withRetParams == false) {
        signalParams.append(QVariant(params));
        return signalParams;
    }

    int len = returnParams.length() - 1;

    for (int i = 0; i < len; i += 2) {
        params << *returnParams[i] << *returnParams[i + 1];
    }

    signalParams.append(QVariant(params));

    return signalParams;
}


QList<QVariant> AuthParams::prepareLogInSignalParams(const QStringList &loginResult)
{
    QList<QVariant> signalParams;
    QStringList params;

    params << ParamsType::LogInResult << loginResult;

    signalParams.append(QVariant(params));

    return signalParams;
}


/**
 * @brief Clear all parameters.
 * Clear parameters, use it before secondary call @see parseParams.
 */
void AuthParams::clear()
{
    int len = regexpList.length();

    for (int i = 0; i < len; ++i) {
        delete regexpList[i];
        regexpList[i] = NULL;
    }

    regexpList.clear();

    len = params.length();

    for (int i = 0; i < len; ++i) {
        delete params[i];
        params[i] = NULL;
    }

    params.clear();
}



/********************************************/
/*********** Private functions **************/
/**
 * @brief Try to find regular expressions parameters.
 * Try to find parameter with type @see RegExp nad return list with expressions.
 *
 * @param params parameters for searching.
 *
 * @return List with regular expressions, expressions are new allocated, you need to delete it.
 */
QList<QRegExp *> AuthParams::getParamsRegExp(const QList<QString*> &params)
{
    int len = params.length() - 1;
    QList<QRegExp *> regexpList;

    for (int i = 0; i < len; i += 2) {
        if (QString::compare(ParamsType::RegExp, params[i], Qt::CaseInsensitive) == 0) {
            QRegExp *regexp = new QRegExp(*params[i + 1], Qt::CaseInsensitive);
            qDebug() << "Regexp = " << *params[i+1];
            regexpList << regexp;
        }
    }

    return regexpList;
}


/**
 * @brief Try to find authentication title parameter.
 * Try to find parameter with type @see AuthTitle nad return parameter value.
 *
 * @param params parameters for searching.
 *
 * @return Authentication title or empty string ("") if parameter is not found.
 */
QString AuthParams::getParamsAuthTitle(const QList<QString*> &params)
{
    int len = params.length() - 1;

    for (int i = 0; i < len; i += 2) {
        if (QString::compare(ParamsType::AuthTitle, params[i], Qt::CaseInsensitive) == 0) {
            qDebug() << "Title = " << *params[i+1];
            return *params[i + 1];
        }
    }

    return "";
}


/**
 * @brief Try to find authentication client ID parameter.
 * Try to find parameter with type @see ClientID nad return parameter value.
 *
 * @param params parameters for searching.
 *
 * @return Authentication title or empty string ("") if parameter is not found.
 */
QString AuthParams::getParamsClientID(const QList<QString*> &params)
{
    int len = params.length() - 1;

    for (int i = 0; i < len; i += 2) {
        if (QString::compare(ParamsType::ClientID, params[i], Qt::CaseInsensitive) == 0) {
            qDebug() << "ClientID = " << *params[i+1];
            return *params[i + 1];
        }
    }

    return "";
}


/**
 * @brief Parse parameters by postfixes.
 * Find return, context(not used now) parameters (with postfixes @see ReturnPostfix
 * and @see ContextPostfix).
 *
 * @param outerParams parameters list.
 */
void AuthParams::parseParametersByPostfixes(const QStringList &outerParams)
{
    int len = outerParams.length() - 1;

    for (int i = 0; i < len; i +=2) {
        QString *parameter = new QString(outerParams[i]);
        QString *value = new QString(outerParams[i + 1]);

        params << parameter << value;

        if (params[i]->endsWith(ParamsType::ReturnPostfix) == true) {
            removePostfix(parameter, ParamsType::ReturnPostfix);
            returnParams << parameter << value;
        }

//        if (params[i]->endsWith(ParamsType::ContextPostfix) == true) {
//            removePostfix(parameter, ParamsType::ContextPostfix);
//            returnParams << parameter << value;
//        }
    }

}


/**
 * @brief Remove postfix from parameter name.
 *
 * @param authResult authentication resul.
 * @param withRetParams flag - insert to signal parameters return parameters.
 *
 * @return list with parameters prepared for sending by signal.
 */
void AuthParams::removePostfix(QString *parameter, QString typePostfix)
{
    int parameterLen = parameter->length();
    int postfixLen = typePostfix.length();

    parameter->replace(parameterLen - postfixLen, postfixLen, "");
}
