/*
 * authdbusadaptor.cpp - dbus auth manager class implementation.
 *
 * Copyright (C) 2010 - Aleksandr A. Lomov.
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include <QtCore/QString>
#include <QNetworkProxy>

#include "mainwindow.h"
#include "captchawindow.h"
#include "authwindow.h"
#include "authdbusadaptor.h"



/**
 * @name Constants of service.
 */
//@{
/** DBus connection name. */
const char* AuthDBusAdaptor::connectionName = "org.lomalan.AuthManager";
/** DBus service/interface name. */
const char* AuthDBusAdaptor::serviceName = "org.lomalan.AuthManager";
/** DBus object name or path. */
const char* AuthDBusAdaptor::objectName = "/AuthManager";
/** DBus signal with auth data. */
const char* AuthDBusAdaptor::messageSignalName = "message";
/** DBus signal with captcha data. */
const char* AuthDBusAdaptor::captchamsgSignalName = "captchamsg";
const char* AuthDBusAdaptor::loginmsgSignalName = "loginmsg";
/** DBus signal to get command for shotdown service. */
const char* AuthDBusAdaptor::shutdownSignalName = "shutdown";
//@}



/**
 * @brief Implementation of adaptor class AuthDBusAdaptor.
 *
 * @param perent parent object.
 */
AuthDBusAdaptor::AuthDBusAdaptor(QObject *parent) : QDBusAbstractAdaptor(parent), connection(connectionName)
{
    setAutoRelaySignals(true);
}


/**
 * @brief Destructor.
 */
AuthDBusAdaptor::~AuthDBusAdaptor()
{
    qDebug() << "Adaptor destructor START";
    connection.unregisterObject(objectName);
    connection.unregisterService(serviceName);
    connection.disconnectFromBus(connectionName);
    qDebug() << "Adaptor destructor END";
}



/********************************************/
/************ Public functions **************/
/**
 *  @brief Initialize dbus connection, service.
 *
 *  @return true if initialize ended with success, false otherwise.
 */
bool AuthDBusAdaptor::InitDBusService()
{
    connection = connection.connectToBus(QDBusConnection::SessionBus, connectionName);

    if (connection.isConnected() == true) {
        qDebug() << "Connetction established.";
    } else {
        return false;
    }

    if (connection.registerService(serviceName) == false) {
        qDebug() << "Can't register service. (" <<  serviceName <<")";

        connection.disconnectFromBus(connectionName);

        return false;
    }

    if (connection.registerObject(objectName, this, QDBusConnection::ExportAllContents) == false) {
        qDebug() << "Can't register object. (" << objectName << ")";

        connection.unregisterService(serviceName);
        connection.disconnectFromBus(connectionName);

        return false;
    }


    // Connect to get shutdown signal
    QDBusConnection::systemBus().connect(QString(), QString(),
            getConnectionName(), getShutdownSignalName(),
            (QObject*) this, SLOT (Shutdown()));

    QDBusConnection::sessionBus().connect(QString(), QString(),
            getConnectionName(), getShutdownSignalName(),
            (QObject*) this, SLOT (Shutdown()));


    return true;
}


/**
 * @brief Send dbus signal.
 * Using given signal name or standart @see messageSignalName if given is empty.
 *
 * @param signalName name of signal.
 * @param signalParams arguments(parameters) for signal.
 * @param auth true - if use auth, false - if use captcha
 *
 * @return true on success, or false otherwise.
 */
bool AuthDBusAdaptor::sendSignal(const QString &signalName, const QList<QVariant> &signalParams, bool auth)
{
    QString name;
    if (signalName.isEmpty() == true) {
        if(auth){
            name = messageSignalName;
        } else {
            name = captchamsgSignalName;
        }
    } else {
            name = signalName;
    }

    QDBusMessage msg = QDBusMessage::createSignal(objectName, serviceName, name);
    msg.setArguments(signalParams);

    qDebug() << "Prepare for sending signal with name " << name;

    return QDBusConnection::sessionBus().send(msg);
}


bool AuthDBusAdaptor::sendLoginSignal(const QString &signalName, const QList<QVariant> &signalParams)
{
    QString name;
    if (signalName.isEmpty() == true) {
            name = loginmsgSignalName;
    } else {
            name = signalName;
    }

    QDBusMessage msg = QDBusMessage::createSignal(objectName, serviceName, name);
    msg.setArguments(signalParams);

    qDebug() << "Prepare for sending signal with name " << name;

    return QDBusConnection::sessionBus().send(msg);
}

/**
 * @brief Getter for connection name.
 *
 * @return name of DBus connection.
 */
const char* AuthDBusAdaptor::getConnectionName()
{
    return connectionName;
}


/**
 * @brief Getter for shutdown signal name.
 *
 * @return name of signal.
 */
const char* AuthDBusAdaptor::getShutdownSignalName()
{
    return shutdownSignalName;
}



/********************************************/
/************** Public slots ****************/
/**
 * @brief unction to hadler dbus call.
 * Function for DBus call, emit setProxy signal.
 *
 * @param proxyIP IP address of proxy.
 * @param proxyPort port of proxy.
 * @param username name of user (if needed).
 * @param username password for proxy (if needed).
 */
void AuthDBusAdaptor::SetProxy(const QString &proxyIP, const QString &proxyPort,
        const QString &username, const QString &password)
{
  emit setProxy(proxyIP, proxyPort, username, password);
}


/**
 * @brief Function to hadler dbus call.
 * Function for DBus call, emit auth signal, simple authorization.
 *
 * @param url ar for authorization.
 *
 * @return now always return "ok" string.
 */
QString AuthDBusAdaptor::Auth(const QString &url) {

    emit auth(url);

    return "ok";
}


/**
 * @brief Function to hadler dbus call.
 * Function for DBus call, emit auth signal, extended auth.
 * If parameters not set using simple authorization.
 *
 * @param url ar for authorization.
 * @param params parameters for extended authorization.
 *
 * @return now always return "ok" string.
 */
QString AuthDBusAdaptor::Auth(const QString &url, const QStringList &params)
{
    if (params.isEmpty() == false) {
        emit auth(url, params);
    } else {
        qDebug("Parameters not set.");
        emit auth(url);
    }

    return "ok";
}

/**
 * @brief Function to hadler dbus call.
 * Function for DBus call, emit auth signal, simple test.
 *
 * @param strBase64 ar for test.
 *
 * @return now always return "ok" string.
 */
QString AuthDBusAdaptor::Captcha(const QString &strBase64, const QString clientID){

    emit captcha(strBase64, clientID);

    return "ok";
}

QString AuthDBusAdaptor::Login(const QString &clientID){

    emit login(clientID);

    return "ok";
}


void AuthDBusAdaptor::Shutdown()
{
    emit shutdown();
}
