/*
 * authmanager.h - authentication manager implementation.
 *
 * Copyright (C) 2010 - Aleksandr A. Lomov.
 *
 * Webauth is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Webauth is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Webauth; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */


#include "authmanager.h"
#include "authdbusadaptor.h"


/**
 * @brief Constructor.
 */
AuthManager::AuthManager(QObject *parent) :
    QObject(parent)
{
    lastWindowsID = 0;
    adapter = NULL;
}


/**
 * @brief Destructor
 */
AuthManager::~AuthManager()
{
    qDebug() << "AuthManager destructor START.";

    delete adapter;
    adapter = NULL;

    QList<MainWindow *> values = windows.values();

    int len = values.length();

    for (int i = 0; i < len; ++i) {
        delete values[i];
        values[i] = NULL;
    }

    windows.clear();

    qDebug() << "AuthManager destructor END.";
}



/********************************************/
/************ Public functions **************/
/**
 * @brief Initialize DBus connection.
 *
 * @param parent parent object.
 *
 * return 0 on success, -1 otherwise.
 */
int AuthManager::initDBus(QObject *parent)
{
    adapter = new AuthDBusAdaptor(parent);

    if (adapter == NULL || adapter->InitDBusService() == false) {
        delete adapter;
        adapter = NULL;
        return -1;
    }


    connect(adapter, SIGNAL (setProxy(const QString &, const QString &,
                                      const QString &, const QString &)),
            SLOT (setProxy(const QString &, const QString &,
                           const QString &, const QString &)));

    connect(adapter, SIGNAL (auth(const QString &)), SLOT (auth(const QString &)));
    connect(adapter, SIGNAL (auth(const QString &, const QStringList &)),
            SLOT (auth(const QString &, const QStringList &)));

    connect(adapter, SIGNAL (shutdown()), SLOT (shutdown()));

    return 0;
}



/********************************************/
/************** Public slots ****************/
/**
 * @brief Set proxy.
 *
 * Function for DBus call, using to setup proxy.
 *
 * @param proxyIP IP address of proxy.
 * @param proxyPort port of proxy.
 * @param username name of user (if needed).
 * @param username password for proxy (if needed).
 */
void AuthManager::setProxy(const QString &proxyIP, const QString &proxyPort,
        const QString &username, const QString &password)
{
    qDebug() << "\nSetting proxy...(host = " << proxyIP.toAscii().constData()
        << " port = " << proxyPort.toAscii().constData()
        << " username = " << username.toAscii().constData()
        << " password = " << password.toAscii().constData() << ").";

    uint port = proxyPort.toUInt();

    if (proxyIP.isNull() == true || proxyIP.isEmpty() == true
           || port <= 0) {
        qDebug() << "Proxy not set(incorrect host or port).";
        return;
    }

    if (username.isNull() != true
            && username.isEmpty() != true) {
        proxy.setUser(username);
        proxy.setPassword(password);
    }

    proxy.setType(QNetworkProxy::HttpProxy);
    proxy.setHostName(proxyIP);
    proxy.setPort(port);

    QNetworkProxy::setApplicationProxy(proxy);
}


/**
 * @brief Shutdown webauth service.
 */
void AuthManager::shutdown()
{
    qDebug() << "\nShudown webauth...";
    QApplication::quit();
}



/********************************************/
/************* Private slots ****************/
/**
 * @brief Setup for simple authentication.
 *
 * @param url url for user athentication.
 */
void AuthManager::auth(const QString &url)
{

    qDebug() << "\nUsing simple auth...";
    ++lastWindowsID;
    MainWindow *mw = new MainWindow(lastWindowsID);

    windows.insert(lastWindowsID, mw);
    connect(mw, SIGNAL (closed(const int)), SLOT (windowsClosed(const int)));
    mw->startAuth(url);
}


/**
 * @brief Setup for extended authentication.
 *
 * @param url url for user athentication.
 * @param params list witn regexps.
 */
void AuthManager::auth(const QString &url, const  QStringList &params)
{
    qDebug() << "\nUsing extended auth...";

    ++lastWindowsID;
    MainWindow *mw = new MainWindow(lastWindowsID);

    windows.insert(lastWindowsID, mw);
    connect(mw, SIGNAL (closed(const int)), SLOT (windowsClosed(const int)));

    mw->startAuth(url, params);

}


/**
 * @brief Slot for get wondow closed signal.
 * Get window identifier and remove it(window).
 * If no opened windows when shutdown webauth.
 *
 * @param windowID window identifier.
 */
void AuthManager::windowsClosed(const int windowID)
{
    qDebug() << "Window closed (ID = " << windowID << ").";

    MainWindow *mv = windows[windowID];
    windows.remove(windowID);

    delete(mv);
    mv = NULL;

    if (windows.isEmpty() == true) {
        shutdown();
    }

}
