/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "orientationhelper.hpp"

// Global includes
#include <QtCore/QPointer>
#include <QtGui/QHBoxLayout>
#include <QtGui/QApplication>
#include <QtGui/QDesktopWidget>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class OrientationHelperPrivate : public QObject
{
  Q_OBJECT

public:
  ~OrientationHelperPrivate()
  {
    if ( landscapeLayout && !landscapeLayout->parent() )
      delete landscapeLayout;

    if ( portraitLayout && !portraitLayout->parent() )
      delete portraitLayout;
  }

  OrientationHelper* self;

  // Properties
  QPointer<QWidget> widget;
  QPointer<QLayout> landscapeLayout;
  QPointer<QLayout> portraitLayout;

  QHBoxLayout defaultLayout;

  // Functions
  void init();

public slots:
  void onOrientationChanged();
};

//////////////////////////////////////////////////////////////////////////

void OrientationHelperPrivate::init()
{
#ifdef Q_WS_MAEMO_5
  // Connect the roration signal
  connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(onOrientationChanged()));
#endif // Q_WS_MAEMO_5
}

//////////////////////////////////////////////////////////////////////////

void OrientationHelperPrivate::onOrientationChanged()
{
  // Get the screen geometry
  QRect screenGeometry = QApplication::desktop()->screenGeometry();

  // Determine the orientation
  Qt::Orientation orientation = screenGeometry.width() > screenGeometry.height() ? Qt::Horizontal : Qt::Vertical;

  // Get the layout settings
  QLayout* oldLayout = orientation == Qt::Horizontal ? portraitLayout : landscapeLayout;
  QLayout* newLayout = orientation == Qt::Horizontal ? landscapeLayout : portraitLayout;

  if ( !oldLayout || !newLayout )
    return;

  // Remove the old layout
  defaultLayout.removeItem(oldLayout);
  oldLayout->setParent(NULL);

  // Set the new layout
  defaultLayout.addLayout(newLayout);

  // Emit that the orientation changed
  emit self->orientationChanged(orientation);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

OrientationHelper::OrientationHelper(QObject* parent /* = 0 */)
: QObject(parent)
, d(new OrientationHelperPrivate)
{
  d->self = this;
  d->init();
}

//////////////////////////////////////////////////////////////////////////

OrientationHelper::~OrientationHelper()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void OrientationHelper::setWidget(QWidget* widget)
{
  if ( d->widget == widget )
    return;

  d->widget = widget;

  if ( d->widget)
  {
    // Make sure the widget doesn't have a layout
    Q_ASSERT(d->widget->layout() == NULL);

    // Give the widget our default layout
    d->widget->setLayout(&d->defaultLayout);
  }
}

//////////////////////////////////////////////////////////////////////////

QWidget* OrientationHelper::widget() const
{
  return d->widget;
}

//////////////////////////////////////////////////////////////////////////

void OrientationHelper::setLandscapeLayout(QLayout* layout)
{
  if ( d->landscapeLayout == layout )
    return;

  if ( layout )
    layout->setParent(NULL);

  d->landscapeLayout = layout;
  d->onOrientationChanged();
}

//////////////////////////////////////////////////////////////////////////

QLayout* OrientationHelper::landscapeLayout() const
{
  return d->landscapeLayout;
}

//////////////////////////////////////////////////////////////////////////

void OrientationHelper::setPortraitLayout(QLayout* layout)
{
  if ( d->portraitLayout == layout )
    return;

  if ( layout )
    layout->setParent(NULL);

  d->portraitLayout = layout;
  d->onOrientationChanged();
}

//////////////////////////////////////////////////////////////////////////

QLayout* OrientationHelper::portraitLayout() const
{
  return d->portraitLayout;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "orientationhelper.moc"
