/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "networkhelper.hpp"

// Global includes
#include <QtCore/QHash>
#include <QtNetwork/QNetworkAccessManager>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class NetworkHelperPrivate : public QObject
{
  Q_OBJECT

public:
  NetworkHelperPrivate()
  {
    connect(&accessManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(onReplyFinished(QNetworkReply*)));
  }

  NetworkHelper* self;

  QObjectList listeners;
  QNetworkAccessManager accessManager;
  QHash<QNetworkReply*, QString> replyStateHash;

public slots:
  void onReplyFinished(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void NetworkHelperPrivate::onReplyFinished(QNetworkReply *reply)
{
  // Handle a redirection if needed
  QVariant redirectLocation = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
  if ( !redirectLocation.isNull() )
  {
    // Where are we going to?
    QUrl newUrl = reply->url().resolved(redirectLocation.toUrl());

    // Prepare the request and send it
    QNetworkRequest request = self->prepareRequest();
    request.setRawHeader("Referer", reply->url().toEncoded());
    request.setUrl( newUrl );
    self->get(request, replyStateHash.value(reply));

    return; // Don't continue
  }

  // If some error happend while transmitting HTTP content, also signalize it
//  if ( reply->error() )
//    emit self->error( tr("HTTP error happened:\n%1").arg(reply->errorString()) );

  if ( !replyStateHash.contains(reply) )
    return;

  QString stateName = replyStateHash.value(reply);
  QString functionName = QString("handle%1").arg(stateName);

  foreach(QObject* listener, listeners)
    QMetaObject::invokeMethod(listener, functionName.toUtf8().constData(), Q_ARG(QNetworkReply*, reply));

  // Remove hash values
  replyStateHash.remove(reply);

  // Delete the reply at a later time
  reply->deleteLater();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

NetworkHelper::NetworkHelper(QObject* parent /* = 0 */)
: QObject(parent)
, d(new NetworkHelperPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

NetworkHelper::~NetworkHelper()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

QNetworkRequest NetworkHelper::prepareRequest() const
{
  QNetworkRequest request;
  request.setRawHeader("User-Agent", "Opera/9.80 (Linux x86_64; U; en) Presto/2.5.18 Version/10.50 Gentoo");

  return request;
}

//////////////////////////////////////////////////////////////////////////

QNetworkReply* NetworkHelper::get(const QNetworkRequest& request, const QString& state)
{
  // Get the request
  QNetworkReply* reply = d->accessManager.get(request);

  // Remember the reply with the given state
  d->replyStateHash.insert(reply, state);

  return reply;
}

//////////////////////////////////////////////////////////////////////////

QNetworkReply* NetworkHelper::post(const QNetworkRequest& request, const QByteArray& data, const QString& state)
{
  // Get the post
  QNetworkReply* reply = d->accessManager.post(request, data);

  // Remember the reply with the given state
  d->replyStateHash.insert(reply, state);

  return reply;
}

//////////////////////////////////////////////////////////////////////////

void NetworkHelper::addListener(QObject* listener)
{
  d->listeners.append(listener);
}

//////////////////////////////////////////////////////////////////////////

void NetworkHelper::removeListener(QObject* listener)
{
  d->listeners.removeAll(listener);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "networkhelper.moc"
