/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "betamax.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QUrl>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QFormLayout>
#include <QtXml/QDomDocument>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class BetamaxPrivate : public QObject
{
  Q_OBJECT

public:
  BetamaxPrivate()
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Standard"), "standard", QIcon(), QString(), 160, 160, 250) );

    // Add us as network listener
    network.addListener(this);
  }

  Betamax* self;

  // Properties
  QString providerName;
  QUrl sendUrl;

  QString username;
  QString password;
  QString fromNumber;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void BetamaxPrivate::handleSendMessage(QNetworkReply* reply)
{
  QString errorMessage;

  QString xmlErrorMessage;
  int xmlErrorLine   = 0;
  int xmlErrorColumn = 0;

  if ( reply->error() == QNetworkReply::NoError )
  {
    // Read the reply data
    QDomDocument document;
    if ( document.setContent(reply, &xmlErrorMessage, &xmlErrorLine, &xmlErrorColumn) )
    {
      QDomElement docElement = document.documentElement();

      // Get the result element
      QDomElement resultElement = docElement.firstChildElement("result");
      if ( !resultElement.isNull() )
      {
        // The result element is either "0" (failure) or "1" (success)
        if ( resultElement.text().toInt() != 1 )
        {
          // Get the description message
          QDomElement descriptionElement = docElement.firstChildElement("description");

          // Setup a description
          QString errorDescription = descriptionElement.text();
          if ( errorDescription.isEmpty() )
            errorDescription = tr("<no description>");

          // Build a readable error message
          errorMessage = tr("Error: %1\n\nCauses could be: Wrong username or password.").arg(errorDescription);
        }
      }
      else
        errorMessage = tr("Returned XML does not contain a 'result' element");
    }
    else
    {
      errorMessage = tr("Error parsing XML output:\nMessage: %1\nLine: %2\nColumn: %3")
        .arg(xmlErrorMessage)
        .arg(xmlErrorLine)
        .arg(xmlErrorColumn);
    }
  }
  else
    errorMessage = reply->errorString();


  // Get the message
  Message message = qvariant_cast<Message>( reply->property("message") );

  // Emit the send reply signal
  emit self->sendMessageReply(errorMessage.isEmpty(), message);

  // Error output
  if ( !errorMessage.isEmpty() )
    emit self->error(errorMessage);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

Betamax::Betamax(const QString& name, const QUrl& sendUrl, QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new BetamaxPrivate)
{
  d->self         = this;
  d->providerName = name;
  d->sendUrl      = sendUrl;
}

//////////////////////////////////////////////////////////////////////////

Betamax::~Betamax()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void Betamax::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->username;
  stream >> d->password;
  stream >> d->fromNumber;

  if ( d->providerName.isEmpty() )
    stream >> d->sendUrl;
}

//////////////////////////////////////////////////////////////////////////

QByteArray Betamax::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->username;
  stream << d->password;
  stream << d->fromNumber;

  if ( d->providerName.isEmpty() )
    stream << d->sendUrl;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool Betamax::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle(d->providerName.isEmpty() ? tr("Other Betamax") : d->providerName);
  dialog.setUserName(d->username);
  dialog.setPassword(d->password);

  // Add additional fields
  QLabel numberLabel(tr("From number:"), &dialog);
  QLineEdit numberEdit(d->fromNumber, &dialog);

  QLabel sendUrlLabel(tr("Send URL:"), &dialog);
  QLineEdit sendUrlEdit(d->sendUrl.toString(), &dialog);
  sendUrlLabel.setInputMethodHints(Qt::ImhUrlCharactersOnly);

  QFormLayout* formLayout = dialog.formLayout();
  int row = formLayout->rowCount();
  formLayout->setWidget(row, QFormLayout::LabelRole, &numberLabel);
  formLayout->setWidget(row, QFormLayout::FieldRole, &numberEdit);

  formLayout->setWidget(++row, QFormLayout::LabelRole, &sendUrlLabel);
  formLayout->setWidget(  row, QFormLayout::FieldRole, &sendUrlEdit);

  sendUrlLabel.setVisible( d->providerName.isEmpty() );
  sendUrlEdit.setVisible( d->providerName.isEmpty() );

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->username   = dialog.userName();
    d->password   = dialog.password();
    d->fromNumber = numberEdit.text();

    if ( d->providerName.isEmpty() )
      d->sendUrl = sendUrlEdit.text();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> Betamax::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void Betamax::sendMessage(const Message& message)
{
  QStringList numbers;

  // List the contacts
  foreach(const ContactInfo& contact, message.receivers())
    numbers << contact.number();

  QString toField = numbers.join("\n");

  QByteArray text = toUrlPercentEncoding(message.text().toLocal8Bit());

  // Setup the message
  // sendsms.php?username=xxxxxxxxxx?&password=xxxxxxxxxx&from=xxxxxxxxxx&to=xxxxxxxxxx&text=xxxxxxxxxx
  QUrl url(d->sendUrl);
  url.addQueryItem("username", d->username);
  url.addQueryItem("password", d->password);
  url.addQueryItem("from", d->fromNumber);
  url.addQueryItem("to", toField);
  url.addEncodedQueryItem("text", text);

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "SendMessage");
  reply->ignoreSslErrors();
  reply->setProperty("message", QVariant::fromValue(message)); // Remember the message for later use
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class BetamaxProviderPlugin : public QObject
                            , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

public:
  BetamaxProviderPlugin()
  {
    addProvider(tr("VoipCheap.com"), "fbba682a-3139-472d-934d-228de216426a", QIcon(":/providers/betamax/voipcheap.com.png"), "https://www.voipcheap.com/myaccount/sendsms.php");
    addProvider(tr("SMSdiscount.com"), "91ac866a-a013-4f84-9def-204c366b8468", QIcon(":/providers/betamax/smsdiscount.com.png"), "https://www.smsdiscount.com/myaccount/sendsms.php");
    addProvider(tr("Lowratevoip.com"), "4a8dead9-6df8-4ff9-adf5-5604bf739755", QIcon(), "https://www.lowratevoip.com/myaccount/sendsms.php");
    addProvider(tr("Poivy.com"), "7f3086fa-2db7-4619-860f-12c364d79790", QIcon(), "https://www.poivy.com/myaccount/sendsms.php");    
    addProvider(tr("12voip.com"), "8297063d-38d7-454a-9281-5a538095981a", QIcon(), "https://www.12voip.com/myaccount/sendsms.php");
    addProvider(tr("ActionVoip.com"), "92924d13-791a-439e-beaf-5752dacdbf63", QIcon(), "https://www.actionvoip.com/myaccount/sendsms.php");
    addProvider(tr("BudgetSip.com"), "90d36590-b99e-421d-8f60-d45c5180dbfe", QIcon(), "https://www.budgetsip.com/myaccount/sendsms.php");
    addProvider(tr("CallEasy.com"), "292c0fa5-939d-47b6-80ca-90b1af58de83", QIcon(), "https://www.calleasy.com/myaccount/sendsms.php");
    addProvider(tr("DialNow.com"), "2d509516-db9c-4499-b4a6-53b2df7a036a", QIcon(), "https://www.dialnow.com/myaccount/sendsms.php");
    addProvider(tr("FreeCall.com"), "3f08f04a-2180-418a-a959-202dde3c43d2", QIcon(), "https://www.freecall.com/myaccount/sendsms.php");
    addProvider(tr("InternetCalls.com"), "3205db32-f0aa-47c8-ab09-3e46af38f6b9", QIcon(), "https://www.internetcalls.com/myaccount/sendsms.php");
    addProvider(tr("InterVoip.com"), "a57774f3-1a4e-437b-b4c2-f5d50f80ee74", QIcon(), "https://www.intervoip.com/myaccount/sendsms.php");
    addProvider(tr("Jumblo.com"), "2df9685e-6d93-4587-a27c-822da40dde58", QIcon(), "https://www.jumblo.com/myaccount/sendsms.php");
    addProvider(tr("JustVoip.com"), "13eecbdf-5f64-4888-abf6-0bf55e273c2a", QIcon(), "https://www.justvoip.com/myaccount/sendsms.php");
    addProvider(tr("NetAppel.fr"), "725aa6de-c47f-458f-91ee-88aaa6ef3c47", QIcon(), "https://www.netappel.fr/myaccount/sendsms.php");
    addProvider(tr("nonoh.net"), "7f899380-5f36-476c-a128-b6b3681f920f", QIcon(), "https://www.nonoh.net/myaccount/sendsms.php");
    addProvider(tr("rynga.com"), "c983dedd-528b-4079-bc53-de80fb240ab4", QIcon(), "https://www.rynga.com/myaccount/sendsms.php");
    addProvider(tr("sipdiscount.com"), "1c91c3e5-910d-44d9-93d9-98468da01d36", QIcon(), "https://www.sipdiscount.com/myaccount/sendsms.php");
    addProvider(tr("smartvoip.com"), "e0c40844-4887-4ff7-8486-6d349122f5e0", QIcon(), "https://www.smartvoip.com/myaccount/sendsms.php");
    addProvider(tr("smslisto.com"), "0b6a4e88-985c-4f88-925b-9a63fa12a234", QIcon(), "https://www.smslisto.com/myaccount/sendsms.php");
    addProvider(tr("sparvoip.de"), "68a7c3db-2dad-4df1-92b8-f1fcc079be84", QIcon(), "https://www.sparvoip.com/myaccount/sendsms.php");
    addProvider(tr("voipbuster.com"), "857c8bd4-de4f-4bbf-9f7c-617a090494e1", QIcon(), "https://www.voipbuster.com/myaccount/sendsms.php");
    addProvider(tr("voipbusterpro.com"), "eb17845f-ecab-4c1f-bb89-fdcf820b7941", QIcon(), "https://www.voipbusterpro.com/myaccount/sendsms.php");
    addProvider(tr("voipcheap.co.uk"), "f784fd68-edfb-4060-9268-8609460d23f1", QIcon(), "https://www.voipcheap.co.uk/myaccount/sendsms.php");
    addProvider(tr("voipian.com"), "441681d6-1463-4092-8bc5-99491a7da8f6", QIcon(), "https://www.voipian.com/myaccount/sendsms.php");
    addProvider(tr("voipraider.com"), "8c018e10-9edf-499c-ae67-73571ca1a1ed", QIcon(), "https://www.voipraider.com/myaccount/sendsms.php");
    addProvider(tr("voipstunt.com"), "9409f44a-8f64-455e-af45-ff1dd29d84de", QIcon(), "https://www.voipstunt.com/myaccount/sendsms.php");
    addProvider(tr("voipwise.com"), "e4ccb7cb-d32a-4db1-ba8f-91855baf8d26", QIcon(), "https://www.voipwise.com/myaccount/sendsms.php");
    addProvider(tr("voipzoom.com"), "1737784e-ad65-456a-aa03-af01d7c7365a", QIcon(), "https://www.voipzoom.com/myaccount/sendsms.php");
    addProvider(tr("webcalldirect.com"), "69aebbcf-54ac-4d63-b5eb-936d1783c1a5", QIcon(), "https://www.webcalldirect.com/myaccount/sendsms.php");
    
    addProvider(QString(), "a1834119-d9f6-44d5-80b8-5550946d5df7", QIcon(), QString());
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( nameHash.contains(providerId) )
      return new Betamax(nameHash.value(providerId), sendUrlHash.value(providerId));

    return NULL;
  }

protected:
  void addProvider(const QString& name, const QUuid& uuid, const QIcon& icon, const QString& sendUrl)
  {
    infoList << ProviderInfo(name.isEmpty() ? tr("Other Betamax") : name, uuid, icon, QLocale::AnyCountry);
    
    nameHash.insert(uuid, name);
    sendUrlHash.insert(uuid, sendUrl);
  }

private:
  QList<ProviderInfo> infoList;
  QHash<QUuid, QString> nameHash;
  QHash<QUuid, QUrl> sendUrlHash;
};

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(Betamax, BetamaxProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "betamax.moc"
