/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "sms77de.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"

// Global includes
#include <QtCore/QDataStream>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class sms77dePrivate : public QObject
{
  Q_OBJECT

public:
  sms77dePrivate()
  : isLoggedIn(false)
  {
    // Initialize the message types
    //messageTypes.append( MessageType(tr("Standard"), "standard", QIcon(), QString(), 160, 160, 1) );
    messageTypes.append( MessageType(tr("Quality"), "quality", QIcon(), QString("0,079 %1").arg(QChar(0x20AC)), 160, 1600, 1) );
    messageTypes.append( MessageType(tr("BasicPlus"), "basicplus", QIcon(), QString("0,035 %1").arg(QChar(0x20AC)), 160, 1600, 1) );
    messageTypes.append( MessageType(tr("Flash"), "flash", QIcon(), QString("0,039 %1").arg(QChar(0x20AC)), 160, 160, 1) );
    messageTypes.append( MessageType(tr("Festnetz"), "festnetz", QIcon(), QString("0,12 %1").arg(QChar(0x20AC)), 160, 160, 1) );

    // Add us as network listener
    network.addListener(this);

    // Setup the error codes
    errorCodes.insert("100", tr("SMS wurde erfolgreich verschickt"));
    errorCodes.insert("101", tr("Versand an mindestens einen Empfnger fehlgeschlagen"));
    errorCodes.insert("201", tr("Lndercode fr diesen SMS-Typ nicht gltig. Bitte als Basic SMS verschicken."));
    errorCodes.insert("202", tr("Empfngernummer ungltig"));
    errorCodes.insert("300", tr("Bitte Benutzer/Passwort angeben"));
    errorCodes.insert("301", tr("Variable to nicht gesetzt"));
    errorCodes.insert("304", tr("Variable type nicht gesetzt"));
    errorCodes.insert("305", tr("Variable text nicht gesetzt"));
    errorCodes.insert("306", tr("Absendernummer ungltig (nur bei Standard SMS). Diese muss vom Format 0049... sein und eine gltige Handynummer darstellen."));
    errorCodes.insert("307", tr("Variable url nicht gesetzt"));
    errorCodes.insert("400", tr("type ungltig. Siehe erlaubte Werte oben."));
    errorCodes.insert("401", tr("Variable text ist zu lang"));
    errorCodes.insert("402", tr("Reloadsperre  diese SMS wurde bereits innerhalb der letzten 90 Sekunden verschickt"));
    errorCodes.insert("500", tr("Zu wenig Guthaben vorhanden."));
    errorCodes.insert("600", tr("Carrier Zustellung misslungen"));
    errorCodes.insert("700", tr("Unbekannter Fehler"));
    errorCodes.insert("801", tr("Logodatei nicht angegeben"));
    errorCodes.insert("802", tr("Logodatei existiert nicht"));
    errorCodes.insert("803", tr("Klingelton nicht angegeben"));
    errorCodes.insert("900", tr("Benutzer/Passwort-Kombination falsch"));
    errorCodes.insert("902", tr("http API fr diesen Account deaktiviert"));
    errorCodes.insert("903", tr("Server IP ist falsch"));
  }

  sms77de* self;

  // Properties
  QString username;
  QString password;

  bool isLoggedIn;

  QHash<QByteArray, QString> errorCodes;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleBalance(QNetworkReply* reply);
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void sms77dePrivate::handleBalance(QNetworkReply* reply)
{
  // Read the reply data
  QByteArray data = reply->readAll();

  // Is this an error message?
  QString error = errorCodes.value(data);
  if ( !error.isEmpty() )
    emit self->error(error);
  else
  {
    // Make the balance readable
    QString balance = QString("%1 %2").arg(QString::fromUtf8(data)).arg(QChar(0x20AC));

    // Emit the balance changed signal
    emit self->balanceReply(balance);
  }
}

//////////////////////////////////////////////////////////////////////////

void sms77dePrivate::handleSendMessage(QNetworkReply* reply)
{
  // Read the reply data
  QByteArray data = reply->readAll();

  // Get the success state
  bool success = (data == "100");

  // Get the message
  Message message = qvariant_cast<Message>( reply->property("message") );

  // Emit the send reply signal
  emit self->sendMessageReply(success, message);

  // Error output
  if ( !success )
    emit self->error( errorCodes.value(data, tr("Unbekannter Fehlercode")) );
  else
    self->updateBalance();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

sms77de::sms77de(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new sms77dePrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

sms77de::~sms77de()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void sms77de::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->username;
  stream >> d->password;
}

//////////////////////////////////////////////////////////////////////////

QByteArray sms77de::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->username;
  stream << d->password;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool sms77de::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dialog(parent);
  dialog.setWindowTitle( tr("sms77.de") );
  dialog.setUserName(d->username);
  dialog.setPassword(d->password);

  if ( dialog.exec() == QDialog::Accepted )
  {
    d->username = dialog.userName();
    d->password = dialog.password();

    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> sms77de::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void sms77de::sendMessage(const Message& message)
{
  // Get the first contact as we only support sending to one
  ContactInfo contact = message.receivers().first();

  // Fix for numbers. This provider doesn't like the "+" sign, so we replace it by "00"
  QString number = contact.number();
  number.replace(QRegExp("^\\+"), "00");

  // Setup the message
  QUrl url("https://gateway.sms77.de");
  url.addQueryItem("u", d->username);
  url.addQueryItem("p", d->password);
  url.addQueryItem("to", number);
  url.addQueryItem("text", message.text());
  url.addQueryItem("type", message.messageType().name());
  //url.addQueryItem("debug", "1");

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "SendMessage");
  reply->ignoreSslErrors();
  reply->setProperty("message", QVariant::fromValue(message)); // Remember the message for later use
}

//////////////////////////////////////////////////////////////////////////

void sms77de::updateBalance()
{
  // Setup the url
  QUrl url("https://gateway.sms77.de/balance.php");
  url.addQueryItem("u", d->username);
  url.addQueryItem("p", d->password);

  // Setup the request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl(url);

  // Transmit the request
  QNetworkReply* reply = d->network.get(request, "Balance");
  reply->ignoreSslErrors();
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class sms77deProviderPlugin : public QObject
                            , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  static QUuid uuid;

public:
  sms77deProviderPlugin()
  {
    infoList << ProviderInfo(tr("sms77.de"), uuid, QIcon(":/providers/sms77de/sms77de.png"), QLocale::Germany);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( providerId == uuid )
      return new sms77de;

    return NULL;
  }

private:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

QUuid sms77deProviderPlugin::uuid = QUuid("9a93704f-88cd-45a7-b831-173784307f7c");

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(sms77de, sms77deProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "sms77de.moc"
