/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "biteSMS.hpp"
#include "bitesmsaccountsettingsdialog.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QDataStream>
#include <QtCore/QDateTime>
#include <QtXml/QDomDocument>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class biteSMSPrivate : public QObject
{
  Q_OBJECT

public:
  biteSMSPrivate()
  : credits(-1)
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Standard"), "standard", QIcon(), QString("One credit"), 160, -1, 250) );

    // Add us as network listener
    network.addListener(this);
  }

  biteSMS* self;

  // Properties
  QString token;
  int credits;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void biteSMSPrivate::handleSendMessage(QNetworkReply* reply)
{
  /*
  ### Bad result (invalid token)
  <?xml version="1.0" encoding="UTF-8"?>
  <result>
     <vouchers>
        <voucher>
           <token>asdf</token>
           <credits-left>0</credits-left>
        </voucher>
     </vouchers>
     <user-message>
        <type>ERROR</type>
        <message>Es wurden keine gültigen Gutscheine gefunden.</message>
     </user-message>
  </result>

  ### Good result
  <?xml version="1.0" encoding="UTF-8"?>
  <result>
     <vouchers>
        <voucher>
           <token>xxxxx</token>
           <credits-left>549</credits-left>
           <expires-at>1307450613</expires-at>
        </voucher>
     </vouchers>
  </result>
  */

  // TODO: Remove after debugging
  QByteArray data = reply->readAll();

  QString errorMessage;

  QString xmlErrorMessage;
  int errorLine = 0;
  int errorColumn = 0;

  // Read the XML data
  QDomDocument document;
  if ( document.setContent(data, &xmlErrorMessage, &errorLine, &errorColumn) )
  {
    // Get the document element
    QDomElement documentElement = document.documentElement();

    // Get the vouchers to get the credit info
    QDomElement vouchersElement = documentElement.firstChildElement("vouchers");
    QDomElement voucherElement = vouchersElement.firstChildElement("voucher");
    while ( !voucherElement.isNull() )
    {
      // Get the token and the remaining credits
      QDomElement tokenElement = voucherElement.firstChildElement("token");
      QDomElement creditsElement = voucherElement.firstChildElement("credits-left");

      // Read the balance data
      if ( tokenElement.text() == this->token )
      {
        this->credits = creditsElement.text().toInt();
        self->updateBalance();
      }

      // There may be more voucher elements (seems like we can have multiple tokens)
      voucherElement = voucherElement.nextSiblingElement();
    }

    // Read the user message
    QDomElement userMessageElement = documentElement.firstChildElement("user-message");
    if ( !userMessageElement.isNull() )
    {
      //QDomElement typeElement = userMessageElement.firstChildElement("type");
      QDomElement messageElement = userMessageElement.firstChildElement("message");
      
      // Get the error message if there's one
      //if ( typeElement.text() == "ERROR" )
        errorMessage = messageElement.text();
    }
  }
  else
  {
    QString msg = tr("Error Reading XML reply.\nError: %1\nLine: %2\nColumn: %3").arg(xmlErrorMessage).arg(errorLine).arg(errorColumn);
    emit self->error(msg);
  }

  // Show the error message
  if ( !errorMessage.isEmpty() )
    emit self->error(errorMessage);

  // Get the message and send the send reply
  Message message = qvariant_cast<Message>( reply->property("message") );
  emit self->sendMessageReply(errorMessage.isEmpty() && xmlErrorMessage.isEmpty(), message);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

biteSMS::biteSMS(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new biteSMSPrivate)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

biteSMS::~biteSMS()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void biteSMS::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->token;
  stream >> d->credits;
}

//////////////////////////////////////////////////////////////////////////

QByteArray biteSMS::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->token;
  stream << d->credits;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool biteSMS::showAccountSettingsDialog(QWidget* parent)
{
  biteSMSAccountSettingsDialog dlg(parent);
  dlg.setToken(d->token);

  if ( QDialog::Accepted == dlg.exec() )
  {
    d->token = dlg.token();
    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> biteSMS::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void biteSMS::sendMessage(const Message& message)
{
  /*
  <?xml version='1.0' encoding='UTF-8'?>
  <sms>
    <timestamp>2008-05-28 16:00:21 +0200</timestamp>
    <recipients>
      <to>+61418469200</to>
    </recipients>
    <vouchers>
      <token>xxx</token>
    </vouchers>
    <message>Test</message>
  </sms>
  */

  // Setup the XML data
  QDomDocument document;
  QDomElement smsElement = document.createElement("sms");
  document.appendChild(smsElement);

  QDomElement timestampElement = document.createElement("timestamp");
  timestampElement.appendChild( document.createTextNode(QDateTime::currentDateTime().toString(Qt::ISODate)) );
  smsElement.appendChild(timestampElement);

  QDomElement recipientsElement = document.createElement("recipients");
  foreach(const ContactInfo& contact, message.receivers())
  {
    QDomElement toElement = document.createElement("to");
    toElement.appendChild( document.createTextNode(contact.number()) );

    recipientsElement.appendChild(toElement);
  }
  smsElement.appendChild(recipientsElement);

  QDomElement vouchersElement = document.createElement("vouchers");
  QDomElement tokenElement = document.createElement("token");
  tokenElement.appendChild( document.createTextNode(d->token) );
  vouchersElement.appendChild(tokenElement);
  smsElement.appendChild(vouchersElement);

  QDomElement messageElement = document.createElement("message");
  messageElement.appendChild( document.createTextNode(message.text()) );
  smsElement.appendChild(messageElement);

  // Create a request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl( QUrl("http://smsgateway.bitesms.com/deliver") );
  
  // Post the request and bind the message data
  QNetworkReply* reply = d->network.post(request, document.toByteArray(), "SendMessage");
  reply->setProperty("message", QVariant::fromValue(message));
}

//////////////////////////////////////////////////////////////////////////

void biteSMS::updateBalance()
{
  QString text = d->credits == -1 ? tr("unknown Credits") : tr("%n Credit(s)", "", d->credits);
  emit balanceReply(text);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class biteSMSProviderPlugin : public QObject
                            , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  static QUuid uuid;

public:
  biteSMSProviderPlugin()
  {
    infoList << ProviderInfo(tr("biteSMS"), uuid, QIcon(":/providers/biteSMS/biteSMS.png"), QLocale::AnyCountry);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( providerId == uuid )
      return new biteSMS;

    return NULL;
  }

private:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

QUuid biteSMSProviderPlugin::uuid = QUuid("8ffe69b6-055c-47e5-a250-bdb92ed8e605");

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(biteSMS, biteSMSProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "biteSMS.moc"
