/*
 *
 *  Copyright (c) 2010 Christoph Keller <gri@nospam@not-censored.com>
 *
 *  This file is part of Web2SMS.
 *
 *  Web2SMS is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Web2SMS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Web2SMS. If not, see <http://www.gnu.org/licenses/>
 *
 */

// Local includes
#include "160by2.hpp"
#include "../providerplugin.hpp"
#include "../networkhelper.hpp"
#include "../accountsettingsdialog.hpp"
#include "../util.hpp"

// Global includes
#include <QtCore/QDataStream>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class P160by2Private : public QObject
{
  Q_OBJECT

public:
  P160by2Private()
  {
    // Initialize the message types
    messageTypes.append( MessageType(tr("Standard"), "standard", QIcon(), QString(), 140, 140, 250) );

    // Add us as network listener
    network.addListener(this);
  }

  P160by2* self;

  // Properties
  QString user;
  QString password;

  NetworkHelper network;

  QList<MessageType> messageTypes;

  // Functions
  Q_INVOKABLE void handleSendMessage(QNetworkReply* reply);
};

//////////////////////////////////////////////////////////////////////////

void P160by2Private::handleSendMessage(QNetworkReply* reply)
{
  QString errorMessage = reply->readAll();

  // Show the error message
  if ( !errorMessage.isEmpty() )
    emit self->error(errorMessage);

  // Get the message and send the send reply
  Message message = qvariant_cast<Message>( reply->property("message") );
  emit self->sendMessageReply(errorMessage.isEmpty(), message);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

P160by2::P160by2(QObject* parent /* = 0 */)
: ProviderInterface(parent)
, d(new P160by2Private)
{
  d->self = this;
}

//////////////////////////////////////////////////////////////////////////

P160by2::~P160by2()
{
  delete d;
}

//////////////////////////////////////////////////////////////////////////

void P160by2::loadSettings(const QByteArray& data)
{
  QDataStream stream(data);
  stream >> d->user;
  stream >> d->password;
}

//////////////////////////////////////////////////////////////////////////

QByteArray P160by2::saveSettings() const
{
  QByteArray data;
  QDataStream stream(&data, QIODevice::WriteOnly);
  stream << d->user;
  stream << d->password;

  return data;
}

//////////////////////////////////////////////////////////////////////////

bool P160by2::showAccountSettingsDialog(QWidget* parent)
{
  AccountSettingsDialog dlg(parent);
  dlg.setWindowTitle( "160by2" );
  dlg.setUserName(d->user);
  dlg.setPassword(d->password);

  if ( QDialog::Accepted == dlg.exec() )
  {
    d->user = dlg.userName();
    d->password = dlg.password();
    return true;
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////

QList<MessageType> P160by2::messageTypes() const
{
  return d->messageTypes;
}

//////////////////////////////////////////////////////////////////////////

void P160by2::sendMessage(const Message& message)
{
  QStringList numbers;
  foreach(const ContactInfo& contact, message.receivers())
  {
    QString number = contact.number();
    number.remove( QRegExp("^\\+") );

    numbers.append(number);
  }

  // Create a request
  QNetworkRequest request = d->network.prepareRequest();
  request.setUrl( QUrl("http://www.160by2.com/sendsms.aspx") );
  
  QByteArray postData;
  postData += "apiuser=" + toUrlPercentEncoding( toLatin1Euro(d->user) );
  postData += "&password=" + toUrlPercentEncoding( toLatin1Euro(d->password) );
  postData += "&mobilenumbers=" + toUrlPercentEncoding( numbers.join(",").toLatin1() );
  postData += "&message=" + toUrlPercentEncoding( toLatin1Euro(message.text()) );
  postData += "&iphone=F44C0F20";
  
  // Post the request and bind the message data
  QNetworkReply* reply = d->network.post(request, postData, "SendMessage");
  reply->setProperty("message", QVariant::fromValue(message));
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class P160by2ProviderPlugin : public QObject
                            , public ProviderPlugin
{
  Q_OBJECT
  Q_INTERFACES(ProviderPlugin)

  static QUuid uuid;

public:
  P160by2ProviderPlugin()
  {
    infoList << ProviderInfo(tr("160by2"), uuid, QIcon(":/providers/160by2/160by2.png"), QLocale::India);
  }

  QList<ProviderInfo> info() const
  {
    return infoList;
  }

  ProviderInterface* createProvider(const QUuid& providerId) const
  {
    if ( providerId == uuid )
      return new P160by2;

    return NULL;
  }

private:
  QList<ProviderInfo> infoList;
};

//////////////////////////////////////////////////////////////////////////

QUuid P160by2ProviderPlugin::uuid = QUuid("51b17990-ee64-11df-98cf-0800200c9a66");

//////////////////////////////////////////////////////////////////////////

Q_EXPORT_PLUGIN2(160by2, P160by2ProviderPlugin);

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#include "160by2.moc"
